#include <sys/param.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/vnode.h>
#include <sys/vfs.h>
#include <sys/uio.h>
#include <sys/cred.h>
#include <sys/pathname.h>
#include <sys/dirent.h>
#include <sys/debug.h>
#include <sys/sysmacros.h>
#include <sys/tiuser.h>
#include <sys/cmn_err.h>
#include <sys/stat.h>
#include <sys/mode.h>
#include <rpc/types.h>
#include <rpc/auth.h>
#include <rpc/clnt.h>
#include <sys/fs_subr.h>
#include <sys/mman.h>
#include <sys/vm.h>
#include <vm/as.h>
#include <vm/pvn.h>
#include <vm/seg_vn.h>
#include <vm/seg_kp.h>
#include <vm/seg_map.h>
#include <vm/page.h>

#include <fist.h>
#include <rot1fs.h>

/*
 *  Vnode ops for fist_rot1fs
 */
static int fist_rot1_open(vnode_t **, int, cred_t *);
static int fist_rot1_close(vnode_t *, int, int, offset_t, cred_t *);
static int fist_rot1_getattr(vnode_t *, vattr_t *, int, cred_t *);
static int fist_rot1_access(vnode_t *, int, int, cred_t *);
static int fist_rot1_lookup(vnode_t *, char *, vnode_t **,
			     pathname_t *, int, vnode_t *, cred_t *);
static int fist_rot1_remove(vnode_t *, char *, cred_t *);
static int fist_rot1_rename(vnode_t *, char *, vnode_t *, char *, cred_t *);
static int fist_rot1_mkdir(vnode_t *, char *, vattr_t *, vnode_t **, cred_t *);
static int fist_rot1_rmdir(vnode_t *, char *, vnode_t *, cred_t *);
static int fist_rot1_readdir(vnode_t *, uio_t *, cred_t *, int *);
static int fist_rot1_symlink(vnode_t *, char *, vattr_t *, char *, cred_t *);
static int fist_rot1_fsync(vnode_t *, int, cred_t *);
static void fist_rot1_inactive(vnode_t *, cred_t *);
static void fist_rot1_rwlock(vnode_t *, int);
static void fist_rot1_rwunlock(vnode_t * vp, int);
static int fist_rot1_seek(vnode_t * vp, offset_t, offset_t *);
static int fist_rot1_cmp(vnode_t *, vnode_t *);
static int fist_rot1_read(vnode_t *, uio_t *, int, cred_t *);
static int fist_rot1_write(vnode_t *, uio_t *, int, cred_t *);
static int fist_rot1_ioctl(vnode_t *, int, int, int, cred_t *, int *);
static int fist_rot1_setfl(vnode_t *, int, int, cred_t *);
static int fist_rot1_setattr(vnode_t *, vattr_t *, int, cred_t *);
static int fist_rot1_create(vnode_t *, char *, vattr_t *, vcexcl_t, int,
			     vnode_t **, cred_t *, int);
static int fist_rot1_link(vnode_t *, vnode_t *, char *, cred_t *);
static int fist_rot1_readlink(vnode_t *, uio_t *, cred_t *);
static int fist_rot1_fid(vnode_t *, fid_t *);
static int fist_rot1_frlock(vnode_t *, int, struct flock64 *, int, offset_t, cred_t *);
static int fist_rot1_space(vnode_t *, int, struct flock64 *, int, offset_t, cred_t *);
static int fist_rot1_realvp(vnode_t *, vnode_t **);

extern int fist_rot1_getpage(vnode_t *, offset_t, u_int, u_int *, page_t **,
		       u_int, struct seg *, caddr_t, enum seg_rw, cred_t *);
extern int fist_rot1_getapage(vnode_t *, u_int, u_int, u_int *, page_t **,
		       u_int, struct seg *, caddr_t, enum seg_rw, cred_t *);
extern int fist_rot1_putpage(vnode_t *, offset_t, u_int, int, cred_t *);
extern int fist_rot1_putapage(struct vnode *vp, page_t * pp, u_offset_t * offp, u_int * lenp, int flags, struct cred *cr);
extern int fist_rot1_map(vnode_t *, offset_t, struct as *, caddr_t *, u_int,
			  u_char, u_char, u_int, cred_t *);
extern int fist_rot1_addmap(vnode_t *, offset_t, struct as *, caddr_t, u_int,
			     u_char, u_char, u_int, cred_t *);
extern int fist_rot1_delmap(vnode_t *, offset_t, struct as *, caddr_t, u_int,
			     u_int, u_int, u_int, cred_t *);

static int fist_rot1_poll(vnode_t *, short, int, short *, pollhead_t **);
static int fist_rot1_dump(vnode_t *, caddr_t, int, int);
static int fist_rot1_pathconf(vnode_t *, int, u_long *, cred_t *);
static int fist_rot1_pageio(vnode_t *, page_t *, u_offset_t, u_int, int, cred_t *);
static int fist_rot1_dumpctl(vnode_t *, int);
static void fist_rot1_dispose(vnode_t *, page_t *, int, int, cred_t *);
static int fist_rot1_setsecattr(vnode_t *, vsecattr_t *, int, cred_t *);
static int fist_rot1_getsecattr(vnode_t *, vsecattr_t *, int, cred_t *);

vnodeops_t fist_rot1_vnodeops =
{
  fist_rot1_open,		/* open */
  fist_rot1_close,		/* close */
  fist_rot1_read,		/* read */
  fist_rot1_write,		/* write */
  fist_rot1_ioctl,		/* ioctl */
  fist_rot1_setfl,		/* setfl */
  fist_rot1_getattr,		/* getattr */
  fist_rot1_setattr,		/* setattr */
  fist_rot1_access,		/* access */
  fist_rot1_lookup,		/* lookup */
  fist_rot1_create,		/* create */
  fist_rot1_remove,		/* remove */
  fist_rot1_link,		/* link */
  fist_rot1_rename,		/* rename */
  fist_rot1_mkdir,		/* mkdir */
  fist_rot1_rmdir,		/* rmdir */
  fist_rot1_readdir,		/* readdir */
  fist_rot1_symlink,		/* symlink */
  fist_rot1_readlink,		/* readlink */
  fist_rot1_fsync,		/* fsync */
  fist_rot1_inactive,		/* inactive */
  fist_rot1_fid,		/* fid */
  fist_rot1_rwlock,		/* rwlock */
  fist_rot1_rwunlock,		/* rwunlock */
  fist_rot1_seek,		/* seek */
  fist_rot1_cmp,		/* cmp */
  fist_rot1_frlock,		/* frlock */
  fist_rot1_space,		/* space */
  fist_rot1_realvp,		/* realvp */
  fist_rot1_getpage,		/* getpage */
  fist_rot1_putpage,		/* putpage */
  fist_rot1_map,		/* map */
  fist_rot1_addmap,		/* addmap */
  fist_rot1_delmap,		/* delmap */
  fist_rot1_poll,		/* poll */
  fist_rot1_dump,		/* dump */
  fist_rot1_pathconf,		/* pathconf */
  fist_rot1_pageio,		/* pageio */
  fist_rot1_dumpctl,		/* dumpctl */
  fist_rot1_dispose,		/* dispose */
  fist_rot1_setsecattr,	/* setsecattr */
  fist_rot1_getsecattr		/* getsecattr */
};

#define	NEED_MOUNT(vp)	(vntoan(vp)->an_mntflags & MF_MNTPNT)
#define	SPECIAL_PATH(pnp) ((pnp)->pn_path[pn_pathleft(pnp) - 1] == ' ')
#define	MOUNTED_ON(vp) (vntoan(vp)->an_mntflags & MF_MOUNTED)


/* interpose on an old vnode and return new one */
vnode_t *
fist_rot1_interpose(vnode_t * hidden_vp, vfs_t * this_vfsp)
{
  vnode_t *fw_vp;
  fist_rot1node_t *fwnp;
  struct fist_rot1info *fwip = NULL;

  mutex_enter(&vfstofwi(this_vfsp)->fwi_ht_lock);

  /* look if entry exists in HT */
  fw_vp = fist_ht_find_vp(hidden_vp, this_vfsp);
  if (fw_vp) {
    fist_dprint(6, "ROT1_INTERPOSE found vp in HT!\n");
    VN_HOLD(fw_vp);		/* maybe VN_RELE(hidden_vp) */
    VN_RELE(hidden_vp);
    goto out;
  }
  /* allocate new vnode */
  fw_vp = kmem_zalloc(sizeof(vnode_t), KM_SLEEP);
  /* XXX: need VN_INIT2 that will reuse lock var of interposed vnode */
  VN_INIT(fw_vp, this_vfsp, hidden_vp->v_type, (dev_t) NULL);

  /* allocate and fill in fist_rot1node_t */
  fwnp = (fist_rot1node_t *)
    kmem_zalloc(sizeof(fist_rot1node_t), KM_SLEEP);
  if (!fwnp) {
    kmem_free(fw_vp, sizeof(vnode_t));
    fw_vp = NULL;
    goto out;
  }

  /* init fwnp */
  fwnp->fwn_mapcnt = 0;		/* no mapped pages */
  mutex_init(&fwnp->fwn_lock, "fist_rot1fs private data lock",
	     MUTEX_DEFAULT, NULL);
  /* store new vnode */
  fwnp->fwn_vnodep = hidden_vp;
  fw_vp->v_data = (caddr_t) fwnp;

  /* operations are fist_rot1 vnode ops */
  fw_vp->v_op = &fist_rot1_vnodeops;

  /* set rest of fields to NULL */
  fw_vp->v_vfsmountedhere = NULL;
  fw_vp->v_filocks = NULL;

  /* don't do this one for now */
  /* fw_vp->v_cv = NULL; */

  /* increment interposed vnodes counter */
  fwip = vfstofwi(this_vfsp);
  fwip->fwi_num_vnodes++;

  /* insert into HT */
  fist_ht_insert_vp(hidden_vp, fw_vp);
  /* VN_RELE(hidden_vp); */

#if 0
  /* hold our vnode */
  VN_HOLD(fw_vp);
#endif

out:
  mutex_exit(&vfstofwi(this_vfsp)->fwi_ht_lock);

  print_location();

  /* finally  return vnode to this newly created one */
  return (fw_vp);
}


static int
fist_rot1_open(
		 vnode_t ** vpp,
		 int flag,
		 cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp, *new_vp;

  fist_dprint(4, "fist_rot1_open vpp %x, flag 0x%x\n", vpp, flag);
  if (flag & FAPPEND) {
    fist_dprint(6, "***WARNING*** file is opened in append-only mode!!!\n");
    flag &= ~FAPPEND;		/* turn off FAPPEND flag */
  }

  new_vp = hidden_vp = vntofwn(*vpp)->fwn_vnodep;
  /* we NEED read access */
  flag |= FREAD;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_OPEN(&hidden_vp, flag, cr);

  /* check if a new vnode was returned (cloned) */
  if (!error && new_vp != hidden_vp) {
    fist_dprint(6, "ROT1_OPEN1: hidden_vp->v_count %d\n",
		hidden_vp->v_count);
    /*
     * yes: need to allocate a new fist_rot1 vnode,
     * initialize it, store interposed vnode in it, and
     * finally return the fist_rot1 vnode back.
     */
    *vpp = fist_rot1_interpose(hidden_vp, (*vpp)->v_vfsp);
    fist_dprint(6, "ROT1_OPEN2: vpp->v_count %d, hidden_vp->v_count %d\n",
		(*vpp)->v_count, hidden_vp->v_count);
  }
  print_location();
  return (error);
}


static int
fist_rot1_close(
		  vnode_t * vp,
		  int flag,
		  int count,
		  offset_t offset,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_close vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  fist_dprint(6, "ROT1_CLOSE1: vp->v_count %d, hidden_vp->v_count %d\n",
	      vp->v_count, hidden_vp->v_count);

  /* pass operation to hidden filesystem, and return status */
  error = VOP_CLOSE(hidden_vp, flag, count, offset, cr);

  fist_dprint(6, "ROT1_CLOSE2: vp->v_count %d, hidden_vp->v_count %d\n",
	      vp->v_count, hidden_vp->v_count);
  print_location();
  return (error);
}

static int
fist_rot1_read(
		 vnode_t * vp,
		 uio_t * uiop,
		 int ioflag,
		 cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;
  uio_t temp_uio;
  iovec_t *temp_iovec;
  caddr_t current_base;
  int i, bytes_read;
  int num_pages, resid;
  long long start_loffset, end_loffset;
  long long cleartext_start_loffset, cleartext_end_loffset, current_loffset;

  fist_dprint(4, "fist_rot1_read vp %x\n", vp);

#ifdef FIST_DEBUG
  fist_print_uios("fist_rot1_read", uiop);
#endif /* FIST_DEBUG */

  cleartext_start_loffset = uiop->uio_loffset;
  cleartext_end_loffset = uiop->uio_loffset + uiop->uio_resid;
  start_loffset = cleartext_start_loffset & ~(PAGESIZE - 1);
  end_loffset = cleartext_end_loffset & ~(PAGESIZE - 1);
  /* if not multiple of PAGESIZE, then the above formula loses one page.
   * adjust for it */
  if (cleartext_end_loffset > end_loffset)
    end_loffset += PAGESIZE;
  resid = end_loffset - start_loffset;
  num_pages = resid >> PAGESHIFT;

  fist_dprint(6,
	      "READ: so=%d eo=%d cs=%d es=%d res=%d np=%d ps=%d\n",
	      (int) start_loffset,
	      (int) end_loffset,
	      (int) cleartext_start_loffset,
	      (int) cleartext_end_loffset,
	      resid,
	      num_pages,
	      PAGESIZE);

  temp_iovec = kmem_zalloc(num_pages * sizeof(iovec_t), KM_SLEEP);
  for (i = 0; i < num_pages; i++) {
    temp_iovec[i].iov_len = PAGESIZE;
    temp_iovec[i].iov_base = kmem_zalloc(PAGESIZE, KM_SLEEP);
    fist_dprint(6, "READ allocated %d address 0x%x\n",
		i, temp_iovec[i].iov_base);
  }

  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = num_pages;
  temp_uio.uio_loffset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_fmode = uiop->uio_fmode;
  temp_uio.uio_llimit = uiop->uio_llimit;
  temp_uio.uio_resid = resid;

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /*
   * pass operation to hidden filesystem, and return status
   */

  error = VOP_READ(hidden_vp, &temp_uio, ioflag, cr);

  if (error) {
    fist_dprint(4, "VOP_READ in read returned error - not good\n");
    /* XXX to be checked */
    goto out_free;
  }

  current_loffset = start_loffset;
  for (i = 0; i < num_pages; i++) {
    bytes_read = PAGESIZE - temp_iovec[i].iov_len;
    if (bytes_read == 0)
      break;

    temp_iovec[i].iov_base -= bytes_read;
    current_base = temp_iovec[i].iov_base;

    /* derot1 the page/block */
    rot1_decode_block(__FUNCTION__,__LINE__,current_base, bytes_read, vp);

    /*
     * save the original size, for kmem_free.
     * no need for it w/ rot1fs; size is always PAGESIZE, hence this line
     * is commented out:
     *		temp_iovec[i].iov_len = uiop->uio_iov[i].iov_len;
     */
    /* treat first and last iovec separately, not all data in them is needed */
    if (current_loffset + bytes_read > cleartext_end_loffset) {
      bytes_read = cleartext_end_loffset - current_loffset;
    }
    if (i == 0) {
      bytes_read -= cleartext_start_loffset - start_loffset;
      current_loffset += cleartext_start_loffset - start_loffset;
      current_base += cleartext_start_loffset - start_loffset;
    }
    if ((error = uiomove(current_base, bytes_read, UIO_READ, uiop)))
      /*
       * XXX: we have to see the exact semantics of returning with an
       * EFAULT from read
       */
      break;
    current_loffset += bytes_read;
  }

out_free:
  for (i = 0; i < num_pages; i++) {
    fist_dprint(6, "READ freeing %d address 0x%x\n",
		i, temp_iovec[i].iov_base);
    kmem_free(temp_iovec[i].iov_base, PAGESIZE);
  }
  kmem_free(temp_iovec, num_pages * sizeof(iovec_t));

#ifdef FIST_DEBUG
  fist_print_uios("fist_rot1_read (END)", uiop);
#endif /* FIST_DEBUG */

  print_location();
  return (error);
}

static int
fist_rot1_write(
		  vnode_t * vp,
		  uio_t * uiop,
		  int ioflag,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;
  vattr_t va;
  uio_t temp_uio;
  iovec_t *temp_iovec;
  iovec_t *free_iovec;		/* for freeing allocated memory */
  int i;
  caddr_t current_base;
  int resid, bytes_read, num_pages, first_page_bytes, real_first_page;
  long long start_loffset, end_loffset, real_start_loffset;
  long long cleartext_start_loffset, cleartext_end_loffset, current_loffset;
  int hidden_ioflag = (ioflag & ~FAPPEND);

  fist_dprint(4, "fist_rot1_write vp %x, ioflag 0x5x\n", vp, ioflag);

#ifdef FIST_DEBUG
  fist_print_uios("fist_rot1_write (START)", uiop);
#endif /* FIST_DEBUG */

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* we don't want anybody to do updates while we write, so lock the vnode */
  mutex_enter(&vp->v_lock);

  /* get the attributes, length is necessary for correct updates */
  va.va_mask = AT_SIZE;
  if ((error = VOP_GETATTR(hidden_vp, &va, 0, cr))) {
    fist_dprint(4, "VOP_GETATTR returned error - not good\n");
    /* XXX to be checked */
    goto out;
  }

  /* just in case someone tries to pull a fast one */
  if (uiop->uio_resid == 0) {
    error = 0;
    goto out;
  }

  cleartext_start_loffset = uiop->uio_loffset;
  cleartext_end_loffset = uiop->uio_loffset + uiop->uio_resid;

  if (ioflag & FAPPEND) {
    fist_dprint(6, "WRITE: turning off append flag\n");
    cleartext_start_loffset += va.va_size;
    cleartext_end_loffset += va.va_size;
  }

  start_loffset = MIN(cleartext_start_loffset, va.va_size) & ~(PAGESIZE - 1);
  real_start_loffset = cleartext_start_loffset & ~(PAGESIZE - 1);
  first_page_bytes = MIN(cleartext_start_loffset, va.va_size) - start_loffset;
  /* must use this to avoid shifting a quad w/ gcc */
  real_first_page = (int)(real_start_loffset - start_loffset) >> PAGESHIFT;
  end_loffset = cleartext_end_loffset & ~(PAGESIZE - 1);
  ASSERT(first_page_bytes <= PAGESIZE);
  /*
   * if not multiple of PAGESIZE, then the above formula loses one page.
   * adjust for it
   */
  if (cleartext_end_loffset > end_loffset)
    end_loffset += PAGESIZE;
  resid = end_loffset - start_loffset;
  num_pages = resid >> PAGESHIFT;

  if (num_pages == 1)
    first_page_bytes = PAGESIZE;

  temp_iovec = kmem_zalloc(num_pages * sizeof(iovec_t), KM_SLEEP);
  free_iovec = kmem_zalloc(num_pages * sizeof(iovec_t), KM_SLEEP);
  for (i = 0; i < num_pages; i++) {
    temp_iovec[i].iov_len = free_iovec[i].iov_len = PAGESIZE;
    /* we need the pages to be zeroed out */
    temp_iovec[i].iov_base = free_iovec[i].iov_base = kmem_zalloc(PAGESIZE, KM_SLEEP);
  }

  fist_dprint(6,
	      "WRITE: so=%d eo=%d cso=%d ceo=%d rso=%d res=%d np=%d rfp=%d\n",
	      (int) start_loffset,
	      (int) end_loffset,
	      (int) cleartext_start_loffset,
	      (int) cleartext_end_loffset,
	      (int) real_start_loffset,
	      resid,
	      num_pages,
	      real_first_page
	      );

  current_loffset = start_loffset;

  /* read first block XXX check length of file */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_loffset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_fmode = FREAD;
  temp_uio.uio_llimit = uiop->uio_llimit;
  temp_uio.uio_resid = first_page_bytes;
  fist_print_uios("WRITE (before VOP_READ 1)", &temp_uio);
  error = VOP_READ(hidden_vp, &temp_uio, hidden_ioflag, cr);
  if (error) {
    fist_dprint(5, "VOP_READ returned error - not good\n");
    /* XXX to be checked */
    goto out_free;
  }
  fist_print_uios("WRITE (after VOP_READ 1)", &temp_uio);
  bytes_read = PAGESIZE - temp_iovec[0].iov_len;
  temp_iovec[0].iov_base -= bytes_read;
  temp_iovec[0].iov_len = PAGESIZE;
  /* derot1 block read */
  rot1_decode_block(__FUNCTION__,__LINE__,
		     temp_iovec[0].iov_base, bytes_read, vp);

  /*
   * if num_pages == 1, we already read the page... don't clobber it
   * if num_pages > 1, then we must read the last page, and derot1 it
   * completely, before clobbering it.
   * XXX: if end offset is on page boundary, we don't have to do this.
   */
  if (num_pages > 1) {
    /* read last block XXX check length of file */
    temp_uio.uio_iov = temp_iovec + (num_pages - 1);
    temp_uio.uio_iovcnt = 1;
    temp_uio.uio_loffset = end_loffset - PAGESIZE;
    temp_uio.uio_segflg = UIO_SYSSPACE;
    temp_uio.uio_fmode = FREAD;
    temp_uio.uio_llimit = uiop->uio_llimit;
    temp_uio.uio_resid = PAGESIZE;

    fist_print_uios("WRITE (before VOP_READ 2)", &temp_uio);
    error = VOP_READ(hidden_vp, &temp_uio, hidden_ioflag, cr);
    fist_print_uios("WRITE (after VOP_READ 3)", &temp_uio);
    if (error) {
      fist_dprint(4, "VOP_READ returned error - not good\n");
      /* XXX to be checked */
      goto out_free;
    }
    bytes_read = PAGESIZE - temp_iovec[num_pages - 1].iov_len;
    temp_iovec[num_pages - 1].iov_base -= bytes_read;
    temp_iovec[num_pages - 1].iov_len = PAGESIZE;
    /* derot1 block read */
    rot1_decode_block(__FUNCTION__,__LINE__,
		       temp_iovec[num_pages-1].iov_base, bytes_read, vp);
  }
  /*
   * Now we are ready to write the bytes within the start/end
   * cleartext offsets in the buffers we allocated.
   */
  for (i = 0; i < num_pages; i++) {
    if (i >= real_first_page) {
      bytes_read = PAGESIZE;
      current_base = temp_iovec[i].iov_base;
      if (i == real_first_page) {
#define real_first_page_offset (cleartext_start_loffset - real_start_loffset)
	bytes_read -= real_first_page_offset;
	current_loffset += real_first_page_offset;
	current_base += real_first_page_offset;
#undef real_first_page_offset
      }
      if (current_loffset + bytes_read > cleartext_end_loffset) {
	bytes_read = cleartext_end_loffset - current_loffset;
      }
      if ((error = uiomove(current_base, bytes_read, UIO_WRITE, uiop)))
	break;
    }
    /* encode block before writing */
    rot1_encode_block(__FUNCTION__,__LINE__,temp_iovec[i].iov_base, PAGESIZE, vp);
    current_loffset += bytes_read;
  }
  fist_print_uios("WRITE (after for loop 4)", &temp_uio);

  if (va.va_size < end_loffset) {
    if (va.va_size < cleartext_end_loffset)
      resid -= end_loffset - cleartext_end_loffset;
    else
      resid -= end_loffset - va.va_size;
  }

  /* XXX: no need for full initialization here */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = num_pages;
  temp_uio.uio_loffset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_fmode = uiop->uio_fmode;
  temp_uio.uio_llimit = uiop->uio_llimit;
  temp_uio.uio_resid = resid;

  /*
   * pass operation to hidden filesystem, and return status
   */
  fist_print_uios("WRITE (before write)", &temp_uio);
  error = VOP_WRITE(hidden_vp, &temp_uio, hidden_ioflag, cr);
  fist_print_uios("WRITE (after write)", &temp_uio);

  if (temp_uio.uio_loffset < cleartext_end_loffset) {
    /* incomplete write: this case is an error and should not happen */
    uiop->uio_loffset = temp_uio.uio_loffset;
    uiop->uio_resid = cleartext_end_loffset - temp_uio.uio_loffset;
  } else {
    /* we may have written more than what was asked of us to preserve the
     * enrot1ion/encoding over a whole page
     */
    uiop->uio_loffset = cleartext_end_loffset;
    uiop->uio_resid = 0;
  }
  /* if FAPPEND was used, return offset of 0 to upper level */
  if (ioflag & FAPPEND) {
    uiop->uio_loffset = 0;
  }

out_free:
  for (i = 0; i < num_pages; i++) {
    fist_dprint(6, "PRINT_BASE1 %d: 0x%x (len=%d)\n", i,
		temp_iovec[i].iov_base,
		temp_iovec[i].iov_len);
    fist_dprint(6, "PRINT_BASE2 %d: 0x%x (len=%d)\n", i,
		free_iovec[i].iov_base,
		free_iovec[i].iov_len);
    kmem_free(free_iovec[i].iov_base, PAGESIZE);
  }
  kmem_free(free_iovec, num_pages * sizeof(iovec_t));
  kmem_free(temp_iovec, num_pages * sizeof(iovec_t));

out:
  mutex_exit(&vp->v_lock);

#ifdef FIST_DEBUG
  fist_print_uios("fist_rot1_write (END)", uiop);
#endif /* FIST_DEBUG */

  print_location();
  return (error);
}


static int
fist_rot1_ioctl(
		  vnode_t * vp,
		  int cmd,
		  int arg,
		  int flag,
		  cred_t * cr,
		  int *rvalp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;
  int val = 0;

  fist_dprint(4, "fist_rot1_ioctl vp %x\n", vp);

  /* check if asked for local commands */
  switch (cmd) {
  case FIST_IOCTL_GET_DEBUG_VALUE:
    if (copyin((caddr_t) arg, (caddr_t) & val, sizeof(int))) {
      error = EFAULT;
      goto out;
    }
    val = fist_get_debug_value();
    if (copyout((caddr_t) & val, (caddr_t) arg, sizeof(int))) {
      error = EFAULT;
      goto out;
    }
    error = 0;
    goto out;
    break;

  case FIST_IOCTL_SET_DEBUG_VALUE:
    val = (int) arg;
    if (val < 0 || val > 20) {
      error = EINVAL;
      goto out;
    }
    fist_dprint(6, "IOCTL: got arg %d\n", val);
    fist_set_debug_value(val);
    error = 0;
    goto out;
    break;
  }

  hidden_vp = vntofwn(vp)->fwn_vnodep;
  /* pass operation to hidden filesystem, and return status */
  error = VOP_IOCTL(hidden_vp, cmd, arg, flag, cr, rvalp);

out:
  print_location();
  return (error);
}


static int
fist_rot1_setfl(
		  vnode_t * vp,
		  int oflags,
		  int nflags,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_setfl vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SETFL(hidden_vp, oflags, nflags, cr);

  print_location();
  return (error);
}


static int
fist_rot1_getattr(
		    vnode_t * vp,
		    vattr_t * vap,
		    int flags,
		    cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_getattr vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_GETATTR(hidden_vp, vap, flags, cr);

  if (error) {
    fist_dprint(4, "ERROR: fist_rot1_getattr %d\n", error);
  }
  print_location();
  return (error);
}


static int
fist_rot1_setattr(
		    vnode_t * vp,
		    vattr_t * vap,
		    int flags,
		    cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_setattr vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SETATTR(hidden_vp, vap, flags, cr);

  print_location();
  return (error);
}


static int
fist_rot1_access(
		   vnode_t * vp,
		   int mode,
		   int flags,
		   cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_access vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_ACCESS(hidden_vp, mode, flags, cr);

  print_location();
  return (error);
}


static int
fist_rot1_lookup(
		   vnode_t * dvp,
		   char *name,
		   vnode_t ** vpp,
		   pathname_t * pnp,
		   int flags,
		   vnode_t * rdir,
		   cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_dvp;

  fist_dprint(4, "fist_rot1_lookup dvp %x, rdir %x, name \"%s\"\n",
	      dvp, rdir, (name ? name : "Null"));

  hidden_dvp = vntofwn(dvp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_LOOKUP(hidden_dvp, name, vpp, pnp, flags, rdir, cr);
  /* if no error, interpose vnode */
  if (!error) {
    fist_dprint(6, "ROT1_LOOKUP1: hidden_vp->v_count=%d, vpp->v_type=%d\n",
		(*vpp)->v_count, (*vpp)->v_type);
    *vpp = fist_rot1_interpose(*vpp, dvp->v_vfsp);
    fist_dprint(6, "ROT1_LOOKUP2: vpp->v_count=%d, hidden_vp->v_count=%d, vpp->v_type=%d\n",
		(*vpp)->v_count, vntofwn(*vpp)->fwn_vnodep->v_count,
		(*vpp)->v_type);
  }

  if (error) {
    fist_dprint(4, "ERROR: fist_rot1_lookup %d\n", error);
  }

  print_location();
  return (error);
}


static int
fist_rot1_create(
		   vnode_t * dvp,
		   char *name,
		   vattr_t * vap,
		   vcexcl_t excl,
		   int mode,
		   vnode_t ** vpp,
		   cred_t * cr,
		   int flag  /* XXX: EZK: new in 2.6, why? 64-bit support? */
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_create vp=%x name=\"%s\" mode=%x\n", dvp, name, mode);

  hidden_vp = vntofwn(dvp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_CREATE(hidden_vp, name, vap, excl, mode, vpp, cr, flag);

  /* if no error, interpose vnode */
  if (!error) {
    fist_dprint(6, "ROT1_CREATE1: hidden_vp->v_count %d\n", (*vpp)->v_count);
    *vpp = fist_rot1_interpose(*vpp, dvp->v_vfsp);
    fist_dprint(6, "ROT1_CREATE2: vpp->v_count %d, hidden_vp->v_count %d\n",
		(*vpp)->v_count, vntofwn(*vpp)->fwn_vnodep->v_count);
  }

  print_location();
  return (error);
}


static int
fist_rot1_remove(
		   vnode_t * vp,
		   char *name,
		   cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_remove vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_REMOVE(hidden_vp, name, cr);

  print_location();
  return (error);
}


static int
fist_rot1_link(
		 vnode_t * tdvp,
		 vnode_t * svp,
		 char *name,
		 cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_tdvp, *hidden_svp;

  fist_dprint(4, "fist_rot1_link tdvp %x\n", tdvp);

  /* MUST make sure we only hard link into our own file system! */
  if (svp->v_op != &fist_rot1_vnodeops)
    return EXDEV;

  hidden_tdvp = vntofwn(tdvp)->fwn_vnodep;
  hidden_svp = vntofwn(svp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_LINK(hidden_tdvp, hidden_svp, name, cr);

  print_location();
  return (error);
}


static int
fist_rot1_rename(
		   vnode_t * sdvp,
		   char *snm,
		   vnode_t * tdvp,
		   char *tnm,
		   cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_sdvp, *hidden_tdvp;

  fist_dprint(4, "fist_rot1_rename sdvp %x, tdvp %x\n", sdvp, tdvp);

  hidden_sdvp = vntofwn(sdvp)->fwn_vnodep;
  hidden_tdvp = vntofwn(tdvp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_RENAME(hidden_sdvp, snm, hidden_tdvp, tnm, cr);

  print_location();
  return (error);
}


static int
fist_rot1_mkdir(
		  vnode_t * dvp,
		  char *name,
		  vattr_t * vap,
		  vnode_t ** vpp,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_dvp;

  fist_dprint(4, "fist_rot1_mkdir vp %x\n", dvp);

  hidden_dvp = vntofwn(dvp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_MKDIR(hidden_dvp, name, vap, vpp, cr);

  /* if no error, interpose vnode */
  if (!error) {
    fist_dprint(6, "ROT1_MKDIR1: hidden_vp->v_count %d\n", (*vpp)->v_count);
    *vpp = fist_rot1_interpose(*vpp, dvp->v_vfsp);
  }

  print_location();
  return (error);
}


static int
fist_rot1_rmdir(
		  vnode_t * vp,
		  char *name,
		  vnode_t * cdir,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp, *hidden_cdir;

  fist_dprint(4, "fist_rot1_rmdir vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;
  hidden_cdir = vntofwn(cdir)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_RMDIR(hidden_vp, name, hidden_cdir, cr);

  print_location();
  return (error);
}


static int
fist_rot1_readdir(
		    vnode_t * vp,
		    uio_t * uiop,
		    cred_t * cr,
		    int *eofp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_readdir vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_READDIR(hidden_vp, uiop, cr, eofp);

  print_location();
  return (error);
}


static int
fist_rot1_symlink(
		    vnode_t * vp,
		    char *linkname,
		    vattr_t * vap,
		    char *target,
		    cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_symlink vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SYMLINK(hidden_vp, linkname, vap, target, cr);

  print_location();
  return (error);
}


static int
fist_rot1_readlink(
		     vnode_t * vp,
		     uio_t * uiop,
		     cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_readlink vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_READLINK(hidden_vp, uiop, cr);

  print_location();
  return (error);
}


static int
fist_rot1_fsync(
		  vnode_t * vp,
		  int syncflag,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_fsync vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_FSYNC(hidden_vp, syncflag, cr);

  print_location();

  return (error);
}


/*
 * Free up a vnode once the last reference count to it
 * has been released (via vn_rele()).
 */
static void
fist_rot1_inactive(
		     vnode_t * vp,
		     cred_t * cr
)
{
  vnode_t *hidden_vp;
  struct fist_rot1info *fwip = NULL;
  int ret;

  fist_dprint(4, "fist_rot1_inactive vp %x\n", vp);

  /* flush all pages if need to */
  if (vp->v_pages) {
    /*
     * using B_INVAL will instruct *_dispose() to remove the
     * mappings of the page from the vnode and the page hash.
     */
    ret = pvn_vplist_dirty(vp, 0, fist_rot1_putapage, B_INVAL, cr);
    if (ret) {
      fist_dprint(6, "ROT1_INACTIVE, pvn_vnlist_dirty returns %d.\n", ret);
    }
  }

  hidden_vp = vntofwn(vp)->fwn_vnodep;

#if 0
  /*
   * XXX: EZK experimental.
   * flush hidden_vp's pages.
   * don't do it: too slow
   */
  if (hidden_vp->v_pages) {
    printk("INACTIVE: hidden_vp has pages to flush...\n");
    ret = VOP_PUTPAGE(hidden_vp, (offset_t) 0, 0, B_INVAL, cr);
    printk("INACTIVE: hidden PUTPAGE returned %d\n", ret);
  }
#endif

  /*
   * This is a tricky function.  We need to first perform the action to the
   * interposed vnode, and then to this vnode.  This call can only have come
   * normally from VN_RELE() which called vn_rele().
   *
   * XXX: should I call VN_RELE on the interposed vnode instead?
   */

  /*
   * XXX: Is this right?  Should I call inactive on the interposed vode as
   * well?  If I do it means that the reference count for an interposed and
   * interposing vnodes are always the same.  That cannot always be true
   * because I can access the interposed filesystem from the original path.
   * It might be best if I ensure that the refcount is exactly 1 here.
   * I must investigate this bug!!! -Erez.
   */

  /* pass operation to hidden filesystem, and return status */
  /* VOP_INACTIVE(hidden_vp, cr); */
  fist_dprint(6, "ROT1_INACTIVE1: hidden_vp->v_count %d\n", hidden_vp->v_count);
  VN_RELE(hidden_vp);
  fist_dprint(6, "ROT1_INACTIVE2: hidden_vp->v_count %d\n", hidden_vp->v_count);

  mutex_enter(&vfstofwi(vp->v_vfsp)->fwi_ht_lock);
  fist_ht_del_vp(hidden_vp, vp->v_vfsp);

  /* free space used by opaque v_data field */
  if (vp->v_data) {
    mutex_destroy(&vntofwn(vp)->fwn_lock);
    kmem_free(vp->v_data, sizeof(fist_rot1node_t));
    vp->v_data = NULL;
  }
  /* free actual vnode */
  kmem_free(vp, sizeof(vnode_t));
  /* XXX: should I reset this vnode? */
  /* vp = NULL; */

  /* decrement interposed vnodes counter */
  fwip = vfstofwi(vp->v_vfsp);
  fwip->fwi_num_vnodes--;

  mutex_exit(&vfstofwi(vp->v_vfsp)->fwi_ht_lock);

  print_location();
  return;
}


static int
fist_rot1_fid(
		vnode_t * vp,
		fid_t * fidp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_fid vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_FID(hidden_vp, fidp);

  print_location();
  return (error);
}


static void
fist_rot1_rwlock(
		   vnode_t * vp,
		   int write_lock
)
{
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_rwlock vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  VOP_RWLOCK(hidden_vp, write_lock);

  print_location();
  return;
}


static void
fist_rot1_rwunlock(
		     vnode_t * vp,
		     int write_lock
)
{
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_rwunlock vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  VOP_RWUNLOCK(hidden_vp, write_lock);

  print_location();
  return;
}


static int
fist_rot1_seek(
		 vnode_t * vp,
		 offset_t offset,
		 offset_t * offsetp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_seek vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SEEK(hidden_vp, offset, offsetp);

  print_location();
  return (error);
}


static int
fist_rot1_cmp(
		vnode_t * vp1,
		vnode_t * vp2
)
{
  int error = EPERM;
  vnode_t *hidden_vp1;
  vnode_t *hidden_vp2;

  fist_dprint(4, "fist_rot1_cmp vp1 %x, vp2\n", vp1, vp2);

  hidden_vp1 = vntofwn(vp1)->fwn_vnodep;
  hidden_vp2 = vntofwn(vp2)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_CMP(hidden_vp1, hidden_vp2);

  print_location();
  return (error);
}


static int
fist_rot1_frlock(
		   vnode_t * vp,
		   int cmd,
		   struct flock64 * bfp,
		   int flag,
		   offset_t offset,
		   cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_frlock vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_FRLOCK(hidden_vp, cmd, bfp, flag, offset, cr);

  print_location();
  return (error);
}


static int
fist_rot1_space(
		  vnode_t * vp,
		  int cmd,
		  struct flock64 * bfp,
		  int flag,
		  offset_t offset,
		  cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_space vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SPACE(hidden_vp, cmd, bfp, flag, offset, cr);

  print_location();
  return (error);
}


static int
fist_rot1_realvp(
		   vnode_t * vp,
		   vnode_t ** vpp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_realvp vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_REALVP(hidden_vp, vpp);

  /* if no error, interpose vnode */
  if (!error) {
    /* XXX: is this right? Is the vfs type passed to _interpose right? */
    *vpp = fist_rot1_interpose(*vpp, vp->v_vfsp);
  }
  print_location();
  return (error);
}

/* mmap functions were here and were moved to fist_rot1_mmap.c */


static int
fist_rot1_poll(
		 vnode_t * vp,
		 short events,
		 int anyyet,
		 short *reventsp,
		 pollhead_t ** phpp
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_poll vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_POLL(hidden_vp, events, anyyet, reventsp, phpp);

  print_location();
  return (error);
}


static int
fist_rot1_dump(
		 vnode_t * vp,
		 caddr_t addr,
		 int ldbn,
		 int dblks
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_dump vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_DUMP(hidden_vp, addr, ldbn, dblks);

  print_location();
  return (error);
}


static int
fist_rot1_pathconf(
		     vnode_t * vp,
		     int cmd,
		     u_long * valp,
		     cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_pathconf vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_PATHCONF(hidden_vp, cmd, valp, cr);

  print_location();
  return (error);
}


static int
fist_rot1_pageio(
		   vnode_t * vp,
		   page_t * pp,
		   u_offset_t io_off,
		   u_int io_len,
		   int flags,
		   cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_pageio vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_PAGEIO(hidden_vp, pp, io_off, io_len, flags, cr);

  print_location();
  return (error);
}


static int
fist_rot1_dumpctl(
		    vnode_t * vp,
		    int free
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_dumpctl vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_DUMPCTL(hidden_vp, free);

  print_location();
  return (error);
}


static void
fist_rot1_dispose(
		    vnode_t * vp,
		    page_t * pp,
		    int fl,
		    int dn,
		    cred_t * cr
)
{
  fist_dprint(4, "fist_rot1_DISPOSE vp 0x%x, page 0x%x fl=0x%x, dn=0x%x\n", vp, pp, fl, dn);
  /* fist_rot1_DISPOSE vp 0x600cf2b0, page 0x104f59c0 fl=0x8000, dn=0x2000 */

  fs_dispose(vp, pp, fl, dn, cr);
  return;
}


static int
fist_rot1_setsecattr(
		       vnode_t * vp,
		       vsecattr_t * vsap,
		       int flag,
		       cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_setsecattr vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_SETSECATTR(hidden_vp, vsap, flag, cr);

  print_location();
  return (error);
}


static int
fist_rot1_getsecattr(
		       vnode_t * vp,
		       vsecattr_t * vsap,
		       int flag,
		       cred_t * cr
)
{
  int error = EPERM;
  vnode_t *hidden_vp;

  fist_dprint(4, "fist_rot1_getsecattr vp %x\n", vp);

  hidden_vp = vntofwn(vp)->fwn_vnodep;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_GETSECATTR(hidden_vp, vsap, flag, cr);

  print_location();
  return (error);
}
