/*------------------------------------------------------
 *
 * MIME.C
 *
 * This file is the core component of the ripMIME
 * and other packages
 *
 * Sun Jan 28 16:16:12 EST 2001 worker <worker@xamime>
 * 	Prerelease stage 1 completed.
 * Sat Jan 13 21:05:51 EST 2001 worker <worker@xamime>
 * 	Going into PreRelease stage 1.
 *
 * Sat Nov 25 16:25:06 SAST 2000 worker <worker@xamime>
 * 	Add some more comments, as more people are reading this
 * 	c code ;)
 *
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>

#include "debugging.h"
#include "strlower.h"
#include "mime.h"

#define _SL 1

#define _ENC_UNKNOWN 0
#define _ENC_BASE64 1
#define _ENC_PLAINTEXT 2
#define _ENC_QUOTED 3
#define _ENC_EMBEDDED 4
#define _ENC_NOFILE -1
#define _MIME_CHARS_PER_LINE 32
#define _MIME_MAX_CHARS_PER_LINE 76
#define _MAX_RECURSION_LEVEL 3

/* single-character decode */
#define UUDEC(c)  (((c) - ' ') & 077)


/*----Structures-----*/
struct _MIME_info {
	int lastlinewasboundary;	/* Last line we read in had the boundary in it */
	int lastlinewasfrom;		 /* last line we read in had the From: line proceeded by a blank */
	int lastlinewasblank;		/* (part of lastlinewasfrom) */
	int lastencoding;			 /* the encoding specifier last read in */
	int boundarylen;			  /* Length of the boundary */
	char *inputok;				 /* Indicator of file reading sanity */
	char boundary[513]; 		 /* The boundary */
	char filename[513];		  /* Filename of current attachment */
	};
	

/* our base 64 decoder table */
unsigned char b64[256];

char OK[]="OKAY";

/* HexConversion table, rather than doing a atoi (with base 16) */
int hexconv[256];

/* How many attachments have we produced */
int filecount = 0;

/* If the attachment has no name, then we will need it to have one (else how are you going store it on your system ? */
char blankfileprefix[128]="textfile";

/* Verbosity - talk as we walk through the MIMEpack to the shell */
int verbosity = 0;

/* Our logging options */
int syslogging = 0;
int stderrlogging = 1;


/* The name of the file which will contain all the non-attachment 
 * lines from the MIMEpack
 */
char headersname[1024]="_headers_";

/* The variable that indicates (if not zero) that we are to dump our
 * non-attachment lines to the file "headersname"
 */
int dumpheaders = 0;


/* Attachment count - how many attachment with FILENAMES */
int _attachment_count = 0;

/* Current line read number */
int _current_line = 0;

/* if we dont want nameless-files, this is non-zero */
int _no_nameless = 0;

/* File pointer for the headers output */
FILE *headers;

char _tmpdir[1024];

struct _tbsnode {
	char *boundary;
	struct _tbsnode *next;
	};
	
struct _tbsnode *boundarystack = NULL;
char *boundarystacksafe;

/*----------------------------------------------------------
 * MIME_BS_push()
 *
 * Push a boundary onto the stack */
int MIME_BS_push( char *boundary ){
	
	struct _tbsnode *node = malloc(sizeof(struct _tbsnode));
	
	if (node){
		node->next = boundarystack;
		boundarystack = node;
		boundarystack->boundary = strdup(boundary);
		_debug("MIME_BS_push(): BOundary '%s' pushed into stack",boundarystack->boundary);
		}
	else {
		syslog(_SL,"MIME_BS_push(): Cannot allocate memory for PUSH(), %s",strerror(errno));
		}
		
	return 0;
	}


/*----------------------------------------------------------
 * MIME_BS_pop()
 *
 * pop's the top boundar off */
char *MIME_BS_pop( void ){
	
	struct _tbsnode *node = boundarystack;
	if (boundarystack){
		boundarystack = boundarystack->next;
		boundarystacksafe = node->boundary;
		free(node);
		}
		
	return boundarystacksafe;
	}

/*----------------------------------------------------------
 * MIME_BS_top()
 *
 * returns the top item in the stack, without popping off */
char *MIME_BS_top( void ){
	
	if (boundarystack){
		return boundarystack->boundary;
		}
	else return NULL;
	}

/*----------------------------------------------------------
 * MIME_BS_cmp()
 *
 */
int MIME_BS_cmp( char *boundary ){
	
	char *top, *spot;
	int spin=1;
	struct _tbsnode *node=boundarystack;
		
	while((node)&&(spin)){
		top = node->boundary;
		spot = strstr(boundary,top);
		if (spot) spin = 0;
		else node = node->next;
		_debug("MIME_BS_cmp: resulted = %s, comparing %s in %s",spot, top,boundary);
		}
		
	if (spin == 0) return 1; else return 0;
	}
	
	
/*----------------------------------------------------------
 * MIME_set_tmpdir()
 *
 */
int MIME_set_tmpdir( char *setto ){
	
	sprintf(_tmpdir,"%s",setto);
	
	return 0;
	}
	
	

/*-------------------------------------------------------
 * MIME_set_blankfileprefix
 * 
 * Sets the prefix for blank files
 *
 * the blank file prefix is used when (typ) we have text
 * attachments, ie, when there is both a HTML and a 
 * plain-text version of the message in the same email.
 *
 */
int MIME_set_blankfileprefix( char *prefix ){
	
	/* copy over the prefix name, ensure that we 
	 * dont break buffers here */
	strncpy(blankfileprefix,prefix,127);
	
	return 0;
	}
	
	
/*-------------------------------------------------------
 * MIME_set_verbosity
 * 
 * By default, MIME reports nothing as its working
 * Setting the verbosity level > 0 means that it'll
 * report things like the name of the files it's
 * writing/extracting.
 *
 */
int MIME_set_verbosity( int level ){
	
	verbosity = level;
	
	return 0;
	}
	

/*-------------------------------------------------------
 * MIME_set_dumpheaders
 *
 * By default MIME wont dump the headers to a text file
 * but at times this is useful esp for checking
 * for new styles of viruses like the KAK.worm
 *
 * Anything > 0 will make the headers be saved
 *
 */
int MIME_set_dumpheaders( int level ){
	
		dumpheaders = level;
		
		return 0;
		}
	

/*------------------------------------------------------
 * MIME_set_headersname
 * 
 * by default, the headers would be dropped to a file
 * called '_headers_'.  Using this call we can make
 * it pretty much anything we like 
 */
int MIME_set_headersname( char *fname ){
		
		strncpy(headersname, fname, 1023);
		
		return 0;
		}
		
		
/*-------------------------------------------------------
 * MIME_set_syslogging
 *
 * By default, syslogging is off, setting the "level" to a value above zero(0)
 * will make MIME log to the system 
 */
int MIME_set_syslogging( int level ){
	
	syslogging = level;
	
	return 0;
	}
	
	
/*-------------------------------------------------------
 * MIME_set_stderrlogging
 *
 * By default logging is done via stderr, set this to zero (0) to turn off
 */
int MIME_set_stderrlogging(int level){
	
		stderrlogging = level;
		
		return 0;
		}
		

int MIME_set_no_nameless( int level ){
	
	_no_nameless = level;
	
	return 0;
	}
	
int MIME_get_attachment_count( void ) {
	
	return _attachment_count;
	}
	
		
/*-------------------------------------------------------
 * MIME_tmpnam()
 *
 * Creates a temporary filename
 *
 * NOTE - P_tmpdir comes from stdio.h, see 'man tmpfile'
 */
int MIME_tmpnam( char *_temp_fname ){
	
	int i;
	char sequence[12];
	FILE *f;
	
	do {
		for (i=0; i <11; i++) sequence[i] = random()%(122 -97) +97;
		sequence[11] = '\0';
		sprintf(_temp_fname,"%s/%s_ripmime_%d",_tmpdir, sequence, getpid());
		} while ((f=fopen(_temp_fname,"r")));
		
	return 0;
	}
	
		
/*-------------------------------------------------------
 * MIME_base64_init
 *
 * Prepare the base64 decoding array
 *
 */
int MIME_base64_init( void ){

	int i;
	
	/* preset every encodment to 0x80 */
   for (i = 0; i < 255; i++) b64[i] = 0x80;
   
   /* put in the capital letters */
   for (i = 'A'; i <= 'Z'; i++) b64[i] = 0 + (i - 'A');
   
   /* put in the lowere case letters */
   for (i = 'a'; i <= 'z'; i++) b64[i] = 26 + (i - 'a');
   
   /* the digits */
   for (i = '0'; i <= '9'; i++) b64[i] = 52 + (i - '0');
   
   /* and our special chars */
   b64['+'] = 62; b64['/'] = 63; b64['='] = 0;

	return 0;
	}





/*-------------------------------------------------------------------*
 * MIME_find_next_MIME_boundary
 *
 * moves to the position immediately after the next boundary 
 *
 */
int MIME_find_next_boundary( FILE *f, struct _MIME_info *info ){
	
	char line[1024];
		
	_debug("MIME_find_next_boundary: start.");
	while((info->inputok = fgets(line,1023,f))){

		_current_line++;
		
		/* if we're supposed to dump all headers... */		
		if ((dumpheaders != 0)&&(headers)) fprintf(headers,"%s",line);

		/* check we've not encounted the blank line, ie end of headers... */		
		if (MIME_BS_cmp(line)!=0){
			info->lastlinewasboundary = 1;
			break;
			}
		if ((line[1] == 10) || (line[1] == 12)||(line[0] == 10) || (line[0] == 12)){
			info->lastlinewasblank=1;
			}
		else
		if ((info->lastlinewasblank == 1)&&((strncasecmp(line,"from ",5)==0)||(strncasecmp(&line[1],"from ",5)==0))){
			info->lastlinewasfrom=1;
			break;
			}
		else info->lastlinewasblank = 0;			

		/* see if we have a line containing the boundary */
/*vPR3		result = strstr(line,info->boundary);*/
		} /* while */

	return 0;
	}
 




/*-------------------------------------------------------------------*
 * MIME_clean_MIME_filename - 
 * 	Cleans up the file name, ie, strips it of unrequired
 *	 inverted comma's etc.
 */
int MIME_clean_MIME_filename( char *fname ){
	
	int fnl = strlen(fname); /* FileNameLength */
	int i;
	char *fnp, *iso, fname_copy[1024];
	char tmp[1024];
	char *p = strrchr(fname,'/');

	/* scan out any directory separators */
	if (p) 
	{
		p++; sprintf(tmp,"%s",p);
		strcpy(fname,p);
	}
	else if ( (p = strrchr(fname,'\\'))) 
	{
		p++; sprintf(tmp,"%s",p);
		strcpy(fname,p);
	}
		

	
	if ((fnl > 2)&&(!strchr(fname,' '))) {  /* if there's no spaces in our MIME_filename */

		if (strstr(fname,"=?")) { /* we may have an ISO filename */
			fnp = fname;
			iso = strtok(fname,"?"); /* iso encoding prefix */
			if (iso) iso = strtok(NULL,"?"); /* The leading = */
			if (iso) iso = strtok(NULL,"?"); /* unknown singe char */
			if (iso) iso = strtok(NULL,"?"); /* filename! */
			if (iso) {
				MIME_decode_text_line(iso);
				sprintf(fname_copy,"%s",iso);
				sprintf(fname,"%s",fname_copy);
				}
			}			
		/* if the MIME_filename starts and ends with "'s */
		if ((fname[0] == '\"') && (fname[fnl-1] == '\"')) {
			
			/* reduce the file namelength by two*/
			fnl=-2;
			
			/* shuffle the MIME_filename chars down */
			for (i = 0; i < fnl;) fname[i] = fname[++i];
			
			/* terminate the string */
			fname[fnl] = '\0';
			} /* if */
		} /* if */
	
	return 0;
	}
				
			
/*-------------------------------------------------------------------*
 * quick_clean_filename - 
 * 	Cleans up filename by replacing non alfanum chars with '_'
 */
void quick_clean_filename( char *fname )
{
	char tmp[1024];
	char *p = strrchr(fname,'/');

	/* scan out any directory separators */
	if (p) 
	{
		p++; sprintf(tmp,"%s",p);
		strcpy(fname,p);
	}
	else if ( (p = strrchr(fname,'\\'))) 
	{
		p++; sprintf(tmp,"%s",p);
		strcpy(fname,p);
	}
		

  while (*fname) {
	if ( !isspace(*fname) && !isalnum(*fname) && (*fname != '.') ) *fname='_'; 
	fname++;
  }
}

/*-------------------------------------------------------
 * MIME_getchar_start
 *
 * This routine is actually a copy of MIME_getchar()
 * except that it'll return ONLY a valid MIME-encoded
 * char, rather than also a \n
 *
 */

int MIME_getchar_start( FILE *f ){

	int c;

	/* loop for eternity, as we're "returning out" */	
	while (1) {

		/* get a single char from file */
		c = fgetc(f); 

		/* if that char is an EOF, or the char is something beyond
		 * ASCII 32 (space) then return */
		if ((c == EOF) || (c > ' ')){
			return c;
			}

		} /* while */

	/* Although we shouldn't ever actually get here, we best put it in anyhow */
	return EOF;
	}
	


/*-------------------------------------------------------
 * MIME_getchar()
 *
 * get the next char from the MIME-encoded-stream
 *
 */

int MIME_getchar( FILE *f ){

	int c;

	/* loop for eternity, as we're "returning out" */	
	while (1) {
		/* get a single char from file */
		c = fgetc(f); 
		/* if that char is an EOF, or the char is something beyond
		 * ASCII 32 (space) then return */
		if ((c == EOF) || (c > ' ')|| (c == '\n')){
			return c;
			}
			
		} /* While */
	
	/* Though we shouldn't get this far, we should put in something just incase! */	
	return EOF;
	}
	



/*-------------------------------------------------
 * MIME_decode_text_line
 *
 * This function just decodes a single print-quotable line into
 * it's normal/original format
 */
int MIME_decode_text_line( char *line ) {
	
	char c;								/* The Character to output */
	int op, ip; 						/* OutputPointer and InputPointer */
	int slen = strlen(line); /* Length of our line */

	/* Initialise our "pointers" to the start of the encoded string */
	ip=op=0;

	/* for every character in the string... */
	for (ip = 0; ip < slen; ip++){
		
		c = line[ip];
		
		/* if we have the quote-printable esc char, then lets get cracking */
		if (c == '=') {

			/* if we have another two chars... */
			if (ip <= (slen-2)){

				/* convert our encoded character from HEX -> decimal */
				c = (char)hexconv[(int)line[ip+1]]*16 +hexconv[(int)line[ip+2]];
							
				/* shuffle the pointer up two spaces */
				ip+=2;
				} /* if there were two extra chars after the ='s */
				
			/* if we didn't have enough characters, then  we'll make the char the 
			 * string terminator (such as what happens when we get a =\n
			 */
			else {
				line[ip] = '\0';
				} /* else */
				
			} /* if c was a encoding char */
			
		/* put in the new character, be it converted or not */
		line[op] = c;
		
		/* shuffle up the output line pointer */
		op++;
		} /* for */
		
	/* terminate the line */
	line[op]='\0';		
		
	return 0;
	
	}
	
	
				
/*-------------------------------------------------------------------*
 * MIME_decode_text - 
 * 	Decodes text type streams 
 */
int MIME_decode_text( FILE *f, char *unpackdir, struct _MIME_info *info ) {	

	FILE *of; 											/* output file */
	int linecount = 0;  			/* The number of lines */
	char fullfilename[1024]=""; 	/* Filename of the output file */
	char line[1024]; 							/* The input lines from the file we're decoding */
/*vPR3	char *boundaryhit = NULL; */
		
	/* generate the full MIME_filename */
	sprintf(fullfilename,"%s/%s",unpackdir,info->filename);
	
	/* check our input stream sanity */
	if (!f) {
		if (syslogging > 0) syslog(_SL,"Error: ripMIME print-quotable input stream broken.");
		if (stderrlogging > 0) fprintf(stderr,"Error: ripMIME print-quotable input stream broken.");
		exit(_EXITERR_PRINT_QUOTABLE_INPUT_NOT_OPEN);
		}
		
	/* if our FILE stream is open */	
	if (f) {	
	
		/* try open the new file to write data to */
		of = fopen(fullfilename,"w");
		_debug("MIME_decode_text: opened new file '%s'",fullfilename);
		
		/* if we cannot open the file, then report to syslog */
		if (!of) {
			if (syslogging > 0) syslog(_SL,"Error: cannot open %s for writing",fullfilename);
			if (stderrlogging > 0) fprintf(stderr,"Error: cannot open %s for writing",fullfilename);
			exit(_EXITERR_PRINT_QUOTABLE_OUTPUT_NOT_OPEN);
			}
	
		/* while we can still get lines of data */
		while ((info->inputok=fgets(line,1023,f))&&(of)){

			_current_line++;
			
			/* check we've not encounted the blank line, ie end of 
			 * headers... */		
			if ((line[1] == 10) || (line[1] == 12)||(line[0] == 10) || (line[0] == 12)){
				info->lastlinewasblank=1;
				}
			else
			if ((info->lastlinewasblank == 1)&&((strncasecmp(line,"from ",5)==0)||(strncasecmp(&line[1],"from ",5)==0))){
				info->lastlinewasfrom=1;
				break;
				}
			else info->lastlinewasblank = 0;			
									
			/* if we have a boundary-specified,  and the line doesn't contain our boundary */			
/*vPR3			if (info->boundarylen > 0) boundaryhit = strstr(line, info->boundary);
			if (!boundaryhit){ */
			if (!MIME_BS_cmp(line)){

					/* if we're encoded with quoted-printable, then we better un-code it */
					if (info->lastencoding == _ENC_QUOTED)  MIME_decode_text_line(line);

					/* print the line to the new output file */
					fprintf(of,"%s",line);
		
					/* increment our line count */			
					linecount++;
				
					} /* if the line didn't contain a boundary */
				 
				else {
					/* else if our line was the boundary, set the boundary flag and exit out */
					info->lastlinewasboundary = 1;
					break;
					} /* else */

			} /* while */

		/* if the file is still safely open */		
		if (of) {
			
			/* close it */
			fclose(of);
			_debug("MIME_decode_text: Closed text file.");
			/* if we wrote nothing, then trash the file */
			if (linecount == 0) {
				unlink(fullfilename);
			} else {
				if (verbosity > 0){
					if (info->lastencoding == _ENC_QUOTED) fprintf(stdout,"Extracted quoted-printable: %s\n",fullfilename); 
					else fprintf(stdout,"Extracted plaintext: %s\n",fullfilename);
					} /* if verbosity */
				} /* else */
			} /* if file still safely open */
		} /* if main input file stream was open */
		
	return 0;
	}
			
	
	
	
	
/*-------------------------------------------------------------------*
 * MIME_decode64 - 
 * 	Decodes base64 type streams 
 *
 * This routine is very very very important, it's the key to ensuring
 * we get our attachments out of the email file without trauma!
 *
 * NOTE - this has been -slightly altered- in order to make provision 
 * of the fact that the attachment may end BEFORE the EOF is received
 * as is the case with multiple attachments in email.  Hence, we 
 * now have to detect the start character of the "boundary" marker
 *
 * I may consider testing the 1st n' chars of the boundary marker
 * just incase it's not always a hypen '-'.
 *
 */
int MIME_decode_64( FILE *f, char *unpackdir, struct _MIME_info *info ){

	int i;
	int t_length = 0; /* the length of the current line we're decoding */
	int p_length = -1; /* the length of the PREVIOUS line we were decoding */
	int cr_count = 0; /* the number of consecutive \n's we've read in, used to detect End of B64 enc */
	int stopcount = 0; /* How many stop (=) characters we've read in */
	int eom_reached = 0; /* flag to say that we've reached the End-Of-MIME encoding. */
	int status = 0; /* Overall status of decoding operation */
	int c; /* a single char as retrieved using MIME_get_char() */
	int char_count = 0; /* How many chars have been received */
	long int bytecount=0; /* The total file decoded size */
	char output[3]; /* The 4->3 byte output array */
	char input[4]; /* The 4->3 byte input array */
	char fullMIME_filename[1024]=""; /* Full Filename of output file */
	FILE *of; /* output file pointer */

	/* generate the MIME_filename, and open it up... */
	sprintf(fullMIME_filename,"%s/%s",unpackdir,info->filename);
	of = fopen(fullMIME_filename,"wb");
	
	
	/* if we were unable to open the output file, then we better log an error and drop out */
	if (!of) {
		if (syslogging > 0) syslog(_SL,"Error: Cannot open output file %s for BASE64 decoding.",fullMIME_filename);
		if (stderrlogging > 0) fprintf(stderr,"Error: Cannot open output file %s for BASE64 decoding.",fullMIME_filename);
		exit(_EXITERR_BASE64_OUTPUT_NOT_OPEN);
		}

	
	/* collect prefixing trash (if any, such as spaces, CR's etc)*/
	c = MIME_getchar_start(f);
	
	/* and push the good char back */
	ungetc(c,f);
		
	/* do an endless loop, as we're -breaking- out later */
	while (1) {

		/* Initialise the decode buffer */
		input[0] = input[1] = input[2] = input[3] = 0;
		
		/* snatch 4 characters from the input */
		for (i = 0; i < 4; i++) {
	    
			/* get a char from the filestream */
			c = MIME_getchar(f);

			/* if we get a CR then we need to check a few things...*/
			if (c == '\n') {
				
				/* if we've already got our encoded-line length */
				if (p_length != -1) {
					
					/* if the lengths DIFFER, then we have the end of our encoding */
					if (p_length != t_length) {
						eom_reached++;
						break;
						}
						
				} else if (p_length == -1) {
					p_length = t_length;
					}
											
				/* increment the number of -consecutive- CR's we've received... */
				cr_count++;
				
				/* if we get two consecutive CR's, then we're at the end of the line */
				if (cr_count > 1){
					
					/* set the EOM (End of MIME) reached flag */
					eom_reached++;
					break;
					}
				/* else, if we are only on our 1st CR, then it means we have reached the end of the MIME encoded data line */
				else {
					/* reset how many chars we're read in for this line */
					char_count = 0;
					t_length = 0;
					/* push back the character read and try read in again */
					i--;
					continue;
					} /* else */
					
				} /* if c was a \n */			
				
			/* else if we didn't get a CR, then reset the CR counter at least */
			else {
				cr_count=0;
				t_length++;
				}
				
			
			/* if we get an EOF char, then we know something went wrong */
			if ( c == EOF) {
				if (syslogging > 0) syslog(_SL, "Errror: input stream broken for base64 decoding for file %s\n",info->filename);
				status = -1;
				fclose(of);
				return status;
				break;
				} /* if c was the EOF */

			/* assuming we've gotten this far, then we increment the char_count */
			char_count++;
							
			/* if we detect the "stopchar" then we better increment the STOP counter */
			if (c == '=') {
				stopcount++;\
				}
							
			/* test for and discard invalid chars */
			if (b64[c] == 0x80) {
				i--;
				continue;
				}
					
			/* do the conversion from encoded -> decoded */
			input[i] = (char)b64[c];
				
			} /* for */
		
		/* now that our 4-char buffer is full, we can do some fancy bit-shifting and get the required 3-chars of 8-bit data */
		output[0] = (input[0] << 2) | (input[1] >> 4);	
		output[1] = (input[1] << 4) | (input[2] >> 2);
		output[2] = (input[2] << 6) | input[3];

		/* determine how many chars to write write and check for errors if our input char count was 4 then we did receive a propper 4:3 Base64 block, hence write it */		
		if (i == 4) {
			if (fwrite(output, 1, 3-stopcount, of) == EOF) {
				/* if there was an error in writing to the file, then we better notify and exit */
				if (syslogging > 0) syslog(_SL,"Error: Cannot write data to %s",fullMIME_filename);
				if (stderrlogging > 0) fprintf(stderr,"Error: Cannot write data to %s",fullMIME_filename);
				exit(_EXITERR_BASE64_UNABLE_TO_OUTPUT);
		   	}
			}
			
		/* tally up our total byte conversion count */
		bytecount+=(3 -stopcount);
						
		/* if we wrote less than 3 chars, it means we were at the end of the encoded file thus we exit */
		if ((eom_reached)||(stopcount > 0)) {
			
			/* close the output file, we're done writing to it */
			fclose(of);
			
			/* if we didn't really write anything, then trash the  file */
			if (bytecount == 0) { 
				unlink(fullMIME_filename);
			} else {
				if (verbosity > 0) fprintf(stdout,"Extracted Base64: %s\n",fullMIME_filename);
				}
         return 0;
         } /* if End-of-MIME or Stopchars appeared */
				
		} /* while */


	return status;

	}

	
	



	
/*----------------------------------------------------------
 * MIME_read
 *
 * This routine reads in from stdin and saves everything
 * to the file mpname, until EOF is receive
 *
 * Equivilant under BASH is just 'cat arbfile > mpname'
 *
 */
int MIME_read( char *mpname ){

	char c;
	long int fsize=-1;

	/* open up our input file */	
	FILE *fout = fopen(mpname,"w");
	

	/* check that out file opened up okay */
	if (!fout) {
		if (syslogging > 0) syslog(_SL,"Error: Cannot open file %s for writing... check permissions perhaps?",mpname);
		if (stderrlogging > 0) fprintf(stderr,"Error: Cannot open file %s for writing... check permissions perhaps?",mpname);
		exit(_EXITERR_MIMEREAD_CANNOT_OPEN_OUTPUT);
		}
		
	/* assuming our file actually opened up */
	if (fout) {

		fsize=0;
		
		/* while there is more data, consume it */
		while ((c = getc(stdin)) != EOF) {
	
			/* write it to file */
			if (fputc(c,fout) != EOF) { fsize++; } else {
				if (syslogging > 0) syslog(_SL,"Error: Cannot write to file %s... maybe you are out of space?", mpname);
				if (stderrlogging > 0) fprintf(stderr,"Error: Cannot write to file %s... maybe you are out of space?",mpname);
				exit(_EXITERR_MIMEREAD_CANNOT_WRITE_OUTPUT);
				}
			}		
 
 		/* clean up our buffers and close */
 		fflush(fout);fclose(fout);
		
		} /* end if fout was received okay */
		
	/* return our byte count in KB */
	return (int)(fsize /1024);
	}
	

/*---------------------------------------------------
 * MIME_collect_headers()
 *
 * extracts header information required to determin
 * what attachments we have etc
 *
 */
int MIME_collect_headers( FILE *f, struct _MIME_info *info ){

	char line[512];
	char lline[512];
	char *fnp, *fnp2, *fnp3; /* file name pointer */
	char *bp, *bp2, *bp3;
	
	/* assume we have no file type encoding at all, ie, it's just the mail message itself */
	info->lastencoding = _ENC_UNKNOWN;	


	/* create a temporary file name, just in case we dont find one here */
	sprintf(info->filename,"%s%d", blankfileprefix, filecount);
	
	/* move the temp file creator counter along one, this
	 * is so that on fast computers, when we come back and
	 * perhaps have another unnamed file, we don't overwrite
	 * the previous one (see the above file name creating
	 * sprintf()
	 */
	filecount++;

	_debug("MIME_collect_headers: fname=%s, enc=%d, filecount=%d",info->filename, info->lastencoding, filecount);

	if (!f) _debug("MIME_collect_headers: file stream \"f\" is not open!");

	/* While we're getting input without hitting the end of the file (and this will happen) */	
	while ((info->inputok = fgets(line,511,f))){
		_current_line++;
		_debug("MIME_collect_headers: Read:%d:%s",_current_line, line);
	
		
		/* check we've not encounted the blank line, ie end of  headers... */		
		if ((line[1] == 10) || (line[1] == 12)||(line[0] == 10) || (line[0] == 12)){
			info->lastlinewasblank=1;
			_debug("MIME_collect_headers: ");
			break; 
			}
		else info->lastlinewasblank = 0;			
			
		/* else make a copy of the string, and conver it to lowercase */
		strcpy(lline, line); strlower(lline);	
		
		/* if the line has a substring of "name=" in it, then we've
		 * located the MIME_filename portion, note this may appear TWICE
		 * in some headers, once as name= and the other as MIME_filename=
		 */
		if ((fnp = strstr(lline,"name="))) {
			fnp3 = lline; /* setup our starter markers */
			fnp2 = line;
			while (fnp3 != fnp){  /* until we've matched bp3 and bp... */
				fnp2++; /* move bp2 by one char */
				fnp3++; /* move bp3 by one char */
				}
			fnp = fnp2 +strlen("name="); /* we now have the uncase-densensitised bp */
			if ((*fnp) == '\"') fnp++;		/* if the first char is a quote, del the char */
			fnp2 = strtok(fnp,"\n\r\"");	/* tokenise to extract file name */
			sprintf(info->filename,"%s",fnp2);		/* Store as our MIME_filename */
			_debug("MIME_collect_headers: Filename found in headers \"%s\"",info->filename);
			} /* if line contained filename= or name= */
			
		/* else if we detect "base64" in the header, then there's
		 * a pretty darn good chance that it's indicating this is
		 * base64 encoded. NOTE - we put this AFTER the MIME_filename
		 * detection, to prevent things like 'MIME_filename="base64" 
		 * fubaring our detection
		 */
		else if (strlen(lline) > 3){
			if (strstr(lline,"base64")) {
				info->lastencoding = _ENC_BASE64;
				_debug("MIME_collect_headers:BASE64 encoding found");
				}
			else 
			if (strstr(lline, "encoding: quoted-printable")){
				info->lastencoding = _ENC_QUOTED;
				_debug("MIME_collect_headers:QUOTED-PRINTABLE encoding found");
				}
			else 
			if (strstr(lline,": text/")){
				info->lastencoding = _ENC_PLAINTEXT;
				_debug("MIME_collect_headers:PLAINTEXT encoding found");
				}
			else 
			if (strstr(lline,"content-type: message/rfc822")){
				info->lastencoding = _ENC_EMBEDDED;
				_debug("MIME_collect_headers: EMBEDDED (RFC822) encoding found");
				} /* if */
			else
			if ((bp=strstr(lline,"boundary="))){
				bp3 = lline; /* setup our starter markers */
				bp2 = line;
				while (bp3 != bp){  /* until we've matched bp3 and bp... */
					bp2++; /* move bp2 by one char */
					bp3++; /* move bp3 by one char */
					}
				bp = bp2 +strlen("boundary="); /* we now have the uncase-densensitised bp */
				
				_debug("MIME_collect_headers: Boundary = %s",bp);

				if ( (*bp) == '\"' ) bp++; 		/* Deal with a quotation if required */
				bp2 = strtok(bp,"\n\r\"");		/* Extract by using tokenisation */

				if (bp2){
					sprintf(info->boundary,"%s",bp2);
					MIME_BS_push(info->boundary);
				} else {								/* Else mark off as an error */
					if (syslogging > 0) syslog(_SL,"Error: could not retrieve boundary from %s",bp);
					if (stderrlogging > 0) fprintf(stderr,"Error: couldn't retreive boundary from %s",bp);
					exit(_EXITERR_UNDEFINED_BOUNDARY);
					}
				} /* If line perhaps contains boundary */
				
			} /* if strlen > 3 */
			
			
		/* if headerdump is on, then dump every line we read in here */
		if ((dumpheaders) && (headers)) {
			fprintf(headers,"%s",line);
			}		
				
		} /* while */	

	_debug("MIME_collect_headers: finished.");
	
	return 0;
	}
	

/*---------------------------------------------------
 * MIME_init_hexconv()
 *
 * Initialises the hex->dec conversion
 *
 */
int MIME_init_hexconv( void ){
	
	hexconv['0'] = 0;
	hexconv['1'] = 1;	
	hexconv['2'] = 2;	
	hexconv['3'] = 3;	
	hexconv['4'] = 4;	
	hexconv['5'] = 5;	
	hexconv['6'] = 6;	
	hexconv['7'] = 7;	
	hexconv['8'] = 8;	
	hexconv['9'] = 9;	
	hexconv['a'] = 10;	
	hexconv['b'] = 11;	
	hexconv['c'] = 12;	
	hexconv['d'] = 13;	
	hexconv['e'] = 14;	
	hexconv['f'] = 15;	
	hexconv['A'] = 10;	
	hexconv['B'] = 11;	
	hexconv['C'] = 12;	
	hexconv['D'] = 13;	
	hexconv['E'] = 14;	
	hexconv['F'] = 15;	
	
	return 0;
	}
	
	
/*----------------------------------------------------------
 * MIME_init
 *
 * Initialises various aspects required for MIME decoding*/
int MIME_init( void ){
	
	/* prepare the base64 conversion dictionary */
	MIME_base64_init();
	MIME_init_hexconv();
	_attachment_count = 0;	
	return 0;
	}
	
/*---------------------------------------------------
 * MIME_unpack
 *
 * Breaks the MIME package into its various files
 *
 * for now we'll use metamail...
 *
 */
int MIME_unpack( char *unpackdir, char *mpname, int current_recursion_level ){

	int from_found;		/* indicates we've located a "From " prefixed line */
	int status = 0;		/* Global status */
	int newline = 0;		/* indicates that the line we just read was a blank */
	int encoding = 0;	/* used to pick up encoding type in the event that we have a file-only email */
	int noname_file = 1; /* set if the file is a no-namer */
	char tmp[1024];		/* temporary, scratchpad string */
	char aline[1024],lline[1024];  /* Our input line and LOWERCASE inputline */
	char *bp, *bp2, *bp3;		/* Boundary scratchpad pointers */
	char *fnp, *fnp2;	/* filename scratchpad pointers */
	struct _MIME_info info;	/* MIME attachment/file information structure */
	FILE *f;			/* Pointer for the MIME file we're going to be going through */


	_debug("MIME_unpack(%s, %s, %d)",unpackdir, mpname, current_recursion_level);
		
			
	if( mpname[0] == '-' && mpname[1] == '\0' ) {
		f = stdin;
	} else {
		f = fopen(mpname,"r");
		if (!f) syslog(_SL,"MIME_unpack: %s",strerror(errno));
		_debug("MIME_unpack: opened f");
		} /* if mailpack name was not stdin */

	/* check that we haven't recursed too far */
	if (current_recursion_level <= _MAX_RECURSION_LEVEL) current_recursion_level++;
	else {
		syslog(_SL,"Error: Maximum recursion depth of %d has been reached",_MAX_RECURSION_LEVEL);
		return 0;
		}
		

	/* check to see if we had problems opening the file */
	if (!f) {
		if (syslogging > 0) syslog(_SL,"Error: could not open mailpack file %s.",mpname);
		if (stderrlogging > 0) fprintf(stderr,"Error: could not open mailpack file %s.",mpname);
		exit(_EXITERR_MIMEUNPACK_CANNOT_OPEN_INPUT_FILE);
		}		

	/* if it actually opened up.. */
	if (f) {

		_debug("MIME_unpack: Initialisation of basic variables");
		/* Initialise some basic variables... */
		info.boundary[0] = '\0';
		info.boundarylen = 0;
		info.lastlinewasboundary = 0;
		info.lastlinewasfrom = 0;
		info.lastlinewasblank = 0;		
		info.inputok = OK;
					
		/* if the user wants us to save the headers, then open up the file where we'll save them */
		if (dumpheaders > 0) {
			
			/* If we dont already have an open headers file (ie from recusion/nesting */
			if (!headers) {
				sprintf(tmp,"%s/%s",unpackdir,headersname);
				headers = fopen(tmp,"w"); /* if the file isn't _ALREADY_ open */
				}
				
			/* if the headers file still doesn't open up */
			if (!headers) {
				if (syslogging > 0) syslog(_SL,"Error: could not open headers file.");
				if (stderrlogging > 0) fprintf(stderr,"Error: could not open headers file.");
				exit(_EXITERR_MIMEUNPACK_CANNOT_OPEN_HEADERS_FILE);
				} /* if headers file failed to open */
			} /* if we need to open the headers file */

		
		_debug("MIME_unpack: preparing to start reading from pack...");

			/* for every possible email in the MAILPACK/Spool */
		while(info.inputok != NULL){
			
				/* Step 1. Locate the header breaker ---- This is the first portion of the Email, where we have things like the sender/receiver/returnpath  */
			info.boundarylen = 0;
			info.lastlinewasboundary = 0;
			info.lastlinewasblank = 0;
			info.lastlinewasfrom = 0;
			info.lastencoding = _ENC_UNKNOWN;
			info.filename[0] = '\0';
			info.boundary[0] = '\0';
			from_found = 0;
			
			_debug("MIME_unpack: done initialising vars within WHILE (step1)");
			
				/* We keep reading lines in until we find a blankline (char[0] == \n or \r) when we find the blank line, we will flag "newline" */		
			while (info.inputok != NULL){

				if (!f) _debug("MIME_unpack: Error - cannot read from file.");
				
					/* get the input line */
				info.inputok = fgets(aline,512,f);
				_current_line++;
				_debug("MIME_unpack: read in line, %d chars",strlen(aline));
				_debug("Read:%d:%s",_current_line,aline);

					/* if we managed to get a line okay */
				if (info.inputok != NULL) {				

						/* Make a copy of it and convert to lowercase */
					strncpy(lline,aline,512); strlower(lline);

						/* if we have dumpheaders set, then output to headers file */
					if (dumpheaders >0) fprintf(headers,"%s",aline);


						/* if we get a blank line, then we have reached the end of our headers, time to deal with the email... */
					if ((aline[1] == 10)||(aline[1] == 12)||(aline[0] == 10)||(aline[0] == 12)) {
						if ((info.lastencoding != _ENC_UNKNOWN)||(info.boundarylen != 0)||(from_found > 0)){
							newline++;
							_debug("MIME_unpack: New line found, end of headers.");
							break;
							}
						}
					else  /* LINE TESTING */
					if ((bp=strstr(lline,"boundary="))){
						bp3 = lline; /* setup our starter markers */
						bp2 = aline;
						while (bp3 != bp){  /* until we've matched bp3 and bp... */
							bp2++; /* move bp2 by one char */
							bp3++; /* move bp3 by one char */
							}
						bp = bp2 +strlen("boundary="); /* we now have the uncase-densensitised bp */

						_debug("MIME_unpack: Boundary = %s",bp);

						if ( (*bp) == '\"' ) bp++; 		/* Deal with a quotation if required */
						bp2 = strtok(bp,"\n\r\"");		/* Extract by using tokenisation */

						if (bp2){
							sprintf(info.boundary,"%s",bp2);
							MIME_BS_push(info.boundary);
						} else {								/* Else mark off as an error */
							if (syslogging > 0) syslog(_SL,"Error: could not retrieve boundary from %s",bp);
							if (stderrlogging > 0) fprintf(stderr,"Error: couldn't retreive boundary from %s",bp);
							exit(_EXITERR_UNDEFINED_BOUNDARY);
							}
							/* get the boundary length */												
						info.boundarylen = strlen(info.boundary);
						} 
					else  /* LINE TESTNG */
					if (strstr(lline,"content-transfer-encoding: base64")) {
						encoding = 1;
						}
					else /* LINE TESTING */
					if (strstr(lline,": text/")){
						encoding = 2;
						}
					else  /* LINE TESTING */
					if (strstr(lline,"content-transfer-encoding: quoted-printable")){
						encoding = 3;
						}
					else  /* LINE TESTING */
					if (strstr(lline,"content-type: message/rfc822")){
						encoding = 4;
						}				
						/* because sometimes we want to know that we've parsed far enough into the headers, we look out to see if we've got any From: type lines */
					else /* LINE TESTING */
					if (strncmp(lline,"from",4)==0){
						from_found++;
						}						
					else /* else anothing type of header we -might- find is an actual file name specifier, rare, but it does some times happen */
					if ((bp=strstr(lline,"name="))){
						bp3 = lline; /* setup our starter markers */
						bp2 = aline;
						while (bp3 != bp){  /* until we've matched bp3 and bp... */
							bp2++; /* move bp2 by one char */
							bp3++; /* move bp3 by one char */
							}
						fnp = bp2 +strlen("name="); /* we now have the uncase-densensitised bp */
						if ((*fnp) == '\"') fnp++;		/* if the first char is a quote, del the char */
						fnp2 = strtok(fnp,"\n\r\"");	/* tokenise to extract file name */
						sprintf(info.filename,"%s",fnp2);		/* Store as our MIME_filename */
					        quick_clean_filename(info.filename); 	/* cleanup garbage characters */
						_debug("MIME_unpack: Filename found in headers \"%s\"",info.filename);
						}
					} /* if inputOK */				
				} /* while (Step 1) */

		/* 
		 * Now that we have delt with the first portion of the email (headers)
		 * we can make a far better assesment of what we need to do next, whether
		 * we'll have to deal with a email-file or various attachments
		 */
		 
		_debug("MIME_unpack: Step 2");
		
		while ((info.inputok != NULL)&&(info.lastlinewasfrom == 0)) {


			/* if we have no filename, and no boundary, then create a fake filename */
			if ( (info.filename[0] == '\0')&&(info.boundarylen == 0) ){
				noname_file = 1;
				sprintf(info.filename,"%s%d",blankfileprefix,filecount);
				_debug("MIME_unpack: no filename and no boundary...use blank prefix, name=%s",info.filename);
				}

			/* if we STILL have no filename, but DO have a boundary, then we better find the attachment start! */					 
			if ((info.filename[0] == '\0')&&(info.boundarylen > 0)) {
				noname_file = 1;
				_debug("MIME_unpack: no filename, but BOUNDARY IS OK");
				if (info.lastlinewasboundary == 0){
					MIME_find_next_boundary(f, &info );
					if (info.lastlinewasfrom != 0) break;
					}
				MIME_collect_headers(f, &info);
				_debug("MIME_unpack: filename = %s",info.filename);
				}
				
			/* now that we MUST have found the boundary, we can reset the flag */
			info.lastlinewasboundary = 0;
			
			/* do an override here in the event of there being no MIME_boundary */
			if (info.boundarylen==0) {
				_debug("MIME_unpack: encoding decisions via nonboundary spec");
				if (encoding == 0) { info.lastencoding = _ENC_PLAINTEXT; }
				else
				if (encoding == 1) { info.lastencoding = _ENC_BASE64; }
				else
				if (encoding == 2) { info.lastencoding = _ENC_PLAINTEXT; }
				else 
				if (encoding == 3) { info.lastencoding = _ENC_QUOTED; }
				else 
				if (encoding == 4) { info.lastencoding = _ENC_EMBEDDED; }
				else
					info.lastencoding = _ENC_PLAINTEXT;
				}
				
			/* decode according to our header information */
			_debug("MIME_unpack: encoding decisions, current encode = %d, converted = %d", encoding, info.lastencoding);

			MIME_clean_MIME_filename(info.filename);	/* check out thefilename for ISO filenames */
			quick_clean_filename(info.filename); 	/* cleanup garbage characters */

			/* if we have an attachment which has a real name, then increment
			 * the attachment counter */			
			if (strstr(info.filename,blankfileprefix) != info.filename) {
				_attachment_count++;
				noname_file = 0;
				}
			
			if (info.lastencoding == _ENC_BASE64) { _debug("MIME_unpack: B64..go.go.go"); MIME_decode_64(f, unpackdir, &info); }
			else 
			if (info.lastencoding == _ENC_PLAINTEXT){ _debug("MIME_unpack: PLAINTEXT..go.go.go"); MIME_decode_text(f, unpackdir, &info); }
			else 
			if (info.lastencoding == _ENC_QUOTED){ _debug("MIME_unpack: QUOTED..go.go.go"); MIME_decode_text(f, unpackdir, &info); }
			else 
			if (info.lastencoding == _ENC_EMBEDDED){ 
				_debug("MIME_unpack: EMBEDDED..go.go.go");			
				MIME_decode_text(f, unpackdir, &info); 
				sprintf(tmp,"%s/%s",unpackdir,info.filename);
				MIME_unpack(unpackdir, tmp, current_recursion_level);
				unlink(tmp); /* delete the original text-output, as we have extracted what we need */
				noname_file = 0;
				}
			else {
			 _debug("MIME_unpack: DEFAULT_PLAINTEXT..go.go.go"); MIME_decode_text(f, unpackdir, &info); 	
			 }			
			

			/* If no nameless is set to something other than zero, we need to remove
			 * any file which has the blankfileprefix */
			if ((_no_nameless != 0)&&(noname_file == 1)){				
				if (unlink(info.filename)==0){
					if (verbosity) fprintf(stdout,"Removed %s\n",info.filename);
					}
				}


			_debug("MIME_unpack: DECODING for %s done.",info.filename);
			
			/* reset the filename, so that we dont confuse the header/encoding selection on our next iteration */
			info.filename[0]='\0';			

			/* Because BASE64 doesn't read in line at a time, we must then explicitly exit out if we have a BASE64, boundaryless email */
			if ((info.lastencoding == _ENC_BASE64)&&(info.boundarylen == 0)){
				info.inputok = NULL;
				}
				
			} /* while inputOK and more lines being read and NOT a FROM line found*/						
			} /* while inputOK */
			
		/* close the mailpack */
		if (f) fclose(f);
		} /* end of FROM: line break out */		

	/* if our headers file is open, and we're at our -top- level (or recursion) then we can safely close the headers file */	
	if ((headers)&&(current_recursion_level == 0)) fclose(headers);

	return status;
	}

	
	
/*--------------------------------------------------------------------
 * MIME_close
 * 
 * Closes the files used in MIME_unpack, such as headers etc */
int MIME_close( void ){
	if (headers){
		fclose(headers);
		}
		
	return 0;
	}
		
	
/*----------------------------------------------------------------------
 * MIME_insert_X_header
 * 
 * Inserts a specified X-header at the end of other X-headers.
 */
int MIME_insert_Xheader( char *fname, char *xheader){
	
	/* Tempfile tmpfile() fix contributed by David DeMaagd - 29/01/2001 */
	char line[1024], lline[1024];
	char tpn[1024];
	int xon = 0;
	int returnvalue = 0;
	FILE *fo;
	FILE *f;

	/* Get temp name */
	sprintf(tpn,"%s.",fname);
			
	fo = fopen(tpn,"w"); if (!fo){ 
		if (syslogging > 0) syslog(_SL,"Error: Cannot open temporary file for writing (%s) : %s",tpn, strerror(errno));
		if (stderrlogging > 0) fprintf(stderr,"Error: Cannot open temporary file for writing");
		returnvalue = 1;
		}
	f = fopen(fname,"r"); if (!f){ 
		if (syslogging > 0) syslog(_SL,"Error: Cannot open Mailpack file for reading (%s) : %s",fname, strerror(errno));
		if (stderrlogging > 0) fprintf(stderr,"Error: Cannot open Mailpack file for reading, %s",fname);
		returnvalue = 2;
		}

	if ( returnvalue == 0 ){
		while (fgets(line,1023,f)){
			_current_line++;
			strncpy(lline,line,1023);
			strlower(lline);
			
			if ((xon==0)&&(strstr(lline,"content-type:"))){
				fprintf(fo,"%s\n",xheader);
				xon=1;
				}	

			fprintf(fo,"%s",line);
			}
		
		/* close files */
		fclose(fo);
		fclose(f);
		
		/* remove the old file */
		if (remove(fname) == -1) {
			syslog(_SL,"Error: xInsertHeader - REMOVE, %s",strerror(errno));
			}
		else {
			/* and rename */
			if (rename(tpn,fname) == -1) {
				syslog(_SL,"Error: xInsertHeader - RENAME, %s",strerror(errno));
				} /*if rename*/
			} /* else */
		
		} /* if */
		
	return returnvalue;
	}
	

	
	
	
/*----------END OF MIME.c------------*/				
 
