#include <stdlib.h>
#include <assert.h>
#include "defines.h"
#include "writer.h"
#include "module.h"
#include "util.h"
#include "module_management.h"

int
parser_hash (const char *s)
{
  int h;

#ifndef NDEBUG
  write ("Hash generated for \"%s\" is ", s);
#endif

  for (h = 0; *s != '\0'; s++)
    h = (64 * h + *s) % 100;

#ifndef NDEBUG
  writeln ("%d", h);
#endif

  return h;
}


struct module_management_node
{
  int key;
  struct module module;
  struct module_management_node *l;
  struct module_management_node *r;
};

struct module_management_node *head;
struct module_management_node *z;

static void
_tree_init (void)
{
  z = (struct module_management_node *) malloc (sizeof *z);
  z->l = z;
  z->r = z;
  z->key = -1;
  head = (struct module_management_node *) malloc (sizeof *head);
  head->r = z;
  head->key = 0;
}

static void
_tree_insert (int v, struct module *m)
{
  struct module_management_node *p, *x;

  p = head;
  x = head->r;
  while (x != z)
    {
      p = x;
      x = (v < x->key) ? x->l : x->r;
    }

  x = (struct module_management_node *) malloc (sizeof *x);
  x->key = v;
  x->module = *m;
  x->l = z;
  x->r = z;
  if (v < p->key)
    p->l = x;
  else
    p->r = x;
}

static struct module_management_node *
_tree_search (int k)
{
  struct module_management_node *p;
  struct module_management_node *x;

  p = head;
  x = head->r;
  while (x != z)
    {
      p = x;
      if (x->key == k)
	return x;
      else
	x = (k < x->key) ? x->l : x->r;
    }
  return (struct module_management_node *) NULL;
}

static void
_tree_print_node (struct module_management_node *x)
{
  if (x != z)
    {
      _tree_print_node (x->l);
      module_management_module_info (&(x->module));
      _tree_print_node (x->r);
    }
}

static void
_tree_remove_node (int k)
{
  struct module_management_node *c;
  struct module_management_node *p;
  struct module_management_node *x;
  struct module_management_node *t;

  z->key = k;
  p = head;
  x = head->r;

  while (k != x->key)
    {
      p = x;
      x = (k < x->key) ? x->l : x->r;
    }
  t = x;
  if (t->r == z)
    x = x->l;
  else if (t->r->l == z)
    {
      x = x->r;
      x->l = t->l;
    }
  else
    {
      c = x->r;
      while (c->l->l != z)
	c = c->l;
      x = c->l;
      c->l = x->r;
      x->l = t->l;
      x->r = t->r;
    }
  free (t);
  if (k < p->key)
    p->l = x;
  else
    p->r = x;
}

static void
_tree_free_node (struct module_management_node *x)
{
  if (x != z)
    {
      _tree_free_node (x->l);
      _tree_free_node (x->r);
// TODO: debug
      _tree_remove_node (x->key);
    }
}

static void
_tree_free (void)
{
  _tree_free_node (head->r);
}

int
module_management_add_module (struct module *m)
{
  struct module_management_node *t;
  int key = parser_hash (m->name);

  if ((t = _tree_search (key)) == NULL)
    {
      _tree_insert (key, m);
      return 0;
    }

  writeln (HEADER_ERROR "the module (ID %d) wasn't loaded.", key);
  return 1;
}

int
module_management_remove_module (struct module *m)
{
  struct module_management_node *t;
  int key = parser_hash (m->name);

  if ((t = _tree_search (key)) != NULL)
    {
      _tree_remove_node (key);
      return 0;
    }

  return 1;
}

int
module_management_module_info (struct module *m)
{
  struct module_management_node *t;
  int key = parser_hash (m->name);

  if ((t = _tree_search (key)) != NULL)
    {
      writeln ("module (ID %d) is loaded.", t->key);
      writeln ("module info: ");
      writeln ("   name  : %s", t->module.name);
      writeln ("   brief : %s", t->module.description);
      return 0;
    }

  return 1;
}

void
module_management_remove_all_modules (void)
{
  _tree_free ();
}

void
module_management_print_modules (void)
{
  _tree_print_node (head->r);
}
void
module_management_init (void)
{
  _tree_init ();
}


#ifdef MODULE_MANAGEMENT_TEST
// to compile this test:
// gcc -DMODULE_MANAGEMENT_TEST -DNDEBUG -lspar module_management.c -o module_management_test

struct module m = {
  "The MODULE name", "a brief description of what the module provide", NULL
};

struct module m1 = {
  "Another MODULE",
  "a brief description of what the module provide bis", NULL
};


int
main (void)
{
  module_management_init ();

  module_management_add_module (&m1);
  module_management_add_module (&m);

  module_management_module_info (&m1);
  module_management_remove_module (&m1);
  module_management_module_info (&m1);
  module_management_print_modules ();

  module_management_remove_all_modules ();
  return 0;
}
#endif
