#include <string.h>
#include <stdlib.h>
#include "parser-conf.h"
#include "writer.h"
#include "defines.h"
#include "vartable.h"
#include "errors.h"
#include "math_consts.h"
#include "math_defs.h"
#include "consts.h"
#include "errors.h"

inline int
varncmp (const char *a, const char *b)
{
  return strncmp (a, b, IDENTIFIER_LENGHT);
}

struct var_entry
{
  identifier (name);		/* variable name        */
  message (comment);		/* comment              */
  double value;			/* value                */
  double old_value;		/* old_value            */

  struct var_entry *l;
  struct var_entry *r;
};

struct var_entry *head;
struct var_entry *z;
static unsigned int var_counter;

inline void
_edit_comment (struct var_entry *entry, const char comment[])
{
  strncpy (entry->comment, comment, MESSAGE_LENGHT);
}

inline void
_edit_name (struct var_entry *entry, const char name[])
{
  strncpy (entry->name, name, IDENTIFIER_LENGHT);
}

inline void
_edit_value (struct var_entry *entry, double value)
{
  entry->value = value;
}

inline void
_set_old_value (struct var_entry *entry)
{
  double tmp = entry->value;
  entry->value = entry->old_value;
  entry->old_value = tmp;
}

static struct var_entry *
_tree_search (const char name[])
{
  struct var_entry *p;
  struct var_entry *x;

  p = head;
  x = head->r;
  while (x != z)
    {
      p = x;
      if ((strcmp (x->name, name)) == 0)
	return x;
      else
	x = ((varncmp (name, x->name)) < 0) ? x->l : x->r;
    }
  return (struct var_entry *) NULL;
}

static void
_tree_insert (const char name[], double value)
{
  if (var_counter > MAX_VARIABLES - 1)
    parser_throw_error (ERROR_MAXVARS, name);
  else
    {
      struct var_entry *p;
      struct var_entry *x;

      p = head;
      x = head->r;
      while (x != z)
	{
	  p = x;
	  x = ((varncmp (name, x->name)) < 0) ? x->l : x->r;
	}

      x = (struct var_entry *) malloc (sizeof *x);
      _edit_name (x, name);
      _edit_comment (x, "no comment");
      _edit_value (x, value);
      x->l = z;
      x->r = z;

      if ((varncmp (name, p->name)) < 0)
	p->l = x;
      else
	p->r = x;
      var_counter++;
    }
}

static void
_tree_print_node (struct var_entry *x)
{
  if (x != z)
    {
      _tree_print_node (x->l);
      writeln ("%15s = %g\t{%-s}", x->name, x->value, x->comment);
      _tree_print_node (x->r);
    }
}

static void
_tree_print (void)
{
  _tree_print_node (head->r);
}

static void
_tree_delete_entry (const char *name)
{
  struct var_entry *c;
  struct var_entry *p;
  struct var_entry *x;
  struct var_entry *t;

  _edit_name (z, name);
  p = head;
  x = head->r;

  while ((varncmp (name, x->name)) != 0)
    {
      p = x;
      if ((varncmp (name, x->name)) < 0)
	x = x->l;
      else
	x = x->r;
    }
  t = x;
  if (t->r == z)
    x = x->l;
  else if (t->r->l == z)
    {
      x = x->r;
      x->l = t->l;
    }
  else
    {
      c = x->r;
      while (c->l->l != z)
	c = c->l;
      x = c->l;
      c->l = x->r;
      x->l = t->l;
      x->r = t->r;
    }

  free (t);

  if ((varncmp (name, p->name)) < 0)
    p->l = x;
  else
    x->r = x;
}

static void
_tree_free_node (struct var_entry *x)
{
  if (x != z)
    {
      _tree_free_node (x->l);
      _tree_free_node (x->r);
      _tree_delete_entry (x->name);
    }
}

static void
_tree_free (void)
{
  _tree_free_node (head->r);
}

/*
 * PARSER INTERFACE 
 */
int
parser_add_comment (const char name[], const char comment[])
{
  struct var_entry *tmp;

  assert (name);
  tmp = _tree_search (name);

  if (tmp != NULL)
    {
      _edit_comment (tmp, comment);
      return FUNCTION_SUCCESS;
    }
  else
    return FUNCTION_ERROR;
}

int
parser_init_vars_table (void)
{
  z = (struct var_entry *) malloc (sizeof *z);
  z->l = z;
  z->r = z;

  head = (struct var_entry *) malloc (sizeof *head);
  head->r = z;
  head->l = z;

  var_counter = 0;
  return FUNCTION_SUCCESS;
}


int
parser_clear_vars_table (void)
{
  _tree_free ();
  parser_init_vars_table ();
  return FUNCTION_SUCCESS;
}

int
parser_clear_var (const char name[])
{
  struct var_entry *tmp;

  assert (name);
  tmp = _tree_search (name);

  if (tmp != NULL)
    {
      _tree_delete_entry (name);
      var_counter--;
      return FUNCTION_SUCCESS;
    }
  else
    return FUNCTION_ERROR;
}


int
parser_set_old_value (const char name[])
{
  struct var_entry *tmp;

  assert (name);
  tmp = _tree_search (name);

  if (tmp != NULL)
    {
      _set_old_value (tmp);
      return FUNCTION_SUCCESS;
    }
  else
    return FUNCTION_ERROR;
}


int
parser_get_value (const char name[], double *r)
{
  struct var_entry *tmp;

  assert (name);
  tmp = _tree_search (name);

  if (tmp != NULL)
    {
      *r = tmp->value;
      return FUNCTION_SUCCESS;
    }
  else if (parser_search_const (name))
    {
      *r = parser_get_const_value (name);
      return FUNCTION_SUCCESS;
    }

  return FUNCTION_ERROR;
}

int
parser_set_value (const char name[], double value)
{
  assert (name);

  if (parser_search_const (name))
    {
      parser_throw_error (ERROR_CONST, name);
      return FUNCTION_ERROR;
    }

  _tree_insert (name, value);

  return FUNCTION_SUCCESS;
}

void
parser_print_vars_table (void)
{
  writeln ("Vars Table");
  _tree_print ();
}
