/*
 * comp.c - Evaluate computer move
 */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

#include "loa.h"


#define SCORE_WIN  1000


#define odd(n) (((n) & 1) == 1)


typedef struct compmove
{
	PIECE           *piece;
	POSITION         to;
	int              score; /* to sort generated moves */
	struct compmove *next;
}
COMPMOVE;


volatile int abort_computation;

static COLOR     comp_side,
                 opp_side;
static int       gamedepth;
static int      *score;
static COMPMOVE *cm_pool = NULL;


static int sort_int_desc( const void *i1, const void *i2 )
{
	return *(int*)i1 < *(int*)i2;
}

static int find_all_connected( COLOR color, int num[NUM_PIECES+1] )
{
	int    i,
	       mark = 0;
	PIECE *p;

	for (i = 0, p = &pieces[color][0]; i < NUM_PIECES; ++i, ++p)
		p->is_marked = 0;
	for (i = 0, p = &pieces[color][0]; i < NUM_PIECES; ++i, ++p)
		if (!p->is_beaten && !p->is_marked)
		{
			num[mark] = mark_piece( p, mark+1 );
			++mark;
		}
	qsort( num, mark, sizeof(int), sort_int_desc );

	return mark;
}

static int calculate_score()
/* Calculate the score of a board situation */
{
	int   mark[NUM_COLORS];
	int   num[NUM_COLORS][NUM_PIECES+1];

	mark[comp_side] = find_all_connected( comp_side, num[comp_side] );
	if (mark[comp_side] == 1)
		return SCORE_WIN;

	mark[opp_side] = find_all_connected( opp_side, num[opp_side] );
	if (mark[opp_side] == 1)
		return -SCORE_WIN;

	return 5*(num[comp_side][0] - num[opp_side][0]) - num[comp_side][1];
}


static COMPMOVE *get_compmove( void )
{
	COMPMOVE *cm;

	if (cm_pool == NULL)
	{
		if ((cm = malloc( sizeof(*cm) )) == NULL)
			fatal( NO_MEMORY );
	}
	else
	{
		cm_pool = (cm = cm_pool)->next;
	}

	return cm;
}


static void return_to_pool( COMPMOVE *cm_head, COMPMOVE *cm_tail )
{
	cm_tail->next = cm_pool;
	cm_pool = cm_head;
}


static COMPMOVE *generate_all_moves( int depth )
{
	int        i,
	           num_opp_pieces;
	PIECE     *p, *q;
	POSITION   to;
	DIRECTION  d, e;
	COMPMOVE  *cm_list, *cm, *cm_curr, *cm_prev;
	WORD       r, c;

	cm_list = NULL;
	for (i = 0, p = &pieces[side_to_move][0]; i < NUM_PIECES; ++i, ++p)
	{
		if (p->is_beaten)
			continue;

		for (d = 0; d < NUM_DIRECTIONS; ++d)
		{
			if (!calculate_target_square( p, d, &to ))
				continue;

			cm = get_compmove();
			cm->piece = p;
			cm->to    = to;
			cm->score = 0;
			cm->next  = NULL;

			/*
			   Static score:
			   -2 for each own piece in p's neibourhooghd,
			   +4 for each own piece in to's neighbourhood
			   +1 if beating move and 2 or more enemy pieces in to's neighbourhood
			   -1 if beating move and less than 2 enemy pieces in to's neighbourhood
			*/
			num_opp_pieces = 0;
			for (e = 0; e < NUM_DIRECTIONS; ++e)
			{
				if ((r = p->pos.row + row_off[e]) >= 0 && r <= 7 &&
				    (c = p->pos.col + col_off[e]) >= 0 && c <= 7 &&
				    (q = board[r][c]) != NULL &&
				    q->color == comp_side)
					cm->score -= 2;
				if ((r = cm->to.row + row_off[e]) >= 0 && r <= 7 &&
				    (c = cm->to.col + col_off[e]) >= 0 && c <= 7 &&
				    (q = board[r][c]) != NULL)
					if (q->color == comp_side)
						cm->score += 4;
					else
						++num_opp_pieces;
			}
			if (board[cm->to.row][cm->to.col] != NULL) /* i.e. beating move */
				if (num_opp_pieces >= 2)
					cm->score += 1;
				else
					cm->score -= 1;

			/* Sort cm into cm_list, descending by score */
			cm_curr = cm_list;
			cm_prev = NULL;
			while (cm_curr != NULL && cm->score < cm_curr->score)
				cm_curr = (cm_prev = cm_curr)->next;
			if (cm_prev == NULL)
			{
				cm->next = cm_list;
				cm_list  = cm;
			}
			else
			{
				cm->next      = cm_prev->next;
				cm_prev->next = cm;
			}
		}
	}

	return cm_list;
}


static int compute_alphabeta( int depth )
/* returns depth where alpha/beta cut is to take place, or depth+1 if no cut */
{
	int alpha = INT_MIN,
	    beta  = INT_MAX,
	    cut   = depth+1,
	    d;

	for (d = depth; d >= 0; --d)
		if (odd( d ))
		{
			if (score[d] <= alpha)
				cut = d;
			if (score[d] < beta)
				beta = score[d];
		}
		else
		{
			if (score[d] >= beta)
				cut = d;
			if (score[d] > alpha)
				alpha = score[d];
		}
	return cut;
}


static COMPMOVE *find_last( COMPMOVE *cm )
{
	while (cm->next != NULL)
		cm = cm->next;
	return cm;
}


static void internal_compute_move(
	int       depth,
	POSITION *from,
	POSITION *to,
	int      *alphabeta
)
{
	COMPMOVE *cm_head,
	         *cm_curr,
	         *cm_tail;
	COLOR     win;

	if (depth >= gamedepth)
	{
		score[depth] = calculate_score();
		*alphabeta = compute_alphabeta( depth );
		return;
	}

	cm_curr =
	cm_head = generate_all_moves( depth );
	cm_tail = NULL;

	if (cm_head == NULL)
	{
		score[depth] = odd( depth ) ? SCORE_WIN : -SCORE_WIN;
		*alphabeta = compute_alphabeta( depth );
		return;
	}

	/* In case the computation gets aborted before the first back-scoring */
	if (from != NULL)
	{
		*from = cm_curr->piece->pos;
		*to   = cm_curr->to;
	}

	score[depth] = odd( depth ) ? INT_MAX : INT_MIN;

	for ( ; cm_curr != NULL; cm_curr = (cm_tail = cm_curr)->next)
	{
		move_piece( cm_curr->piece, cm_curr->to );
		win = winning_side();
		SWITCH_SIDES();

		if (win == NONE)
		{
			internal_compute_move( depth+1, NULL, NULL, alphabeta );
		}
		else
		{
			score[depth] = (win == comp_side) ? SCORE_WIN : -SCORE_WIN;
			if (win == comp_side && depth == 0)
			{
				*from = cm_curr->piece->pos;
				*to   = cm_curr->to;
			}
			*alphabeta = compute_alphabeta( depth );
		}

		take_back();
		SWITCH_SIDES();

		if (*alphabeta < depth || abort_computation)
		{
			cm_tail = find_last( cm_curr );
			break;
		}

		if (win == NONE)
		{
			/* Back-scoring */
			if (odd( depth ))
			{
				/* opponent's move - minimize scores */
				if (score[depth+1] < score[depth])
					score[depth] = score[depth+1];
			}
			else
			{
				/* computer's move - maximize scores */
				if (score[depth+1] > score[depth])
				{
					score[depth] = score[depth+1];
					if (depth == 0)
					{
						*from = cm_curr->piece->pos;
						*to   = cm_curr->to;
					}
				}
			}
		}
		else
		{
			if ((!odd( depth ) && win == comp_side) ||
			    ( odd( depth ) && win == opp_side))
			{
				cm_tail = find_last( cm_curr );
				break;
			}
		}
	}

	return_to_pool( cm_head, cm_tail );
}


void compute_move( int level, int show, POSITION *from, POSITION *to )
{
	int alphabeta;

	message( 0, "Computing" );

	gamedepth = 2*level+1;
	if ((score = (int*)calloc( gamedepth+1, sizeof(int) )) == NULL)
		fatal( NO_MEMORY );

	comp_side = side_to_move;
	opp_side  = OPPOSITE_SIDE(comp_side);

	hide_computing = !show;
	is_analysis    = TRUE;

	abort_computation = FALSE;
	enable_unsolicited_input();

	internal_compute_move( 0, from, to, &alphabeta );

	disable_unsolicited_input();

	is_analysis    = FALSE;
	hide_computing = FALSE;

	free( score );

	clear_message();
}
