/*
** game.c - All around the gameplay
*/

#include <stdio.h>
#include <string.h>

#include "loa.h"


static WORD num_pieces_in_row[8],
            num_pieces_in_col[8],
            num_pieces_in_diag1[15],  /* DOWN_LEFT  -> UP_RIGHT */
            num_pieces_in_diag2[15];  /* DOWN_RIGHT -> UP_LEFT  */


#define min(a,b) (((a) < (b)) ? (a) : (b))
#define max(a,b) (((a) > (b)) ? (a) : (b))
#define sgn(a) (((a) < 0) ? -1 : ((a) > 0) ? 1 : 0)


void insert_piece( PIECE *piece )
{
	board[piece->pos.row][piece->pos.col] = piece;

	++num_pieces_in_row[piece->pos.row];
	++num_pieces_in_col[piece->pos.col];
	++num_pieces_in_diag1[piece->pos.row + piece->pos.col];
	++num_pieces_in_diag2[7 + piece->pos.row - piece->pos.col];

	show_piece( piece->color, piece->pos );
}


void remove_piece( POSITION pos )
{
	board[pos.row][pos.col] = NULL;

	--num_pieces_in_row[pos.row];
	--num_pieces_in_col[pos.col];
	--num_pieces_in_diag1[pos.row + pos.col];
	--num_pieces_in_diag2[7 + pos.row - pos.col];

	show_piece( NONE, pos );
}


int is_valid_move( POSITION from, POSITION to, COLOR color )
{
	WORD rowoff, coloff, tomove, moved;
	POSITION pos;
	int on_target;

	if (POS_EQ( from, to ))
	{
		message( 1, "No move" );
		return FALSE;
	}

	if (board[to.row][to.col] != NULL &&
	    board[to.row][to.col]->color == color)
	{
		message( 1, "You can't beat your own pieces" );
		return FALSE;
	}

	if (from.row == to.row)
		tomove = num_pieces_in_row[from.row];
	else
	if (from.col == to.col)
		tomove = num_pieces_in_col[from.col];
	else
	if (from.row + from.col == to.row + to.col)
		tomove = num_pieces_in_diag1[from.row + from.col];
	else
	if (from.row - from.col == to.row - to.col)
		tomove = num_pieces_in_diag2[7 + from.row - from.col];
	else
	{
		message( 1, "Invalid move" );
		return FALSE;
	}

	rowoff = sgn(to.row - from.row);
	coloff = sgn(to.col - from.col);
	pos = from;
	moved = 0;
	do
	{
		pos.row += rowoff;
		pos.col += coloff;
		on_target = POS_EQ( pos, to );
		if (board[pos.row][pos.col] == NULL)
			++moved;
		else
		if (board[pos.row][pos.col]->color != color)
			if (on_target)
				++moved;
			else
			{
				message( 1, "Can't move over enemy piece at %c%c", POS_TO_2CHARS( pos ) );
				return FALSE;
			}
	}
	while (!on_target);

	if (moved != tomove)
	{
		message( 1, "Wrong number of moves: %d, must be %d", moved, tomove );
		return FALSE;
	}

	return TRUE;
}


int calculate_target_square( PIECE *piece, DIRECTION dir, POSITION *to )
{
	WORD tomove, moved;
	POSITION pos;
	extern const WORD row_off[], col_off[];

	switch (dir)
	{
		case DIR_LEFT :
		case DIR_RIGHT :
			tomove = num_pieces_in_row[piece->pos.row];
			break;
		case DIR_UP :
		case DIR_DOWN :
			tomove = num_pieces_in_col[piece->pos.col];
			break;
		case DIR_UP_RIGHT :
		case DIR_DOWN_LEFT :
			tomove = num_pieces_in_diag1[piece->pos.row + piece->pos.col];
			break;
		case DIR_UP_LEFT :
		case DIR_DOWN_RIGHT :
			tomove = num_pieces_in_diag2[7 + piece->pos.row - piece->pos.col];
			break;
		default :
			fatal( INVALID_DIRECTION );
	}

	pos = piece->pos;
	moved = 0;
	do
	{
		pos.row += row_off[dir];
		pos.col += col_off[dir];
		if (pos.row < 0 || pos.row > 7 ||
		    pos.col < 0 || pos.col > 7)
			return FALSE;

		if (board[pos.row][pos.col] == NULL)
			++moved;
		else
		if (board[pos.row][pos.col]->color != piece->color)
			if (moved == tomove - 1)
				++moved;
			else
				return FALSE;
	}
	while (moved != tomove);

	*to = pos;
	return TRUE;
}


int mark_piece( PIECE *piece, int mark )
{
	DIRECTION  d;
	WORD       r, c;
	PIECE     *p;
	COLOR      color = piece->color;
	int        sum = 1;

	piece->is_marked = mark;
	for (d = 0; d < NUM_DIRECTIONS; ++d)
		if ((r = piece->pos.row + row_off[d]) >= 0 && r <= 7 &&
		    (c = piece->pos.col + col_off[d]) >= 0 && c <= 7 &&
		    (p = board[r][c]) != NULL && p->color == color && !p->is_marked)
			sum += mark_piece( p, mark );
	return sum;
}


static int is_connected( COLOR color )
{
	int i;
	PIECE *p;

	/* Unmark all pieces */
	for (i = 0, p = &pieces[color][0]; i < NUM_PIECES; ++i, ++p)
		p->is_marked = 0;

	/* Find first piece of given color */
	for (i = 0, p = &pieces[color][0]; i < NUM_PIECES; ++i, ++p)
		if (!p->is_beaten)
			break;
	if (i == NUM_PIECES)
		fatal( NO_PIECES_LEFT );

	mark_piece( p, 1 );

	/* Check for unmarked piece */
	for ( ++p; i < NUM_PIECES; ++i, ++p)
		if (!p->is_beaten && !p->is_marked)
			return FALSE;

	return TRUE;
}


COLOR winning_side( void )
{
	int white_wins = is_connected( WHITE ),
	    black_wins = is_connected( BLACK );

	if (white_wins && black_wins)
		return side_to_move;
	else
	if (white_wins)
		return WHITE;
	else
	if (black_wins)
		return BLACK;

	return NONE;
}

