/*
** loa.h - Central include file
*/

/*
** #include's
*/

#include <limits.h>


/*
** #define's
*/

#define MAX_INPUT_LEN 50

#define POS_TO_2CHARS(pos)  'A'+(pos).col, '8'-(pos).row

#define POS_EQ(p1,p2) ((p1).row == (p2).row && (p1).col == (p2).col)
#define POS_NE(p1,p2) ((p1).row != (p2).row || (p1).col != (p2).col)

#define OPPOSITE_SIDE(side) (1 - side)
#define SWITCH_SIDES()      side_to_move = OPPOSITE_SIDE(side_to_move)

#define NUM_COLORS 2
#define NUM_PIECES 12 /* per side */
#define NUM_DIRECTIONS 8


#ifdef LOA_MAIN
#  define EXTERN
#else
#  define EXTERN extern
#endif


/*
** typedef's
*/

typedef unsigned char      UBYTE;
typedef short int          WORD;
typedef unsigned short int UWORD;
typedef int                LONG;
typedef unsigned int       ULONG;

typedef enum
{
	CMD_ERROR = 1,
	CMD_MOVE,
	CMD_TAKEBACK,
	CMD_SAVEGAME,
	CMD_REPLAY,
	CMD_REDRAW,
	CMD_NEWGAME,
	CMD_SETCOMP,
	CMD_EXIT,
	CMD_HELP,
	CMD_NOP
}
CMD;

typedef enum
{
	NO_MEMORY = 1,
	MOVING_BEATEN_PIECE,
	NO_PIECES_LEFT,
	INVALID_DIRECTION,
	NO_POSSIBLE_MOVE
}
ERROR;

typedef enum
{
	WHITE = 0,
	BLACK,
	NONE,
	BOTH
}
COLOR;

typedef enum
{
	DIR_UP = 0,
	DIR_DOWN,
	DIR_LEFT,
	DIR_RIGHT,
	DIR_UP_LEFT,
	DIR_DOWN_RIGHT,
	DIR_UP_RIGHT,
	DIR_DOWN_LEFT,
	DIR_NONE
}
DIRECTION;

typedef struct position
{
	WORD row, col;
}
POSITION;

typedef struct piece
{
	COLOR color;
	POSITION pos;
	int is_beaten;
	int is_marked; /* for connection test */
}
PIECE;

typedef PIECE *BOARD[8][8];

typedef struct
{
	CMD cmd;
	union
	{
		/* CMD_MOVE */
		struct
		{
			POSITION from, to;
		} move;

		/* CMD_SAVEGAME */
		struct
		{
			char filename[PATH_MAX];
		} savegame;

		/* CMD_REPLAY */
		struct
		{
			char filename[PATH_MAX];
		} replay;

		/* CMD_SETCOMP */
		struct
		{
			COLOR side;
			int   level;
			int   show;
		} setcomp;

		/* CMD_HELP */
		struct
		{
			char topic[MAX_INPUT_LEN - sizeof("LOA ")];
		} help;
	} param;
}
COMMAND;

typedef struct move
{
	COLOR color;
	POSITION from, to;
	PIECE *beats_piece;
	struct move *prev, *next;
}
MOVE;


/*
** Global data
*/

extern const WORD row_off[NUM_DIRECTIONS],
                  col_off[NUM_DIRECTIONS];

EXTERN volatile int get_user_input;

EXTERN PIECE  pieces[NUM_COLORS][NUM_PIECES];
EXTERN BOARD  board;
EXTERN COLOR  side_to_move;
EXTERN int    hide_computing,
              is_analysis;
EXTERN char  *program_name;


/*
** Prototypes
*/

/* From main.c */
extern void fatal( ERROR error );
extern void move_piece( PIECE *piece, POSITION to );
extern void take_back( void );

/* From game.c */
extern void insert_piece( PIECE *piece );
extern void remove_piece( POSITION pos );
extern int is_valid_move( POSITION from, POSITION to, COLOR color );
extern int calculate_target_square( PIECE *piece, DIRECTION dir, POSITION *to );
extern int mark_piece( PIECE *piece, int mark );
extern COLOR winning_side( void );

/* From comp.c */
extern void compute_move( int level, int show, POSITION *from, POSITION *to );

/* From cmd.c */
extern void decode_command( char *cmdstr, COMMAND *command );

/* From sio_*.c */
extern void init_io( void );
extern void init_game_io( void );
extern void close_io( void );
extern void welcome( void );
extern void show_board( void );
extern void read_command( char cmdstr[] );
extern void show_piece( COLOR color, POSITION pos );
extern void show_move( int moveno, MOVE *mv );
extern void unshow_move( int moveno, MOVE *mv );
extern void message( int ring_bell, char *format, ... );
extern void clear_message( void );
extern void help( char *topic );
extern void redraw( void );
extern void enable_unsolicited_input( void );
extern void disable_unsolicited_input( void );

