/*
** loa.c - Main program
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define LOA_MAIN
#include "loa.h"


const WORD row_off[NUM_DIRECTIONS] = { -1, 1,  0, 0, -1, 1, -1,  1 };
const WORD col_off[NUM_DIRECTIONS] = {  0, 0, -1, 1, -1, 1,  1, -1 };

static int    moveno = 0;
static MOVE  *moves = NULL,
             *last_move,
             *move_freelist = NULL;
static COLOR  win = NONE;
static COLOR  comp_side  = NONE;
static int    comp_level = 2;
static int    comp_show  = FALSE;


void fatal( ERROR error )
{
	fprintf( stderr, "Something is screwed up: " );
	switch (error)
	{
		case NO_MEMORY :
			fprintf( stderr, "Can't get memory\n" );
			break;
		case MOVING_BEATEN_PIECE :
			fprintf( stderr, "Moving a beaten piece\n" );
			break;
		case NO_PIECES_LEFT :
			fprintf( stderr, "No pieces left?!\n" );
		case INVALID_DIRECTION :
			fprintf( stderr, "Invalid direction" );
			break;
		default :
			fprintf( stderr, "Unknown error %d\n", error );
	}
	exit( EXIT_FAILURE );
}


static void init_game( void )
{
	int i, j;
	PIECE *pw = &pieces[WHITE][0],
	      *pb = &pieces[BLACK][0];

	for (i = 1; i <= 6; ++i)
		for (j = 0; j <= 7; j += 7)
		{
			pw->color = WHITE;
			pw->pos.row = j;
			pw->pos.col = i;
			pw->is_beaten = FALSE;
			insert_piece( pw );
			++pw;

			pb->color = BLACK;
			pb->pos.row = i;
			pb->pos.col = j;
			pb->is_beaten = FALSE;
			insert_piece( pb );
			++pb;
		}

	move_freelist = moves;
	moves     =
	last_move = NULL;

	moveno = 0;
}


static MOVE *get_move( void )
{
	MOVE *move;

	if (move_freelist == NULL)
	{
		if ((move = malloc( sizeof(*move) )) == NULL)
			fatal( NO_MEMORY );
	}
	else
	{
		move          = move_freelist;
		move_freelist = move_freelist->next;
	}
	return move;
}


static void put_move( MOVE *move )
{
	move->next    = move_freelist;
	move_freelist = move;
}


void move_piece( PIECE *piece, POSITION to )
{
	MOVE  *move;
	PIECE *pp;

	move = get_move();
	move->color = piece->color;
	move->from  = piece->pos;
	move->to    = to;
	if ((pp = board[to.row][to.col]) != NULL)
	{
		move->beats_piece = pp;
		pp->is_beaten = TRUE;
		remove_piece( pp->pos );
	}
	else
	{
		move->beats_piece = NULL;
	}
	move->next = NULL;
	if ((move->prev = last_move) == NULL)
		last_move = moves = move;
	else
		last_move = last_move->next = move;

	show_move( ++moveno, move );

	remove_piece( piece->pos );
	piece->pos = to;
	insert_piece( piece );
}


void execute_move( POSITION from, POSITION to )
{
	PIECE *p;

	if ((p = board[from.row][from.col]) == NULL)
	{
		message( 1, "No piece at %c%c", POS_TO_2CHARS( from ) );
		return;
	}

	if (p->color != side_to_move)
	{
		message( 1, "It's %s's turn.", (side_to_move == WHITE) ? "White" : "Black" );
		return;
	}

	if (p->is_beaten)
		fatal( MOVING_BEATEN_PIECE );

	if (!is_valid_move( from, to, p->color))
		return;

	move_piece( p, to );
}


void take_back( void )
{
	PIECE *piece;

	if (last_move == NULL)
		return;

	piece = board[last_move->to.row][last_move->to.col];
	remove_piece( piece->pos );
	piece->pos = last_move->from;
	insert_piece( piece );

	if ((piece = last_move->beats_piece) != NULL)
	{
		piece->is_beaten = FALSE;
		insert_piece( piece );
	}

	unshow_move( moveno--, last_move );

	if (last_move->prev == NULL)
	{
		put_move( last_move );
		last_move = moves = NULL;
	}
	else
	{
		last_move = last_move->prev;
		put_move( last_move->next );
		last_move->next = NULL;
	}
}


static void save_game( char *filename )
{
	FILE *file;
	MOVE *move;

	if ((file = fopen( filename, "w" )) == 0)
	{
		message( 1, "Error opening %s", filename );
		return;
	}

	for (move = moves; move != NULL; move = move->next)
	{
		fprintf( file, "%c%c%c%c%c\n",
		  POS_TO_2CHARS( move->from ),
		  (move->beats_piece != NULL) ? ':' : '-',
		  POS_TO_2CHARS( move->to )
		);
	}

	fclose( file );

	message( 0, "Game saved to %s", filename );
}


static void replay_game( char *filename )
{
	FILE    *file;
	char     cmdstr[MAX_INPUT_LEN];
	COMMAND  command;
	extern void interpret_command( COMMAND *command );

	if ((file = fopen( filename, "r" )) == 0)
	{
		message( 1, "Error opening %s", filename );
		return;
	}

	while (fgets( cmdstr, MAX_INPUT_LEN, file ) != NULL)
	{
		decode_command( cmdstr, &command );
		interpret_command( &command );
	}

	fclose( file );

	message( 0, "Game replayed from %s", filename );
}


void interpret_command( COMMAND *command )
{
	switch (command->cmd)
	{
		case CMD_MOVE :
			if (win != NONE)
			{
				message( 1, "Game over" );
				break;
			}
			execute_move( command->param.move.from, command->param.move.to );
			switch (win = winning_side())
			{
				case WHITE :
					message( 3, "White wins!" ); break;
				case BLACK :
					message( 3, "Black wins!" ); break;
				case NONE :
					;
			}
			SWITCH_SIDES();
			break;

		case CMD_TAKEBACK :
			if (moveno == 0)
			{
				message( 1, "Nothing to take back" );
			}
			else
			{
				win = NONE;
				take_back();
				SWITCH_SIDES();
			}
			break;

		case CMD_SAVEGAME :
			save_game( command->param.savegame.filename );
			break;

		case CMD_REPLAY :
			/* new game */
			win          = NONE;
			side_to_move = WHITE;
			show_board();
			init_game();

			replay_game( command->param.replay.filename );
			break;

		case CMD_REDRAW :
			redraw();
			break;

		case CMD_NEWGAME :
			win          = NONE;
			side_to_move = WHITE;
			show_board();
			init_game();
			break;

		case CMD_SETCOMP :
			if (command->param.setcomp.side != (COLOR)-1)
				comp_side  = command->param.setcomp.side;
			if (command->param.setcomp.level != -1)
				comp_level = command->param.setcomp.level;
			if (command->param.setcomp.show != -1)
				comp_show  = command->param.setcomp.show;
			break;

		case CMD_HELP :
			help( command->param.help.topic );
			break;

		case CMD_ERROR :
			break;

		case CMD_NOP :
			break;

		default :
			break;
	}
}


int main( int argc, char *argv[] )
{
	COMMAND command = { CMD_NEWGAME };
	char    cmdstr[MAX_INPUT_LEN];

	program_name   = argv[0];
	hide_computing = FALSE;
	is_analysis    = FALSE;
	get_user_input = FALSE;

	init_io();
	welcome();
	init_game_io();

	while (command.cmd != CMD_EXIT)
	{
		if (command.cmd == CMD_ERROR)
			message( 1, "Error processing '%s'", cmdstr );
		else
			interpret_command( &command );

		if ((comp_side == BOTH || comp_side == side_to_move) && !get_user_input)
		{
			command.cmd = CMD_MOVE;
			compute_move( comp_level, comp_show, &command.param.move.from, &command.param.move.to );
		}
		else
		{
			read_command( cmdstr );
			decode_command( cmdstr, &command );
			get_user_input = FALSE;
		}
	}

	close_io();
	return EXIT_SUCCESS;
}
