/*
** io_smg.c - Screen I/O, using SMG routines
*/

#include stdio
#include stdlib
#include string
#include ctype
#include stdarg

#define __NEW_STARLET 1
#include descrip
#include ssdef
#include stsdef
#include smgdef
#include smgmsg
#include rmsdef
#include fabdef
#include namdef
#include starlet
#include smg$routines

#include "loa.h"
#include "version.h"


#ifdef __DECC
#  pragma message disable (NEEDCONSTEXT,ADDRCONSTEXT)
#endif


#define HELPLIB "LOA_HELP.HLB;"

#define WHITE_CHAR "X"
#define BLACK_CHAR "O"
#define EMPTY_CHAR "."

#define DISPLAY_WIDTH 80
#define DISPLAY_HEIGHT 24

#define BOARD_LU_ROW 6
#define BOARD_LU_COL 8

#define ROW_STRETCH 1
#define COL_STRETCH 2

#define ROW2SCREEN(r) (BOARD_LU_ROW + ROW_STRETCH * (r))
#define COL2SCREEN(c) (BOARD_LU_COL + COL_STRETCH * (c))

#define ROW2BOARD(r) (1 + ROW_STRETCH * (r))
#define COL2BOARD(c) (1 + COL_STRETCH * (c))

#define MAX_MSG_LENGTH (DISPLAY_WIDTH-1)

#define MAX_MOVES_ROW   (DISPLAY_HEIGHT-6)
#define MOVES_WIDTH    (4+2+5+2+5+1)  /* "nnn.  a1-a1  a1-a1 " */

#define COMMAND_PROMPT "LOA> "


#define chkvmssts(sts)                                     \
  {                                                        \
    ULONG local_sts = (sts);                               \
    if (! $VMS_STATUS_SUCCESS(local_sts))                  \
    {                                                      \
      fprintf( stderr, "Fatal VMS error at %s, line %d\n", \
               __FILE__, __LINE__ );                       \
      exit( local_sts );                                   \
    }                                                      \
  }


/* For smg$enable_unsolicited_input() with __NEW_STARLET */
typedef int (*AST_ROUTINE)(void);


static ULONG pasteboardID,
             keyboardID,
             screenID,
             boardID,
             movesID,
             cmdID,
             msgID;
static WORD  moves_row_off = 0;


void create_display(
	ULONG *displayID_ptr,
	LONG rows,
	LONG cols,
	ULONG display_attr,
	ULONG video_attr
)
{
	chkvmssts( smg$create_virtual_display(
	             &rows,
	             &cols,
	             displayID_ptr,
	             &display_attr,
	             &video_attr,
	             NULL  /* character set */
	         ));
}

void paste_display(
	ULONG displayID,
	LONG pasterow,
	LONG pastecol,
	ULONG top_displayID
)
{
	chkvmssts( smg$paste_virtual_display(
	            &displayID,
	            &pasteboardID,
	            &pasterow,
	            &pastecol,
	            NULL /* &top_displayID */
	         ));
}


void init_io( void )
{
	chkvmssts( smg$create_pasteboard( &pasteboardID ) );

	chkvmssts( smg$create_virtual_keyboard(
	             &keyboardID,
	             NULL, /* input device =SYS$INPUT: */
	             NULL, /* default filespec */
	             NULL, /* resultant filespec */
	             NULL  /* recall size =20 */
	         ));

	create_display(
	  &screenID,
	  DISPLAY_HEIGHT, DISPLAY_WIDTH,
	  SMG$M_BORDER,
	  0
	);
	paste_display( screenID, 1, 1, 0 );
}


static void paste_game_displays( void )
{
	paste_display(
	  boardID,
	  BOARD_LU_ROW, BOARD_LU_COL,
	  screenID
	);

	paste_display(
	  movesID,
	  (DISPLAY_HEIGHT - MAX_MOVES_ROW)/2, DISPLAY_WIDTH - MOVES_WIDTH - 4,
	  screenID
	);

	paste_display(
	  cmdID,
	  DISPLAY_HEIGHT -1, 1,
	  screenID
	);

	paste_display(
	  msgID,
	  DISPLAY_HEIGHT, 1,
	  screenID
	);
}


void init_game_io( void )
{
	create_display(
	  &boardID,
	  8, 2*8-1,
	  SMG$M_BLOCK_BORDER,
	  SMG$M_REVERSE
	);

	create_display(
	  &movesID,
	  MAX_MOVES_ROW, MOVES_WIDTH,
	  SMG$M_BORDER,
	  0
	);

	create_display(
	  &cmdID,
	  1, sizeof(COMMAND_PROMPT)-1 + MAX_INPUT_LEN,
	  0,
	  0
	);

	create_display(
	  &msgID,
	  1, MAX_MSG_LENGTH,
	  0,
	  0
	);

	paste_game_displays();
}


void close_io( void )
{
	chkvmssts( smg$delete_virtual_display( &msgID ) );
	chkvmssts( smg$delete_virtual_display( &cmdID ) );
	chkvmssts( smg$delete_virtual_display( &movesID ) );
	chkvmssts( smg$delete_virtual_display( &boardID ) );
	chkvmssts( smg$delete_virtual_display( &screenID ) );

	/* Closing everything else is taken care of by the SMG exit handler */
}


static void put_string( ULONG displayID, LONG row, LONG col, char *str, ULONG rendition )
{
	struct dsc$descriptor_s str_dsc = { strlen(str), DSC$K_DTYPE_T, DSC$K_CLASS_S, str };

	chkvmssts( smg$put_chars(
	             &displayID,
                     &str_dsc,
	             &row,
	             &col,
	             NULL, /* flags */
	             &rendition,
	             NULL, /* rendition complement */
	             NULL  /* character set */
	         ));
}


void message( int rings, char *format, ... )
{
	va_list ap;
	char msg[MAX_MSG_LENGTH];

	/* possible buffer overflow! */
	va_start( ap, format );
	vsprintf( msg, format, ap );
	put_string( msgID, 1, 1, msg, SMG$M_REVERSE );

	if (rings != 0)
		chkvmssts( smg$ring_bell( &msgID, &rings ) );
}


void clear_message( void )
{
	LONG row = 1,
	     col = 1,
	     num = DISPLAY_WIDTH-1;
	chkvmssts( smg$erase_chars( &msgID, &num, &row, &col ) );
}


static UWORD read_key( LONG timeout )
{
	UWORD terminator;
	ULONG sts;

	sts = smg$read_keystroke(
	             &keyboardID,
	             &terminator,
	             NULL, /* prompt string */
	             (timeout == 0) ? NULL : &timeout,
	             NULL, /* screen ID */
	             NULL, /* rendition set */
	             NULL  /* rendition complement */
	      );
	if (sts == SS$_TIMEOUT)
		return SMG$K_TRM_TIMEOUT;
	if (sts == SMG$_EOF)
		return SMG$K_TRM_CTRLZ;
	chkvmssts( sts );
	return terminator;
}


static void read_command_line( char cmdstr[] )
{
	$DESCRIPTOR( prompt_dsc, COMMAND_PROMPT );
	$DESCRIPTOR( cmd_dsc, cmdstr );
	LONG row, col, num;
	UWORD retlen;
	UWORD terminator;
	ULONG sts;

	/* Clear command line */
	row = 1; col = 1;
	num = sizeof(COMMAND_PROMPT)-1 + MAX_INPUT_LEN;
	chkvmssts( smg$erase_chars( &cmdID, &num, &row, &col ) );

	cmd_dsc.dsc$w_length = MAX_INPUT_LEN - 1;

	sts = smg$read_composed_line(
	        &keyboardID,
	        NULL, /* key table id */
	        &cmd_dsc,
	        &prompt_dsc,
	        &retlen,
	        &cmdID,
	        NULL, /* flags */
	        NULL, /* initial string */
	        NULL, /* timeout */
	        NULL, /* rendition set */
	        NULL, /* rendition complement */
	        &terminator
	      );
	if (sts == SMG$_EOF) /* i.e. terminator == SMG$K_TRM_CTRLZ */
	{
		strcpy( cmdstr, "EXIT" );
	}
	else
	{
		chkvmssts( sts );

		if (get_user_input && retlen == 0)
			strcpy( cmdstr, "NOP" );
		else
			cmdstr[retlen] = '\0';
	}
}


void read_command( char cmdstr[] )
{
	static POSITION current = {0,0};
	PIECE *selected = NULL,
	      *p;
	DIRECTION dir = DIR_NONE;
	LONG row, col, num;

	cmdstr[0] = '\0';

	while (cmdstr[0] == '\0')
	{
		row = ROW2BOARD( current.row );
		col = COL2BOARD( current.col );
		chkvmssts( smg$set_cursor_abs( &boardID, &row, &col ) );

		switch (read_key( 0 ))
		{
			case SMG$K_TRM_DO    :
			case SMG$K_TRM_PF4   : read_command_line( cmdstr );       break;

			case SMG$K_TRM_CTRLB : strcpy( cmdstr, "TAKE_BACK" );     break;
			case SMG$K_TRM_CTRLR : strcpy( cmdstr, "REDRAW_SCREEN" ); break;
			case SMG$K_TRM_CTRLZ : strcpy( cmdstr, "EXIT" );          break;
			case SMG$K_TRM_HELP  : strcpy( cmdstr, "HELP" );          break;

			case SMG$K_TRM_CTRLM :
			case SMG$K_TRM_ENTER :
				if (selected != NULL && POS_NE( selected->pos, current ))
				{
					sprintf( cmdstr, "MOVE %c%c %c%c",
					  POS_TO_2CHARS( selected->pos ),
					  POS_TO_2CHARS( current )
					);
					break;
				}
				/* FALLTHROUGH */
			case SMG$K_TRM_SELECT :
				if (selected == NULL)
				{
					if ((p = board[current.row][current.col]) != NULL)
					{
						if (p->color != side_to_move)
						{
							message( 1, "It's %s's turn.",
							         (side_to_move == WHITE) ? "White" : "Black" );
						}
						else
						{
							selected = p;
							put_string(
							  boardID,
							  ROW2BOARD( selected->pos.row ),
							  COL2BOARD( selected->pos.col ),
							  (selected->color == WHITE) ? WHITE_CHAR : BLACK_CHAR,
							  SMG$M_BLINK
							);
						}
					}
				}
				else
				{
					put_string(
					  boardID,
					  ROW2BOARD( selected->pos.row ),
					  COL2BOARD( selected->pos.col ),
					  (selected->color == WHITE) ? WHITE_CHAR : BLACK_CHAR,
					  0
					);
					selected = NULL;
				}
				break;

			case SMG$K_TRM_KP5 : /* jump to selected position */
				if (selected != NULL)
					current = selected->pos;
				break;

			case SMG$K_TRM_UP    :
			case SMG$K_TRM_KP8   : dir = DIR_UP;         break;
			case SMG$K_TRM_DOWN  :
			case SMG$K_TRM_KP2   : dir = DIR_DOWN;       break;
			case SMG$K_TRM_LEFT  :
			case SMG$K_TRM_KP4   : dir = DIR_LEFT;       break;
			case SMG$K_TRM_RIGHT :
			case SMG$K_TRM_KP6   : dir = DIR_RIGHT;      break;
			case SMG$K_TRM_KP7   : dir = DIR_UP_LEFT;    break;
			case SMG$K_TRM_KP9   : dir = DIR_UP_RIGHT;   break;
			case SMG$K_TRM_KP1   : dir = DIR_DOWN_LEFT;  break;
			case SMG$K_TRM_KP3   : dir = DIR_DOWN_RIGHT; break;

			default :
				if (get_user_input)
					strcpy( cmdstr, "NOP" );
		}

		if (dir != DIR_NONE)
		{
			if (selected != NULL)
			{
				if (calculate_target_square( selected, dir, &current ))
					clear_message();
				else
					message( 1, "Move not possible" );
			}
			else
			{
				current.row += row_off[dir];
				if (current.row < 0)
					current.row = 7;
				else
				if (current.row > 7)
					current.row = 0;

				current.col += col_off[dir];
				if (current.col < 0)
					current.col = 7;
				else
				if (current.col > 7)
					current.col = 0;
			}
			dir = DIR_NONE;
		}
	}

	/* De-select */
	if (selected != NULL)
		put_string(
		  boardID,
		  ROW2BOARD( selected->pos.row ),
		  COL2BOARD( selected->pos.col ),
		  (selected->color == WHITE) ? WHITE_CHAR : BLACK_CHAR,
		  0
		);

	/* Clear command line */
	row = 1; col = 1;
	num = sizeof(COMMAND_PROMPT)-1 + MAX_INPUT_LEN;
	chkvmssts( smg$erase_chars( &cmdID, &num, &row, &col ) );

	clear_message();
}


void show_piece( COLOR color, POSITION pos )
{
	if (hide_computing)
		return;
	put_string(
	  boardID,
	  ROW2BOARD( pos.row ),
	  COL2BOARD( pos.col ),
	  (color == NONE) ? EMPTY_CHAR : (color == WHITE) ? WHITE_CHAR : BLACK_CHAR,
	  0
	);
}


void welcome( void )
{
	LONG row = DISPLAY_HEIGHT/2 -1,
	     col = DISPLAY_WIDTH/2 -3;
	$DESCRIPTOR( title_dsc, "LOA" );
	ULONG rendition = SMG$M_BOLD;

	chkvmssts( smg$put_chars_highwide(
	             &screenID,
	             &title_dsc,
	             &row,
	             &col,
	             &rendition,
	             NULL, /* rendition complement */
	             NULL  /* character set */
	         ));
	put_string( screenID, row+3, DISPLAY_WIDTH/2 -strlen(VERSION)/2, VERSION, rendition );
	put_string( screenID, row+5, DISPLAY_WIDTH/2 -24/2, "A Game by Claude Soucie", rendition );

	read_key( 5 ); /* Wait */

	chkvmssts( smg$erase_display(
	             &screenID,
	             NULL, /* start row */
	             NULL, /* start column */
	             NULL, /* end row */
	             NULL  /* end column */
	         ));
}


void show_board( void )
{
	POSITION pos;
	char chr[] = " ";
	int i;

	if (hide_computing)
		return;

	chkvmssts( smg$begin_display_update( &screenID ) );

	for (i = 0, chr[0] = '8'; i < 8; ++i, --chr[0])
	{
		put_string( screenID, ROW2SCREEN(i), COL2SCREEN(-1)-1, chr, SMG$M_BOLD );
		put_string( screenID, ROW2SCREEN(i), COL2SCREEN( 8)+1, chr, SMG$M_BOLD );
	}
	for (i = 0, chr[0] = 'A'; i < 8; ++i, ++chr[0])
	{
		put_string( screenID, ROW2SCREEN(-1)-1, COL2SCREEN(i), chr, SMG$M_BOLD );
		put_string( screenID, ROW2SCREEN( 8)+1, COL2SCREEN(i), chr, SMG$M_BOLD );
	}

	for (pos.row = 0; pos.row < 8; ++pos.row)
		for (pos.col = 0; pos.col < 8; ++pos.col)
			show_piece( NONE, pos );

	chkvmssts( smg$end_display_update( &screenID ) );
}


void show_move( int moveno, MOVE *move )
{
	int  moveno2 = (moveno + 1) / 2;
	LONG row, col;
	char movenostr[7+2], movestr[12];

	if (hide_computing)
		return;

	if ((row = moveno2 - moves_row_off) < 0)
		return;

	if (row > MAX_MOVES_ROW)
	{
		row = MAX_MOVES_ROW;
		if (move->color == WHITE)
			chkvmssts( smg$scroll_display_area(
			             &movesID,
			             NULL, /* start row =1 */
			             NULL, /* start column =1 */
			             NULL, /* height =display height */
			             NULL, /* width =display width */
			             NULL, /* direction =SMG$M_UP */
			             NULL  /* count =1 */
			          ));
		++moves_row_off;
	}

	if (move->color == WHITE)
	{
		col = 1;
		sprintf( movenostr, "%3d.  ", moveno2 );
	}
	else
	{
		col = 1 + MOVES_WIDTH -1-5;
		movenostr[0] = '\0';
	}

	sprintf( movestr, "%.*s%c%c%c%c%c",
	  (move->color == WHITE) ? 6 : 0,
	  movenostr,
	  POS_TO_2CHARS( move->from ),
	  (move->beats_piece != NULL) ? ':' : '-',
	  POS_TO_2CHARS( move->to )
	);
	put_string( movesID, row, col, movestr, is_analysis ? SMG$M_REVERSE : 0 );
}


void unshow_move( int moveno, MOVE *move )
{
	int  moveno2 = (moveno + 1) / 2;
	LONG row, col, num;

	if (hide_computing)
		return;

	if ((row = moveno2 - moves_row_off) < 0 ||
	     row > MAX_MOVES_ROW)
		return;

	if (move->color == WHITE)
	{
		col = 1;
		num = 4+2+5;
	}
	else
	{
		col = 1 + MOVES_WIDTH -1-5;
		num = 1+5;
	}

	chkvmssts( smg$erase_chars( &movesID, &num, &row, &col ) );
}


void redraw( void )
{
	chkvmssts( smg$repaint_screen( &pasteboardID ) );
}


void help( char *topic )
{
	char fqfn[NAM$C_MAXRSS];
	char topic2[MAX_INPUT_LEN];
	$DESCRIPTOR( fqfn_dsc, fqfn );
	$DESCRIPTOR( topic_dsc, topic2 );
	struct FAB myfab = cc$rms_fab;
	struct NAM mynam = cc$rms_nam;
	ULONG sts;
	ULONG helpID;

	/* Build help library file name from LOA_HELP.HLB and program path */
	/* ToDo: only take path from program_name, supply ".HLB;" in
	         default filespec to faciliate logical name LOA_HELP */
	myfab.fab$l_fna = HELPLIB;
	myfab.fab$b_fns = (UBYTE) strlen( HELPLIB );
	myfab.fab$l_dna = program_name;
	myfab.fab$b_dns = (UBYTE) strlen( program_name );
	myfab.fab$l_nam = &mynam;
	mynam.nam$l_esa = fqfn;
	mynam.nam$b_ess = (UBYTE) sizeof( fqfn );

	sts = sys$parse( &myfab, NULL, NULL );
	if (sts == RMS$_FNF)
	{
		message( 1, "Help file not found" );
		return;
	}
	chkvmssts( sts );
	fqfn_dsc.dsc$w_length = mynam.nam$b_esl;

	chkvmssts( smg$unpaste_virtual_display( &screenID, &pasteboardID ));

	create_display(
	  &helpID,
	  DISPLAY_HEIGHT, DISPLAY_WIDTH,
	  0,
	  0
	);
	paste_display( helpID, 1, 1, 0 );

	strcpy( topic2, "LOA " );
	strcat( topic2, topic );
	topic_dsc.dsc$w_length = strlen( topic2 );

	chkvmssts( smg$put_help_text(
	             &helpID,
	             &keyboardID,
	             &topic_dsc,
	             &fqfn_dsc,
	             NULL, /* rendition set */
	             NULL  /* rendition complement */
	         ));

	chkvmssts( smg$delete_virtual_display( &helpID ) );

	paste_display( screenID, 1, 1, 0 );
	paste_game_displays();
}


static get_input(
	ULONG pasteboard_id,
	ULONG ast_argument,
	ULONG p_r0,
	ULONG p_r1,
	ULONG p_pc,
	ULONG p_ps
)
{
	extern volatile int abort_computation;

	abort_computation = TRUE;
	get_user_input    = TRUE;
}


void enable_unsolicited_input( void )
{
	chkvmssts( smg$enable_unsolicited_input(
	             &pasteboardID,
	             (AST_ROUTINE)get_input,
	             0L
	         ));
}


void disable_unsolicited_input( void )
{
	chkvmssts( smg$disable_unsolicited_input( &pasteboardID ) );
}
