/*
 *  $Id: demo.c,v 1.15 2001/05/21 13:43:21 martin Exp martin $
 *
 *  $Log: demo.c,v $
 *  Revision 1.15  2001/05/21 13:43:21  martin
 *  Minor typo in nootice().
 *
 *  Revision 1.14  2001/05/02 15:55:38  martin
 *  Test output from option [2] is scrolled off by larger number of menu options.
 *  Show out connection string.
 *
 *  Revision 1.13  2001/04/04 11:29:48  martin
 *  Change press_key() to take a message string.
 *  Add new help option.
 *  Minor changes to requirements() - more informative.
 *
 *  Revision 1.12  2001/03/30 09:28:56  martin
 *  Fix bug ID 1098 - targetdsns input for option 7 longer than 4 chrs
 *  	corrupt the built connection string.
 *
 *  Revision 1.11  2001/02/23 14:38:59  martin
 *  Copyright changed to 2001.
 *
 *  Revision 1.10  2000/11/22 13:11:53  martin
 *  When testing connection to ALL DSNs set the connection timeout to 10s.
 *
 *  Revision 1.9  2000/11/15 19:14:35  martin
 *  FIxed compile warnings - mostly not using SQLCHAR.
 *
 *  Revision 1.8  2000/08/10 14:52:01  martin
 *  Remove compilation warnings.
 *  Add DRIVER= for test 7 or it fails on Windows.
 *
 *  Revision 1.7  2000/08/09 16:22:57  martin
 *  Loads of updates with ne options.
 *
 *  Revision 1.6  2000/06/27 13:41:50  martin
 *  Changes for VMS.
 *
 *  Revision 1.5  2000/04/20 10:09:38  martin
 *  Add 2000 to copyright.
 *
 *  Revision 1.4  1999/10/14 13:34:09  martin
 *  Fix unintialised char ptr.
 *
 *  Revision 1.3  1999/10/14 10:50:34  martin
 *  Chnages for Windows.
 *
 *  Revision 1.2  1999/06/11 09:36:20  martin
 *  Update.
 *
 *  Revision 1.1  1999/03/15 15:03:03  martin
 *  Initial revision
 *
 */

#ifdef WIN32
#include <windows.h>
#include <io.h>
#define access _access
#endif

#include <stdio.h>
#include <string.h>
#if defined(VMS)
#include <unixio.h>
#elif !defined(WIN32)
#include <unistd.h>
#endif
#include <errno.h>

#if !defined(F_OK)
# define F_OK 0
#endif

#if !defined(R_OK)
# define R_OK 4
#endif

#include <sql.h>
#include <sqlext.h>

static void notice(void);
static char menu(int redisplay);
static void requirements(void);
static void display_ini(void);
static SQLRETURN test(void);
static SQLRETURN test_all(void);
static SQLRETURN test_new(void);
static SQLRETURN contact(void);
static int check_for_ini(void);
static void extract_error(char *fn, SQLHANDLE handle, SQLSMALLINT type);
static void press_key(char *msg);
static void driver_info(SQLHDBC dbc);
static void help(void);

/*
 *  Global data for test_new() so we can save the defaults.
 */
#define MAX_ATTR_LEN 100
static char     targetdsn[MAX_ATTR_LEN]="pubs";
static char     server[MAX_ATTR_LEN]="demo";
#ifdef WIN32
static char     driver[MAX_ATTR_LEN]="Easysoft ODBC-ODBC Bridge";
#else
static char     driver[MAX_ATTR_LEN]="OOB";
#endif
static char     port[MAX_ATTR_LEN]="8888";
static char     transport[MAX_ATTR_LEN]="TCP/IP";
static char     logonuser[MAX_ATTR_LEN]="demo";
static char     logonauth[MAX_ATTR_LEN]="easysoft";
static char     targetuser[MAX_ATTR_LEN]="demo";
static char     targetauth[MAX_ATTR_LEN]="easysoft";
static char     additional[512]="";

int main(int argc, char *argv[])
{
    char                choice;
    int                 redisplay = 1;
    SQLRETURN           ret;
    
    notice();
    while(1)
    {
        choice = menu(redisplay);
        switch(choice)
        {
          case '1':
          {
              notice();
              redisplay = 1;
              break;
          }
          case '2':
          {
              requirements();
              redisplay = 1;
              break;
          }
          case '3':
          {
              display_ini();
              redisplay = 1;
              break;
          }
          case '4':
          {
              ret = test();
              redisplay = 1;              
              break;
          }
          case '5':
          {
              ret = contact();
              redisplay = 1;
              break;
          }
          case '6':
          {
              test_all();
              redisplay = 1;
              break;
          }
          case '7':
          {
              test_new();
              redisplay = 1;
              break;
          }
          case '8':
          {
              help();
              redisplay = 1;
              break;
          }
          case '9':
          {
              printf("Bye\n");
              return 0;
          }
          default:
          {
              printf("\nInvalid choice\n");
              redisplay = 0;
          }
        }
    }
    
    return 0;
}


static SQLRETURN test_new(void)
{
    struct attrs
    {
        char    *prompt;
        char    *default_value;
        char    *value;
    };
    struct attrs tattrs[]=
    {
        {
            "Driver (ODBC driver)",
            driver,
            driver
        },
        {
            "Server (IP address or FQDN) where OOB Server installed",
            server,
            server
        },
        {
            "Port (the OOB Server usually listens on port 8888)",
            port,
            port
        },
        {
            "Transport (OOB currently only supports TCP/IP",
            transport,
            transport
        },
        {
            "TargetDSN (the name of the SYSTEM DSN on the remote machine",
            targetdsn,
            targetdsn
        },
        {
            "LogonUser (valid username for remote machine)",
            logonuser,
            logonuser
        },
        {
            "LogonAuth (password for LogonUser)",
            logonauth,
            logonauth
        },
        {
            "TargetUser (database username - may not be required)",
            targetuser,
            targetuser
        },
        {
            "TargetAuth (password for TargetUser)",
            targetauth,
            targetauth
        },
        {
            "Additional attributes of the form attribute=value, each attribute/value\n"
            "pair separated by a semicolon. e.g. ENG=engine;DBN=database;",
            additional, additional
        },
        {
            NULL, NULL, NULL
        }
    };
    unsigned int        i;
    char                inconstr[1024];
    char                inbuf[MAX_ATTR_LEN];
    SQLHENV             env;
    SQLHDBC             dbc;
    SQLCHAR             outstr[1024];
    SQLSMALLINT         outstr_len;
    SQLRETURN           ret;
    
    printf("\n"
"This option will allow you to connect to any ODBC driver on a remote\n"
"machine where the OOB Server is running even if you do not have a DSN\n"
"in your odbc.ini (i.e. a DSN-less connection).\n"
"You do not need a DSN defined in an odbc.ini file to connect to a remote\n"
"data source as all the attributes defining a connection may be passed\n"
"to the ODBC function SQLDriverConnect().\n\n"
"This option will prompt you for the attributes, compile a connection\n"
"string and attempt to connect. Defaults are displayed in [] and may\n"
"be used by pressing <enter>.\n\n");

    press_key(NULL);

    ret = SQLAllocHandle(SQL_HANDLE_ENV, 0, &env);
    if (!SQL_SUCCEEDED(ret))
    {
        fprintf(stderr, "Failed to allocate environment handle\n");
        return ret;
    }

    SQLSetEnvAttr(env, SQL_ATTR_ODBC_VERSION, (void *) SQL_OV_ODBC3, 0);

    ret = SQLAllocHandle(SQL_HANDLE_DBC, env, &dbc);
    if (!SQL_SUCCEEDED(ret))
    {
        fprintf(stderr, "Failed to allocate connection handle\n");
        return ret;
    }
    
    for (i = 0; tattrs[i].prompt; i++)
    {
        printf("%s [%s] : ", tattrs[i].prompt, tattrs[i].default_value);
        fgets(inbuf, sizeof(inbuf), stdin);
        if (inbuf[strlen(inbuf) - 1] == '\n')
            inbuf[strlen(inbuf) - 1] = '\0';
        if (strlen(inbuf) != 0) strcpy(tattrs[i].value, inbuf);
    }
    sprintf(inconstr, "Driver=%s;Server=%s;Port=%s;Transport=%s;TargetDSN=%s;"
            "LogonUser=%s;LogonAuth=%s;TargetUser=%s;TargetAuth=%s;%s",
            driver, server, port, transport, targetdsn, logonuser, logonauth,
            targetuser, targetauth, additional);
    printf("\nUsing connection string:\n%s\n", inconstr);
    printf("\nPassing connection string to SQLDriverConnect to attempt "
           "a connection...\n\n");
    
    ret = SQLDriverConnect(dbc, NULL, (SQLCHAR *)inconstr, SQL_NTS,
                           outstr, sizeof(outstr), &outstr_len,
                           SQL_DRIVER_COMPLETE);
    if (!SQL_SUCCEEDED(ret))
    {
        extract_error("SQLDriverConnect", dbc, SQL_HANDLE_DBC);
    }
    else
    {
        printf("Connection OK\n");
        driver_info(dbc);
        SQLDisconnect(dbc);
    }
    
    SQLFreeHandle(SQL_HANDLE_DBC, dbc);
    SQLFreeHandle(SQL_HANDLE_ENV, env);
    
    if (!SQL_SUCCEEDED(ret)) return ret;
    
    
    return SQL_SUCCESS;
}


static SQLRETURN test_all(void)
{
    SQLHENV             env;
    SQLHDBC             dbc;
    SQLUSMALLINT        direction;
    char                dsn[64];
    char                description[128];
    SQLRETURN           ret;

    printf("\n"
"This option runs through all the DSN in the local odbc.ini file and attempts\n"
"to connect to each one. You will get an OK or FAIL next to each data source.\n"
"If a DSN fails the diagnostics are output.\n\n");
    
    ret = SQLAllocHandle(SQL_HANDLE_ENV, 0, &env);
    if (!SQL_SUCCEEDED(ret))
    {
        fprintf(stderr, "Failed to allocate environment handle\n");
        return ret;
    }

    SQLSetEnvAttr(env, SQL_ATTR_ODBC_VERSION, (void *) SQL_OV_ODBC3, 0);

    ret = SQLAllocHandle(SQL_HANDLE_DBC, env, &dbc);
    if (!SQL_SUCCEEDED(ret))
    {
        fprintf(stderr, "Failed to allocate connection handle\n");
        return ret;
    }
    ret = SQLSetConnectAttr(dbc, SQL_ATTR_CONNECTION_TIMEOUT,
                            (void *)10, SQL_IS_UINTEGER);
    if (!SQL_SUCCEEDED(ret))
    {
        fprintf(stderr, "Failed to set connection timeout\n");
    }
    
    direction = SQL_FETCH_FIRST;
    printf("%-32s %-39s %s\n", "DSN", "DESCRIPTION", "STATUS");
    printf("%-32s %-39s %s\n", "---", "-----------", "------");
    
    while(SQL_SUCCESS ==
          SQLDataSources(env, direction, (SQLCHAR *)dsn, sizeof(dsn), NULL,
                         (SQLCHAR *)description, sizeof(description), NULL))
    {
        char            inconstr[512];
        SQLCHAR         outstr[1024];
        SQLSMALLINT     outstr_len;
        
        direction = SQL_FETCH_NEXT;
        printf("%-32s %-39s", dsn, description);

        sprintf(inconstr, "DSN=%s;", dsn);
        ret = SQLDriverConnect(dbc, NULL, (SQLCHAR *)inconstr,
                               SQL_NTS, outstr, sizeof(outstr),
                               &outstr_len, SQL_DRIVER_COMPLETE);
        if (SQL_SUCCEEDED(ret))
        {
            printf(" OK\n");
            SQLDisconnect(dbc);
        }
        else
        {
            printf(" FAIL\n");
            extract_error("SQLDriverConnect", dbc, SQL_HANDLE_DBC);
        }
    }
    SQLFreeHandle(SQL_HANDLE_DBC, dbc);
    SQLFreeHandle(SQL_HANDLE_ENV, env);
    
    press_key(NULL);    
    return SQL_SUCCESS;
}


static SQLRETURN test(void)
{
    SQLHENV             env;
    SQLHDBC             dbc;
    SQLRETURN           ret;
    SQLCHAR             outstr[1024];
    SQLSMALLINT         outstr_len;

    if (check_for_ini() != 0) return SQL_ERROR;
    
    /* SQLSetConnectAttr((SQLHDBC)NULL, SQL_ATTR_TRACEFILE, "trace.log", SQL_NTS); */
    /* SQLSetConnectAttr((SQLHDBC)NULL, SQL_ATTR_TRACE, (SQLPOINTER)SQL_OPT_TRACE_ON, 0); */
    
    ret = SQLAllocHandle(SQL_HANDLE_ENV, 0, &env);
    if (!SQL_SUCCEEDED(ret))
    {
        fprintf(stderr, "Failed to allocate environment handle\n");
        return ret;
    }

    SQLSetEnvAttr(env, SQL_ATTR_ODBC_VERSION, (void *) SQL_OV_ODBC3, 0);
    ret = SQLAllocHandle(SQL_HANDLE_DBC, env, &dbc);
    if (!SQL_SUCCEEDED(ret))
    {
        fprintf(stderr, "Failed to allocate connection handle\n");
        return ret;
    }

    ret = SQLDriverConnect(dbc, NULL, (SQLCHAR *)"DSN=demo", SQL_NTS,
                           outstr, 256, &outstr_len, SQL_DRIVER_COMPLETE);
    if (ret != SQL_SUCCESS)
    {
        extract_error("SQLDriverConnect", dbc, SQL_HANDLE_DBC);
    }
    if (!SQL_SUCCEEDED(ret)) return ret;
    printf("\nConnected\n\n");
    printf("OutConnectionString:\n%s\n\n", outstr);

    driver_info(dbc);
    
    printf("Disconnecting\n");
    
    SQLDisconnect(dbc);
    SQLFreeHandle(SQL_HANDLE_DBC, dbc);
    dbc = NULL;
    SQLFreeHandle(SQL_HANDLE_ENV, env);
    env = NULL;
    
    press_key(NULL);
    return SQL_SUCCESS;
}


static SQLRETURN contact(void)
{
    SQLHENV             env;
    SQLHDBC             dbc;
    SQLHSTMT            stmt;
    SQLRETURN           ret;
    SQLCHAR             outstr[1024];
    SQLSMALLINT         outstr_len;
    unsigned int        i;
    SQLSMALLINT         columns;
    
    if (check_for_ini() != 0) return SQL_ERROR;
    
    ret = SQLAllocHandle(SQL_HANDLE_ENV, 0, &env);
    if (!SQL_SUCCEEDED(ret))
    {
        fprintf(stderr, "Failed to allocate environment handle\n");
        return ret;
    }

    SQLSetEnvAttr(env, SQL_ATTR_ODBC_VERSION, (void *) SQL_OV_ODBC3, 0);
    ret = SQLAllocHandle(SQL_HANDLE_DBC, env, &dbc);
    if (!SQL_SUCCEEDED(ret))
    {
        fprintf(stderr, "Failed to allocate connection handle\n");
        return ret;
    }

    ret = SQLDriverConnect(dbc, NULL, (SQLCHAR *)"DSN=demo", SQL_NTS,
                           outstr, 256, &outstr_len, SQL_DRIVER_COMPLETE);
    if (ret != SQL_SUCCESS)
    {
        extract_error("SQLDriverConnect", dbc, SQL_HANDLE_DBC);
    }
    if (!SQL_SUCCEEDED(ret)) return ret;
    printf("\nConnected\n\n");

    ret = SQLAllocHandle(SQL_HANDLE_STMT, dbc, &stmt);
    if (!SQL_SUCCEEDED(ret))
    {
        extract_error("SQLAllocHandle", dbc, SQL_HANDLE_DBC);
        return ret;
    }

    
    ret = SQLExecDirect(stmt,
                        (SQLCHAR *)"select * from oobdist_contact", SQL_NTS);
    if (!SQL_SUCCEEDED(ret))
    {
        extract_error("SQLExecDirect", stmt, SQL_HANDLE_STMT);
        return ret;
    }

    ret = SQLNumResultCols(stmt, &columns);
    if (!SQL_SUCCEEDED(ret))
    {
        extract_error("SQLNumResultCols", stmt, SQL_HANDLE_STMT);
        return ret;
    }
    
    while(SQL_SUCCEEDED(ret = SQLFetch(stmt)))
    {
        char            cbuf[1024];
        SQLINTEGER      clen;
        
        for (i = 1; i <= columns; i++)
        {
            
            ret = SQLGetData(stmt, i, SQL_C_CHAR, cbuf, sizeof(cbuf), &clen);
            if (!SQL_SUCCEEDED(ret))
            {
                extract_error("SQLGetData", stmt, SQL_HANDLE_STMT);
                return ret;
            }
            if (i == 1)
                printf("%s\n", cbuf);
            else
                printf("\t%s\n", cbuf);                
        }
        printf("\n");
        
    }
    
            
    printf("Disconnecting\n");
    
    SQLDisconnect(dbc);
    SQLFreeHandle(SQL_HANDLE_DBC, dbc);
    dbc = NULL;
    SQLFreeHandle(SQL_HANDLE_ENV, env);
    env = NULL;
    
    press_key(NULL);
    return SQL_SUCCESS;
}


static int check_for_ini(void)
{
#ifndef WIN32
    if (access("odbc.ini", F_OK) != 0)
    {
        fprintf(stderr,
                "\nFailed to find an odbc.ini file in the current directory\n");
        fprintf(stderr,
"You need an odbc.ini file to describe the DSN used by this demo.\n"
"It should look like this:\n"
"\n"
"[demo]\n"
"SERVER = demo.easysoft.com\n"
"PORT = 8888\n"
"TRANSPORT = tcpip\n"
"TARGETDSN = demo\n"
"LOGONUSER = demo\n"
"LOGONAUTH = easysoft\n"
"TARGETUSER = demo\n"
"TARGTEAUTH = easysoft\n\n"
                );
        return 1;
    }
    else if (access("odbc.ini", R_OK) != 0)
    {
        fprintf(stderr, "\nFound an odbc.ini file but cannot read it.\n");
        fprintf(stderr, "Please check the permissions on odbc.ini\n\n");
        return 1;
    }
#endif
    return 0;
}


static void requirements(void)
{
    printf(
"\n"
"If you are using the supplied odbc.ini file containing a demo DSN\n"
"pointing to our demo server then you must be connected to the Internet to\n"
"successfully run the\n"
"\"Test Connection\" and\n"
"\"Get Easysoft contact information from remote database\"\n"
"options in this demonstration.\n"
"\n"
"You should also have an odbc.ini file defining the DSN demo in the\n"
"current working directory (an appropriate odbc.ini came with the OOB\n"
"installation)\n"
"\n"
"If you are not connected to the Internet but have installed your own\n"
"OOB Server on an accessible machine you should edit ./odbc.ini and\n"
"change the demo DSN to point to your server.\n"
);
    press_key(NULL);
    
}


static void display_ini(void)
{
#ifndef WIN32
    FILE        *fp;
    char        buf[256];
#endif
    
#ifdef WIN32
    printf("This option is not supported in Windows\n");
#else
    fp = fopen("./odbc.ini", "r");
    if (!fp)
    {
        fprintf(stderr, "Failed to open ./odbc.ini\n-%s\n",
                strerror(errno));
        return;
    }
    while(fgets(buf, sizeof(buf), fp))
    {
        printf("%s", buf);
    }
    fclose(fp);
#endif
    return;
}


static char menu(int redisplay)
{
    char                buf[5];

    if (redisplay)
    {
        printf("\n");
        printf("[1] Read the initial notice.\n");
        printf("[2] Requirements for the demo.\n");
        printf("[3] Display ./odbc.ini\n");
        printf("[4] Test Connection to demo data source\n");
        printf("[5] Get Easysoft contact information from remote database.\n");
        printf("[6] Test connection to all DSNs in your odbc.ini\n");
        printf("[7] Test connection to new datasource not in odbc.ini\n");
        printf("[8] It did not work, I need help\n");
        printf("[9] Exit.\n");
    }
    printf("\nChoose an option: ");
    fgets(buf, sizeof(buf), stdin);
    return buf[0];

}


static void notice(void)
{
    printf(
"Easysoft Ltd (c) 1999-2001\n\n"
"Demonstration of the Easysoft ODBC-ODBC Bridge\n\n"
"This is a demonstration of the Easysoft ODBC-ODBC Bridge. The best way\n"
"to follow this small demonstration is to select the menu options one at\n"
"a time. The demo connects your machine through the ODBC-ODBC Bridge to\n"
"MS SQLServer running on the demo machine (demo.easysoft.com) at Easysoft\n"
"and downloads data from it. Easysoft respect your privacy and so NO data\n"
"whatsoever from your machine is uploaded to Easysoft. The source code for\n"
"this program is available in the ODBC-ODBC Bridge distribution should you\n"
"want to inspect it before continuing with the demonstration.\n"
);
    return;
}


static void driver_info(SQLHDBC dbc)
{
    struct xinfo
    {
        SQLUSMALLINT    attr;
        char            *attr_name;
        char            *heading;
    };
    unsigned int        i;
    char                info[100];
    SQLSMALLINT         info_len;
    
    struct xinfo xinfo[] =
    {
        {SQL_DATABASE_NAME, "SQL_DATABASE_NAME", "Connected to database: "},
        {SQL_DBMS_NAME, "SQL_DBMS_NAME", "DBMS Name: "},
        {SQL_DRIVER_NAME, "SQL_DRIVER_NAME", "Driver Name: "},
        {SQL_DRIVER_VER, "SQL_DRIVER_VER", "Driver Version: "}
    };
    SQLRETURN           ret;

    for (i = 0; i < (sizeof(xinfo) / sizeof(xinfo[0])); i++)
    {
        ret = SQLGetInfo(dbc, xinfo[i].attr, info, sizeof(info),
                         &info_len);
        if (!SQL_SUCCEEDED(ret))
        {
            char            err[256];

            sprintf(err, "SQLGetInfo(...,%s,...)", xinfo[i].attr_name);
            extract_error(err, dbc, SQL_HANDLE_DBC);
        }
        else
        {
            info[info_len] = '\0';
            printf("%s %s\n", xinfo[i].heading, info);
        }
    }
}


static void extract_error(
    char *fn,
    SQLHANDLE handle,
    SQLSMALLINT type)
{
    SQLINTEGER i = 0, native;
    SQLCHAR state[ 7 ];
    SQLCHAR text[256];
    SQLSMALLINT len;
    int ret;

    fprintf(stderr,
            "\n"
            "The driver reported the following diagnostics whilst running "
            "%s\n\n",
            fn);
    
    do 
    {
        ret = SQLGetDiagRec(type, handle, ++i, state, &native, text,
                            sizeof(text), &len );
        if (SQL_SUCCEEDED(ret))
            printf( "%s:%ld:%ld:%s\n", state, i, native, text );
    }
    while( ret == SQL_SUCCESS );
}


static void press_key(char *msg)
{
    char        buf[5];
    
    if (!msg)
        printf("\nPress enter to continue\n");
    else
        printf(msg);
    
    fgets(buf, sizeof(buf), stdin);
}


static void help(void)
{
    printf(
"\nIf the options to test a connection and/or retrieve the oobdist_contact\n"
"table contents did not work:\n"
"\n"
"[1] Read the demo_help.txt document in this directory\n"
"[2] Consult the OOB FAQ in the OOB doc directory\n");
    press_key("\nPress enter for more information\n");
    printf(
"\nSome common problems are:\n"
"[1] All options in the demo (except option 7) use a data source name\n"
"    hard-wired as \"demo\". Therefore, you must have a DSN called demo\n"
"    in the odbc.ini file in the current working directory to use those\n"
"    options.\n"
"[2] Examine the DSN server attribute and make sure the machine specified\n"
"    is running, contactable over your network and has an OOB Server\n"
"    installed on it.\n"
"[3] Make sure the TargetDSN attribute is set to a SYSTEM DSN you have\n"
"    created on the server machine. The OOB Server can usually only see\n"
"    system dsns.\n"
"[4] Make sure LogonUser and LogonAuth in the DSN are a valid username\n"
"    password for the Server machine. e.g. if the server is running on NT\n"
"    you should be able to physically login at the keyboard/terminal\n"
"    of that server NT machine with LogonUser/LogonAuth.\n"
"[5] Make sure the DSN contains at least the Server, Port, TargetDSN,\n"
"    LogonUser and LogonAuth attributes. The DSN not found and no default\n"
"    driver error can also mean an attribute was not found - perhaps\n"
"    because it was misspelt (attribute names are case insensitive but the\n"
"    attribute values are not).\n"
);
    press_key("\nPress enter to return to menu\n");
    
}
    
