/*
 This file is part of SLRN.

 Copyright (c) 1994, 1999 John E. Davis <davis@space.mit.edu>

 This program is free software; you can redistribute it and/or modify it
 under the terms of the GNU General Public License as published by the Free
 Software Foundation; either version 2 of the License, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 more details.

 You should have received a copy of the GNU General Public License along
 with this program; if not, write to the Free Software Foundation, Inc., 675
 Mass Ave, Cambridge, MA 02139, USA. 
*/
#include "config.h"
#include "slrnfeat.h"

#include <stdio.h>
#include <string.h>

#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif

#include <slang.h>
#include "jdmacros.h"

#include "slrn.h"
#include "misc.h"
#include "util.h"

static char *Global_Help [] = 
{
   "",
     "",
     " More information about slrn can be found on its home page:",
     "   http://slrn.sourceforge.net/",
     " Questions not covered by the documentation can be posted to",
     "   news.software.readers",
     "",
     " Please email bug reports, suggestions or comments to the current maintainer,",
     "   Thomas Schultz <tststs@gmx.de>",
     " The original author of this program is John E. Davis <davis@space.mit.edu>",
   NULL
};

static char *Art_Help[] =
{
/* begin makehelp(article) - do *NOT* modify this line */
	" Note: The keys are case sensitive!  That is, 's' and 'S' are not the same.",
	"General movement:",
	"  n                  Go to the next unread article (or next group, if at end).",
	"  p                  Go to the previous unread article.",
	"  N, ESC RIGHT       Skip to next group.",
	"  ESC LEFT           Go to previous group.",
	"  !                  Go to the next article with a high score.",
	"  =                  Go to the next article with the same subject.",
	"  L                  Go to the last read article and display it.",
	"Actions:",
	"  P                  Post a new article (no followup).",
	"  ESC P              Post or edit a postponed article.",
	"  f                  Post a followup to the current article.",
	"      ESC 1 f        Include all headers in the followup.",
	"      ESC 2 f        Followup without modifying (e.g. quoting) the article.",
	"  r                  Reply to poster (via email).",
	"  F                  Forward the current article to someone (via email).",
	"  ESC Ctrl-S         Supersede article (you have to be the author).",
	"  ESC Ctrl-C         Cancel article (you have to be the author).",
	"  o                  Save article, tagged articles or thread to file.",
	"  |                  Pipe article to an external program.",
	"  y                  Print article (as displayed).",
	"      ESC 1 y        Print article (unwrapped and including hidden lines).",
#if SLRN_HAS_DECODE
	"  :                  Decode article, tagged articles or thread.",
#endif
	"  Ctrl-Z             Suspend slrn.",
	"  q                  Return to group display.",
	"  Q                  Quit slrn immediately.",
	"Moving in the article pager:",
	"  ESC DOWN           Scroll article down one line.",
	"  ESC UP             Scroll article up one line.",
	"  SPACE              Scroll article down one page (or select next, if at end).",
	"  DELETE, b          Scroll article up one page.",
	"  >                  Move to end of the article.",
	"  <                  Move to beginning of the article.",
	"  LEFT               Pan article to the left.",
	"  RIGHT              Pan article to the right.",
	"  /                  Search forward in the article.",
	"  TAB                Skip beyond quoted text.",
	"  g                  Skip to next digest.",
	"Moving in the header display:",
	"  DOWN, Ctrl-N       Move to the next article.",
	"  UP, Ctrl-P         Move to the previous article.",
	"  Ctrl-D             Scroll down one page.",
	"  Ctrl-U             Scroll up one page.",
	"  ESC >              Go to the last article in group.",
	"  ESC <              Go to the first article in group.",
	"  j                  Jump to article (by server number).",
	"  a                  Author search forward.",
	"  A                  Author search backward.",
	"  s                  Subject search forward.",
	"  S                  Subject search backward.",
	"Marking as read/unread:",
	"  d                  Mark article as read.",
	"  u                  Mark article as unread.",
	"  ESC d              Mark entire (sub-)thread as read.",
	"  c                  Catchup - mark all articles as read.",
	"  C                  Mark all articles up to the current position as read.",
	"  ESC u              Un-Catchup - mark all articles as unread.",
	"  ESC U              Mark all articles up to the current position as unread.",
	"  x                  Remove all non-tagged read articles from the list.",
	"Article pager commands:",
	"  t                  Show full headers (on/off).",
	"  ESC r              Decrypt ROT-13 (on/off).",
	"  T                  Display quoted lines (on/off).",
	"  \\                  Show signature (on/off).",
	"  W                  Wrap long lines (on/off).",
#if SLRN_HAS_SPOILERS
	"  ESC ?              Reveal spoilers (on/off).",
#endif
	"  ]                  Show PGP signature (on/off).",
#if SLRN_HAS_VERBATIM_MARKS
	"  [                  Show verbatim marks (on/off).",
#endif
	"  Ctrl-^             Enlarge the article window.",
	"  ^                  Shrink the article window.",
	"  z                  Maximize / Unmaximize the article window.",
	"  h                  Hide / Show the article window.",
	"  U                  Search for URLs and follow them.",
	"Header window commands:",
	"  ESC t              Collapse / Uncollapse thread.",
	"      ESC 1 ESC t    Collapse / Uncollapse all threads.",
	"  ESC a              Toggle between header display formats.",
	"  ESC s              Select threading and sorting method.",
	"Miscellaneous actions:",
	"  K                  Create a scorefile entry interactively.",
	"      ESC 1 K        Edit scorefile.",
	"  * The following five commands query the server if necessary:",
	"  ESC l              Locate article by its Message-ID.",
	"  ESC Ctrl-P         Find all children of current article.",
	"  ESC p              Find parent article.",
	"      ESC 1 ESC p    Reconstruct thread (slow when run on large threads).",
	"      ESC 2 ESC p    Reconstruct thread (faster, may not find all articles).",
	"  ;                  Set a mark at the current article.",
	"  ,                  Return to previously marked article.",
	"  #                  Numerically tag article (for saving / decoding).",
	"  ESC #              Remove all numerical tags.",
	"  *                  Protect article from catchup commands.",
	"      ESC 1 *        Remove all protection marks.",
	"  .                  Repeat last key sequence.",
	"  Ctrl-X ESC         Read line and interpret it as S-Lang.",
#if SLRN_HAS_GROUPLENS
	"  0                  Rate article with GroupLens.",
#endif
	"  Ctrl-R, Ctrl-L     Redraw screen.",
	"  ?                  Display this help screen.",
/* end makehelp - do *NOT* modify this line */
     NULL
};

static char *Group_Help [] =
{
/* begin makehelp(group) - do *NOT* modify this line */
	" Note: The keys are case sensitive!  That is, 's' and 'S' are not the same.",
	"Cursor movement:",
	"  DOWN                    Go to the next group.",
	"  UP                      Go to the previous group.",
	"  Ctrl-V, Ctrl-D          Scroll to the next page.",
	"  ESC V, Ctrl-U           Scroll to the previous page.",
	"  ESC <                   Go to the bottom of the list.",
	"  ESC >                   Go to the top of the list.",
	"  /                       Group keyword search.",
	"Actions:",
	"  SPACE, RETURN           Enter the current newsgroup.",
	"  * The following variations also download previously read articles:",
	"      ESC 1 SPACE         Enter group with article number query.",
	"      ESC 2 SPACE         Enter group, but do not apply score.",
	"      ESC 3 SPACE         Enter group with query, but without scoring.",
	"      ESC 4 SPACE         Enter the current newsgroup.",
	"  P                       Post an article to the current newsgroup.",
	"  ESC P                   Post or edit a postponed article.",
	"  G                       Get new news from server.",
	"  K                       Select scoring mode.",
	"  .                       Repeat last key sequence.",
	"  Ctrl-X ESC              Read line and interpret it as S-Lang.",
	"  Ctrl-Z                  Suspend slrn.",
	"  q                       Quit slrn.",
	"Group management (affects newsrc file):",
	"  c                       Catchup - Mark all articles as read.",
	"  ESC u                   Un-Catchup - Mark all articles as unread.",
	"  a                       Add a new newsgroup.",
	"  s                       Subscribe to the current newsgroup.",
	"      ESC 1 s             Subscribe to groups matching a pattern.",
	"  u                       Unsubscribe from the current newsgroup.",
	"      ESC 1 u             Unsubscribe from groups matching a pattern.",
	"  m                       Move newsgroup to a different location.",
	"  Ctrl-X, Ctrl-T          Transpose position of groups.",
	"  X                       Force a save of the newsrc file.",
	"Display:",
	"  ESC a                   Toggle between group display formats.",
	"  l                       Toggle display of groups without unread articles.",
	"  L                       Toggle listing of unsubscribed groups.",
	"      ESC 1 L             Hide unsubscribed groups.",
	"      ESC 2 L             Show unsubscribed groups.",
	"  Ctrl-L, Ctrl-R          Redraw the screen.",
	"  ?                       Display this help screen.",
/* end makehelp - do *NOT* modify this line */
     NULL
};

#define MAX_HELP_LINES 256
static char *User_Article_Help[MAX_HELP_LINES];
static char *User_Group_Help[MAX_HELP_LINES];

static void do_help (char **help)
{
   int i;
   char **p, *sect = NULL;
   char quit;
   char **this_help;
   
   this_help = p = help;
   
   slrn_enable_mouse (0);
   
   while (1)
     {
	i = 0;
	if (*p == NULL) break;
	
	slrn_push_suspension (0);

	SLsmg_cls ();
	
	if ((sect != NULL) && (**p == ' '))
	  {
	     SLsmg_set_color (1);
	     SLsmg_gotorc (i, 0);
	     SLsmg_write_string (sect);
	     SLsmg_set_color (0);
	     SLsmg_write_string (" (continued)");
	     i += 2;
	  }
	
	while (i < SLtt_Screen_Rows - 4)
	  {
	     char pp;
	     
	     if (*p == NULL)
	       {
		  if (this_help == Global_Help) break;
		  this_help = p = Global_Help;
		  sect = NULL;
	       }
	     
	     pp = **p;
	     if ((pp != ' ') && pp)
	       {
		  sect = *p;
		  if ((i + 6) > SLtt_Screen_Rows) break;
		  i++;
		  SLsmg_set_color (1);
	       }
	     
	     SLsmg_gotorc (i, 0);
	     SLsmg_write_string (*p); i++;
	     if (pp && (pp != ' '))
	       {
		  SLsmg_set_color (0);
		  i++;
	       }
	     p++;
	  }
	
	SLsmg_gotorc (i + 1, 0);
	SLsmg_set_color (1);
	
	if ((*p == NULL)
	    && (this_help == help))
	  {
	     this_help = p = Global_Help;
	     sect = NULL;
	  }
	    
	if (*p == NULL)
	  {
	     SLsmg_write_string ("\
Press '?' to start over, or any other key to return to news reader."
				 );
	  }
	else SLsmg_write_string ("\
Press 'q' to quit help, '?' to start over, or any other key to continue."
				 );
	
	slrn_smg_refresh ();
	
	slrn_pop_suspension ();

	SLang_flush_input ();
	quit = SLang_getkey ();
	if (quit == '?')
	  {
	     this_help = p = help;
	     sect = NULL;
	  }
	else if ((*p == NULL) || ((quit | 0x20)== 'q'))
	    break;
     }
   Slrn_Full_Screen_Update = 1;
   slrn_set_color (0);
   /* slrn_redraw (); */
   SLang_flush_input ();
   slrn_enable_mouse (1);
}

int slrn_parse_helpfile (char *helpfile)
{
   FILE *fp;
   char buf[256];
   char ch;
   char **current_help = NULL;
   int num_lines = 0;
   unsigned char *b;

   if (Slrn_Batch)
     return 0;
   
   if (NULL == (fp = fopen (helpfile, "r"))) return -1;
   while (fgets (buf, sizeof (buf) - 1, fp) != NULL)
     {
	ch = *buf;
	
	/* Skip over common comments */
	if ((ch == '#') || (ch == '%') || (ch == ';') || (ch == '!'))
	  continue;
	
	b = (unsigned char *) slrn_skip_whitespace (buf);
	if (*b == 0) continue;
	
	if (ch == '[')
	  {
	     /* end current help */
	     if (current_help != NULL)
	       {
		  slrn_free (current_help[num_lines]);
		  current_help[num_lines] = NULL;
	       }
	     
	     num_lines = 0;
	     ch = *(buf + 1) | 0x20;
	     if (ch == 'a')
	       {
		  current_help = User_Article_Help;
	       }
	     else if (ch == 'g') current_help = User_Group_Help;
	     else current_help = NULL;
	     
	     continue;
	  }
	
	if (current_help == NULL) continue;
	
	if (MAX_HELP_LINES == num_lines + 1)
	  {
	     current_help[num_lines] = NULL;
	     current_help = NULL;
	     continue;
	  }
	
	slrn_free (current_help [num_lines]);
	
	if (NULL != (current_help [num_lines] = (char *) slrn_strmalloc (buf, 0)))
	  num_lines++;
     }
   if (current_help != NULL)
     {
	slrn_free (current_help[num_lines]);
	current_help[num_lines] = NULL;
     }
   slrn_fclose (fp);
   return 0;
}


void slrn_article_help (void)
{
   char **h;
   if (Slrn_Batch) return;
   if (User_Article_Help[0] != NULL) h = User_Article_Help; else h = Art_Help;
   do_help (h);
}

void slrn_group_help (void)
{
   char **h;
   if (Slrn_Batch) return;
   if (User_Group_Help[0] != NULL) h = User_Group_Help; else h = Group_Help;
   do_help (h);
}
