/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * Configuration management.
 * @author Christian Grothoff
 * @file common/configuration.c
 **/

#include "configuration.h"
#include "config.h"
#include "util/returnvalues.h"
#include "util/storage.h"
#include "util/ipcheck.h"

#include <netdb.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>



#define NETWORK_SECTION       "NETWORK"
#define NETWORK_IP            "IP"
#define NETWORK_INTERFACES    "INTERFACES"
#define NETWORK_PORT          "PORT"
#define NETWORK_BPS_TOTAL     "MAXNETBPSTOTAL"
#define NETWORK_MAX_LOAD      "MAXNETLOAD"
#define NETWORK_NOHELO        "NOHELO"
#define NETWORK_BLACKLIST     "BLACKLIST"
#define NETWORK_TRUSTED       "TRUSTED"
#define NETWORK_HOSTLISTURL   "HOSTLISTURL"

#define FILESHARING_SECTION   "FILESHARING"
#define FILESHARING_MIGRATION "ACTIVEMIGRATION"
#define FILESHARING_TIMEOUT   "SEARCHTIMEOUT"
#define FILESHARING_EXTRACTOR "EXTRACTORS"

#define FILES_SECTION         "FILES"
#define FILES_QUOTA           "DISKQUOTA"
#define FILES_HOSTKEY         "HOSTKEY"
#define FILES_DATABASE        "DATABASE"
#define FILES_AGEFILE         "AGEFILE"
#define FILES_COLLISION	      "COLLISION"
#define FILES_SHARED_LIST     "FILELIST"
#define FILES_CONTENT_DIR     "CONTENTDIR"
#define FILES_CREDIT_DIR      "CREDITDIR"
#define FILES_HOSTS_DIR       "HOSTS"

#define GNUNETD_SECTION       "GNUNETD"
#define GNUNETD_MAX_NODES     "MAXNODES"
#define GNUNETD_HELO_EXPIRES  "HELOEXPIRES"
#define GNUNETD_MAX_LOAD      "MAXLOAD"

/**
 * PRIVATE: gnet configuration
 **/
LHASH * config_;

static FileName sharedFileList_;
static FileName creditDirectory_;
static FileName contentDirectory_;
static FileName hostDirectory_;
static FileName ageFile_;
static FileName privHostKey_;
static FileName databaseFile_;
static FileName collisionFile_;
static CIDRNetwork * filteredNetworks_;
static CIDRNetwork * trustedNetworks_;

static FileName getFileName(char * section,
			    char * value,
			    char * errMsg) {
  FileName fn;
  fn = CONF_get_string(config_,
		       section,
		       value);
  if (fn == NULL) 
    errexit(errMsg,
	    section,
	    value);
  return expandFileName(fn);   
}

void readConfig(FileName filename) {
  long errorLine; /* in which line did an error occur */
  FileName fn;
  char * ch;
  
  fn = expandFileName(filename);
  if (!assertIsFile(fn))
    exit(-1);
#ifdef PRINT_CONFIGURATION
  print("Loading configuration from %s.\n", fn);
#endif  
  config_ = CONF_load(NULL, fn, &errorLine);
  xfree(fn,
	"readConfig: config filename");
  if (config_ == NULL) {
    ERR_load_crypto_strings();
    print("Unable to load configuration from %s. Error at line %ld.\n",
	  filename,
	  errorLine);
    ERR_print_errors_fp(stderr);
    exit(1);
  }
  privHostKey_ = 
    getFileName(FILES_SECTION,
		FILES_HOSTKEY,
		"Configuration file must specify filename for hostkey in section %s under %s.\n");
  sharedFileList_ = 
    getFileName(FILES_SECTION,
		FILES_SHARED_LIST,
		"Configuration file must specify filename for the list of on-demand shared files in section %s under %s.\n");
  creditDirectory_ =
    getFileName(FILES_SECTION,
		FILES_CREDIT_DIR,
		"Configuration file must specify directory for credits in section %s under %s.\n");
  contentDirectory_ =
    getFileName(FILES_SECTION,
		FILES_CONTENT_DIR,
		"Configuration file must specify directory for content in section %s under %s.\n");
  hostDirectory_ =
    getFileName(FILES_SECTION,
		FILES_HOSTS_DIR,
		"Configuration file must specify directory for network identities in section %s under %s.\n");
  collisionFile_ =
    getFileName(FILES_SECTION,
		FILES_COLLISION,
		"Configuration file must specify filename for collisions in section %s under %s.\n");
  databaseFile_ =
    getFileName(FILES_SECTION,
		FILES_DATABASE,
		"Configuration file must specify filename for the database in section %s under %s.\n");
  ageFile_ =
    getFileName(FILES_SECTION,
		FILES_AGEFILE,
		"Configuration file must specify filename for the database in section %s under %s.\n");
  /* create directories */
  mkdirp(hostDirectory_);
  mkdirp(creditDirectory_);
  mkdirp(contentDirectory_);
  ch = CONF_get_string(config_,
		       NETWORK_SECTION,
		       NETWORK_BLACKLIST);
  if (ch == NULL)
    ch = "";
  filteredNetworks_ = parseRoutes(ch);
  if (filteredNetworks_ == NULL) 
    errexit("Malformed entry in the configuration in section %s under %s: %s\n",
	    NETWORK_SECTION,NETWORK_BLACKLIST,ch);  
  ch = CONF_get_string(config_,
		       NETWORK_SECTION,
		       NETWORK_TRUSTED);
  if (ch == NULL)
    ch = "127.0.0.1/8;"; /* by default, trust localhost only */
  trustedNetworks_ = parseRoutes(ch);
  if (trustedNetworks_ == NULL) 
    errexit("Malformed entry in the configuration in section %s under %s: %s\n",
	    NETWORK_SECTION,NETWORK_TRUSTED,ch);  
}

void freeConfig() {
#ifdef PRINT_CONFIGURATION
  print("Freeing configuration.\n");
#endif  
  CONF_free(config_);
  free(privHostKey_);
  free(sharedFileList_);
  free(creditDirectory_);
  free(contentDirectory_);
  free(hostDirectory_);
  free(collisionFile_);
  free(databaseFile_);
  free(ageFile_);
  free(filteredNetworks_);
  free(trustedNetworks_);
}

/* **************** network section ***************** */

unsigned int getMaxNetBPSTotal() {
  return (unsigned int) CONF_get_number(config_, 
					NETWORK_SECTION,
					NETWORK_BPS_TOTAL);
}

unsigned int getMaxNetLoad() {
  return (unsigned int) CONF_get_number(config_, 
					NETWORK_SECTION,
					NETWORK_MAX_LOAD);
}

IPAddress getIP() {
  return CONF_get_string(config_,
			 NETWORK_SECTION,
			 NETWORK_IP);
}

Interfaces getNetInterfaces() {
  Interfaces i = CONF_get_string(config_,
				 NETWORK_SECTION,
				 NETWORK_INTERFACES);
  return i;
}

unsigned short getGNUnetPort() {
  struct servent	*pse;	/* pointer to service information entry	*/
  unsigned short port 
    = (unsigned short) CONF_get_number(config_, 
				       NETWORK_SECTION,
				       NETWORK_PORT);
  if (port == 0) { /* try lookup in services */
    if ((pse = getservbyname("gnunet", "udp"))) 
      port = htons(pse->s_port);      
    else 
      errexit("Cannot determine port to bind to. Define in configuration file in section %s under %s or in /etc/services under UDP/gnet.\n",
	      NETWORK_SECTION, NETWORK_PORT);
  }
  return port;
}

int isHELOEnabled() {
  unsigned char * option = CONF_get_string(config_,
					   NETWORK_SECTION,
					   NETWORK_INTERFACES);
  if (option == NULL)
    return YES; /* default: is gateway */
  if (strcmp(option,"YES"))
    return YES;
  else
    return NO;
}

CIDRNetwork * getNetBlacklist() {
  return filteredNetworks_;
}

CIDRNetwork * getTCPWhitelist() {
  return trustedNetworks_;
}

char * getHostlistURL() {
  char * i = CONF_get_string(config_,
			     NETWORK_SECTION,
			     NETWORK_HOSTLISTURL);
  return i;
}

/* ********************* Filesharing options ********************* */

unsigned int getSearchTimeout() {
  return (unsigned int) CONF_get_number(config_, 
					FILESHARING_SECTION,
					FILESHARING_TIMEOUT);
}

int activeMigration() {
  char *  f = CONF_get_string(config_,
			      FILESHARING_SECTION,
			      FILESHARING_MIGRATION);
  if (f == NULL)
    return NO; /* default: NO */
  if (0 == strcmp(f,"YES") )
      return YES;
  else
    return NO;
}

char * getExtractorConfig() {
  return CONF_get_string(config_,
			 FILESHARING_SECTION,
			 FILESHARING_EXTRACTOR);
}
/* ********************* File options ********************* */

unsigned int getDiskQuota() {
  return (unsigned int) CONF_get_number(config_, 
					FILES_SECTION,
					FILES_QUOTA);
}

FileName getHostKeyFilePriv() {
  return privHostKey_;
}

FileName getAgeFile() {
  return ageFile_;
}

FileName getDatabaseFile() {
  return databaseFile_;
}

FileName getCollisionFile() {
  return collisionFile_;
}

FileName getSharedFileList() {
  return sharedFileList_;
}

FileName getCreditDirectory() {
  return creditDirectory_;
}

FileName getContentDirectory() {
  return contentDirectory_;
}

FileName getHostsDirectory() {
  return hostDirectory_;
}

/* ************* server options ***************** */

unsigned int getMaxLoad() {
  return (unsigned int) CONF_get_number(config_,
					GNUNETD_SECTION,
					GNUNETD_MAX_LOAD);
}

unsigned int getMaxNodes() {
  return (unsigned int) CONF_get_number(config_, 
					GNUNETD_SECTION,
					GNUNETD_MAX_NODES);
}

/**
 * How many hours is a HELO message
 * valid?
 **/
unsigned int getHeloExpiration() {
  unsigned int exp = (unsigned int) CONF_get_number(config_, 
						    GNUNETD_SECTION,
						    GNUNETD_HELO_EXPIRES);
  if (exp < 1) 
    exp = 1;
  return exp;
}




/* end of configuration.c */
