/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * GNUnet Debugging module
 * @author Christian Grothoff
 * @file common/debugging.c
 **/

#include "debugging.h"
#include "config.h"
#include <netinet/in.h>

/**
 * print a debug message
 * @param format the string describing the error message
 **/
void print(const char *format, ...) {
  va_list	args;  
  va_start(args, format);
  vfprintf(PRINTTO, format, args);
  va_end(args);
}

/**
 * errexit - print an error message and exit
 * @param format the string describing the error message
 **/
void errexit(const char *format, ...) {
  va_list args;
  va_start(args, format);
  vfprintf(stderr, format, args);
  va_end(args);
  exit(1);
}

/**
 * Display the contents of an incomming UDP message.
 **/
void showMessageEncrypted(MessagePack * pack) {
  IPPortString ipPort;

  printAddress(&pack->sender,
	       &ipPort);
  print("Incomming packet from %s (size %5d-%4d, CRC %5d)\n",
	&ipPort,
	pack->size, pack->size & 0x7FFF,
	ntohl(pack->checkSum));
}

/**
 * Display the contents of an incomming UDP message.
 **/
void showMessageDecrypted(MessagePack * pack) {
  IPPortString ipPort;

  printAddress(&pack->sender,
	       &ipPort);
  print("Decrypted packet from %s (size %4d-%5d, CRC %5d)\n",
	&ipPort,
	pack->size, pack->size & 0x7FFF,
	ntohl(pack->checkSum));
}

/**
 * Print the connection address (IP and Port) encoded in the given
 * HostAddress to stderr. This also returns the address/port string to
 * the sender.
 * @param conInfo the HostAddress to print.
 * @param result output parameter for the resulting address string
 **/
void printAddress(HostAddress * conInfo,
		  IPPortString * result) {
  if (conInfo == NULL) {
    memcpy(result, 
	   "address unknown\0",
	   strlen("address unknown\0"+1));
    return;
  }
  sprintf((char*)result,
	  "%3d.%3d.%3d.%3d:%5d",
	  ((unsigned char *)&conInfo->senderIP)[0],
	  ((unsigned char *)&conInfo->senderIP)[1],
	  ((unsigned char *)&conInfo->senderIP)[2],
	  ((unsigned char *)&conInfo->senderIP)[3],
	  ntohs(conInfo->senderPort));
}

/**
 * Print the connection address (IP) encoded in the given
 * integer. This also returns the IP address string to the user.
 * @param theIP the encoded IP
 * @param result output parameter for the resulting address string
 **/
void printIP(int theIp,
	     IPString * result) {
  int size;
  size = sprintf((char*)result, 
		 "%3d.%3d.%3d.%3d",
		 ((unsigned char *)&theIp)[0],
		 ((unsigned char *)&theIp)[1],
		 ((unsigned char *)&theIp)[2],
		 ((unsigned char *)&theIp)[3]); 
}

void printSessionKey(SESSIONKEY * key,
		     SKEYString * result) {
  int i;
  int j = 0;
  for (i=0;i<SESSIONKEY_LEN;i++)
    j+=sprintf(&result->skey[j],"%3d- ",key->key[i]);
}

void dumpPacket(unsigned char * message,
		BLOCK_LENGTH len) {
   int total_len;
   int subpart_len;
   int pos;
   unsigned short packet_code;
   MessagePart * part;
   HELO_Body * hbody;
   QUERY_Message * qmsg;
   SKEY_Message * skeymsg;
   HexName temp_str;

   subpart_len = -1;

   total_len = len; 
   pos = 0;

   print("!!! BEGIN PACKET - Total length: %d !!!\n", 
	 total_len);
   while (pos < total_len) {
      part = (MessagePart*) &message[pos];
      subpart_len = ntohs(part->size);
      if (pos + subpart_len > total_len) {
	print("Invalid packet format (%d - %d) - dropping remaining parts (%d).\n",
	      pos, subpart_len, total_len);	
      }
      packet_code = ntohs(part->requestType);
     
      switch (packet_code) {
      case GNET_PROTO_HELO:
	print("HELO      Length: %4u Pos: %4u\n", subpart_len, pos);
	hbody = &(((HELO_Message*)(part))->body);
	break;
      case GNET_PROTO_SKEY:
	skeymsg = (SKEY_Message*) part;
	print("SKEY      Length: %4u Pos: %4u\n", 
	      subpart_len, 
	      pos);
	break;
      case GNET_PROTO_PING:
	print("PING      Length: %4u Pos: %4u\n",
	      subpart_len, pos);
	break;
      case GNET_PROTO_PONG:
	print("PONG      Length: %4u Pos: %4u\n",
	      subpart_len, pos);
	break;
      case GNET_PROTO_3QUERY:
	print("3QUERY    Length: %4u Pos: %4u\n", 
	      subpart_len, 
	      pos);
	qmsg = (QUERY_Message*)part; 
	print("          Priority: %4d\t",
	      ntohl(qmsg->priority));
	print("          TTL:      %4d\n",
	      ntohl(qmsg->ttl));
	hash2hex(&(qmsg->returnTo.hashPubKey), 
		 &temp_str);
	print("          ReturnTo: %s\n", 
	      &temp_str.data[0]);
	hash2hex(&(qmsg->query), &temp_str);
	print("          Query:    %s\n", 
	      &temp_str.data[0]);
	break;
      case GNET_PROTO_CONTENT:
	print("CONTENT   Length: %4u Pos: %4u\n", subpart_len, pos);
	break;
      case GNET_PROTO_TIMESTAMP:
	print("TIMESTAMP Length: %4u Value: %d\n",
	      subpart_len, ntohl((int) ((TIMESTAMP_Message*)part)->timeStamp));
	assert(subpart_len == sizeof(TIMESTAMP_Message),
	       "TIMESTAMP had unexpected size!");	       
	break;
      case GNET_PROTO_SEQUENCE:
	print("SEQUENCE  Length: %4u Value: %d\n",
	      subpart_len, ntohl(((SEQUENCE_Message*)part)->sequenceNumber));
	assert(subpart_len == sizeof(SEQUENCE_Message),
	       "SEQUENCE had unexpected size!");	       
	break;
       case GNET_PROTO_NOISE:
 	print("NOISE     Length: %4u\n",
	      subpart_len);
	break;
      case GNET_PROTO_HANGUP:
 	print("HANGUP    Length: %4u\n",
	      subpart_len);
	break;
      default: 
	print("*UNKNOWN* Length: %4u Pos: %4u Type: %d\n", 
	      subpart_len, pos, packet_code);
      }
      pos += ntohs(part->size);
   }
   assert(pos == total_len,
	  "packet parts larger than packet analyzed!");
   print("---- END PACKET - Total length: %d ---\n", total_len);
   
}

/**
 * Assert something.
 **/
void assert(int condition,
	    char * message) {
#ifdef CHECK_ASSERTS
  if (!condition)
    print("ASSERT FAILED: %s\n",message);
#endif
}


/* end of debugging.c */
