/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Code to maintain the list of currently known hosts
 * (in memory structure of data/hosts) and temporary blacklisting
 * information.
 * @author Christian Grothoff
 **/ 

#include "config.h"
#include "configuration.h"
#include "knownhosts.h"


typedef struct {
  HostIdentity identity;
  /** how long is this host blacklisted? **/
  time_t until;
  /** what would be the next increment for blacklisting? (in seconds) **/
  unsigned int delta;
} HostEntry;

/**
 * The current (allocated) size of knownHosts
 **/
static int max_;

/**
 * The number of actual entries in knownHosts
 **/
static int count_;

/**
 * A lock for accessing knownHosts
 **/
static Mutex lock_;

/**
 * The list of known hosts.
 **/
static HostEntry * hosts_;

/**
 * Initialize this module.
 **/
void initKnownhosts() {
  unsigned int i;
  unsigned int max;

  getHostsDirectory();
  max_   = 128;
  count_ = 0;
  create_mutex(&lock_);
  hosts_ = xmalloc(sizeof(HostEntry) * max_,
		   "initKnownhosts: hostlist");	      
  cronScanDirectoryDataHosts(NULL);
  addCronJob(&cronScanDirectoryDataHosts,
	     15*60, 15*60,
	     NULL); /* scan every 15 minutes */  
}

void doneKnownhosts() {
  destroy_mutex(&lock_);
  xfree(hosts_,
	"doneKnownhosts: hostlist");
}

/**
 * Check if 2 hosts are the same (returns 1 if yes)
 **/
int hostIdentityEquals(HostIdentity * first, 
		       HostIdentity * second) {
  if ( (first == NULL) || 
       (second == NULL) )
    return 0;
  return equalsHashCode160(&first->hashPubKey,
			   &second->hashPubKey);
}

/**
 * Add a host to the list.
 **/
void addHostToKnown(HostIdentity * identity) {
  int i;

  MUTEX_LOCK(&lock_);   
  for (i=0;i<count_;i++)
    if (hostIdentityEquals(identity,
			   &hosts_[i].identity)) {
      MUTEX_UNLOCK(&lock_);   
      return; /* already there */
    }
  if (count_ == max_) { /* grow */
    HostEntry * tmp = xmalloc(sizeof(HostEntry) * max_ * 2,
			      "addHostToKnown: grow hostlist");
    memcpy(tmp, 
	   hosts_,
	   sizeof(HostEntry) * max_);
    max_ = max_ * 2;
  }
  memcpy(&hosts_[count_].identity,
	 identity,
	 sizeof(HostIdentity));
  hosts_[count_].until = 0;
  hosts_[count_].delta = 0;
  count_++;
  MUTEX_UNLOCK(&lock_);   
}

/**
 * Delete a host from the list
 **/
void delHostFromKnown(HostIdentity * identity) {
  int i;

  MUTEX_LOCK(&lock_);   
  for (i=0;i<count_;i++)
    if (hostIdentityEquals(identity,
			   &hosts_[i].identity)) {
      memcpy(&hosts_[i],
	     &hosts_[count_-1],
	     sizeof(HostEntry));
      count_--;    
      MUTEX_UNLOCK(&lock_);   
      return; /* deleted */
    }
  MUTEX_UNLOCK(&lock_);   
}

/**
 * Blacklist a host. This method is called if a host
 * failed to respond to a connection attempt.
 * @return OK on success SYSERR on error
 **/
int blacklistHost(HostIdentity * identity) {
  int i;

  MUTEX_LOCK(&lock_);   
  for (i=0;i<count_;i++)
    if (hostIdentityEquals(identity,
			   &hosts_[i].identity)) {
      hosts_[i].delta = hosts_[i].delta * 2 + randomi(30);
      if (hosts_[i].delta > 60*60*6) 
	hosts_[i].delta = 60*60*6+randomi(30); /* longest blacklist is 6h */
      time(&hosts_[i].until);      
      hosts_[i].until += hosts_[i].delta;
      MUTEX_UNLOCK(&lock_);   
     return OK;
    }
  MUTEX_UNLOCK(&lock_);   
  return SYSERR;
}

/**
 * Whitelist a host. This method is called if a host
 * successfully established a connection. It typically
 * resets the exponential backoff to the smallest value.
 * @return OK on success SYSERR on error
 **/
int whitelistHost(HostIdentity * identity) {
  int i;

  MUTEX_LOCK(&lock_);   
  for (i=0;i<count_;i++)
    if (hostIdentityEquals(identity,
			   &hosts_[i].identity)) {
      hosts_[i].delta = 0;
      hosts_[i].until = 0;
      MUTEX_UNLOCK(&lock_);   
      return OK;
    }
  MUTEX_UNLOCK(&lock_);   
  return SYSERR;
}

/**
 * Call a method for each known host; not reentrant!
 * @param callback the method to call for each host
 * @param now the time to use for excluding hosts due to blacklisting, use 0 
 *        to go through all hosts.
 * @param data an argument to pass to the method
 * @return the number of hosts matching
 **/
int forEachHost(void (*callback)(HostIdentity*, void *), 
		time_t now,
		void * data) {
  int i;
  int count = 0;

  MUTEX_LOCK(&lock_);   
  for (i=0;i<count_;i++) 
    if (now > hosts_[i].until) {
      count++;
      if (callback != NULL)
	callback(&hosts_[i].identity, data);
    }  
  MUTEX_UNLOCK(&lock_);   
  return count;
}

static void cronHelper(HexName * id, void * unused) {
  HostIdentity identity;

  hex2hash(id, &identity.hashPubKey);
  addHostToKnown(&identity);
}
 
/**
 * Call this method periodically to scan data/hosts for new hosts.
 **/
void cronScanDirectoryDataHosts(void * unused) {
  int count;

#if PRINT_CRON
  print("CRON: enter cronScanDirectoryDataHosts\n");
#endif
  count = scanDirectory(getHostsDirectory(),
			&cronHelper,
			NULL);
  if (count <= 0) {
#if PRINT_WARNINGS 
    print("KNOWNHOSTS: scanDirectory %s returned no known hosts!\n",
	  getHostsDirectory());
#endif
  }
#if PRINT_CRON
  print("CRON: exit cronScanDirectoryDataHosts\n");
#endif
}


/* end of knownhosts.c */
