/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * This module is responsible to lookup queries.
 * @file common/lookup.c
 * @author Christian Grothoff
 **/

#include "config.h"
#include "lookup.h"
#include "util/contentdatabase.h"


#include <dirent.h>

/* ********************* GLOBALS ***************** */

/* handle of the database-file, never closed yet */
static int LOOKUP_handle_;

/* semaphore for synced r/w access to the file */
static Mutex LOOKUP_fileSem_;

/* names of known files */
static FileName * LOOKUP_fileNames_;

/* number of files that are directly shared */
static unsigned int LOOKUP_fileCount_;

/* number of files that are in data/content */
static unsigned int LOOKUP_filesStored_;

/* what is the priority of the least important content
   that we are storing */
static int LOOKUP_leastImportance_;

/* the hashcode of the least important entry */
static HashCode160 LOOKUP_lip_;

/* the least important entries */
static int LOOKUP_lips_[LOOKUP_LIPS_COUNT];
/* the number of valid entries in lookup-lips */
static int LOOKUP_lips_count_;
/* the current partition of the database for lips */
static int LOOKUP_lips_db_index_;

/* value of the current base value for
   fresh content (used to time-out
   old content) */
static int LOOKUP_age;

/* ************* internal Methods **************** */

/**
 * Cron-job that makes sure the lookup-module is fine
 * by synchronizing the in-memory database with the drive.
 **/
static void cronSyncLookup(void * unused);

/**
 * Cron-job that decreases the importance-level of all
 * files by 1. Runs 'not very often'.
 **/
static void cronReduceImportance(void * unused);

/* scan the list of on-demand shared files to
   initialize LOOUP_fileNames_ */
static void scanDatabaseList();

/**
 * Given an entry in the cache, obtain the matching
 * message(s). Send them to the given host.
 * @param entry the entry to look for
 * @param method a method pointer used for callback; this is the
 *               method to be called if the lookup succeeds. 
 *               must take three arguments: NULL, a pointer
 *               to the CONTENT_Block, and the query.
 **/ 
static void resolveEntry(ContentIndex * entry,
			 int (* method)(HostIdentity*,CONTENT_Block*,
					 HashCode160*));

/**
 * Lookup the given Hash in the Hashtable-cache structure
 * (load blocks as needed).
 * @returns NO if not found, YES if found
 **/
static int findEntry(HashCode160 * query,
		     ContentIndex * entry);

/**
 * Write the given entry to the database.
 **/
static void writeEntry(ContentIndex * entry);

/**
 * Delete an entry and the corresponding file on the
 * drive.
 **/
static void deleteEntry(HashCode160 * entry);

/**
 * Find the last important entry and
 * set the LOOKUP_lip_ and LOOKUP_leastImportance_
 * variables accordingly.
 **/
static void findLeastImportant();

/**
 * Check if the file the query is for is on the
 * local drive. If yes, send the reply and return YES,
 * otherwise return NO.
 * @param query the filename to look for
 * @param method a method pointer used for callback; this is the
 *               method to be called if the lookup succeeds. 
 *               must take three arguments: NULL, a pointer
 *               to the CONTENT_Block, and the query.
 * @return YES if the reply succeeded, else NO
 **/
static int localLookup(HashCode160 * query,		       
		       int (* method)(HostIdentity*,CONTENT_Block*,
				       HashCode160*));


/* ******************* CODE... ****************** */

static void indexContent(HashCode160 * doubleHash) {
  ContentIndex entry;

  memcpy(&entry.doubleHash,
	 doubleHash,
	 sizeof(HashCode160));
  entry.importance = 0;
  entry.fileNameIndex = htonl(-1); /* directly available in data/content */
  entry.fileOffset = htonl(-1); 
  addEntry(&entry);
}

static void scanThread(void * unused) {
  LOOKUP_filesStored_ = forEachEntryInDatabase(&indexContent);  
}

static void cronPeriodicScan(void * unused) {
  pthread_t scanThread_handle;
#if PRINT_CRON
  print("CRON: enter cronPeriodicScan\n");
#endif
  pthread_create(&scanThread_handle, NULL,
		 (void * (*)(void *)) &scanThread, NULL);
#if PRINT_CRON
  print("CRON: exit cronPeriodicScan\n");
#endif
}

/**
 * Initialize the lookup-module.
 **/
void initLookup() {  
  FileName fileName;
  int handle;
  
  fileName = getAgeFile();
  handle = open(fileName,O_CREAT,S_IREAD|S_IWRITE);
  LOOKUP_age = 0;
  read(handle, 
       &LOOKUP_age,
       sizeof(int));
  close(handle);
  LOOKUP_lips_db_index_ = rand() % LOOKUP_LIPSPART_COUNT;
  LOOKUP_lips_count_ = 0;

#if PRINT_STATUS
  print("Indexing data/content database... ");
#endif
  LOOKUP_leastImportance_ = 0;
  initLookup_collision();
  /* scan lookup-file index */
  scanDatabaseList();

  fileName = getDatabaseFile();
  LOOKUP_handle_ = open(fileName, O_CREAT|O_RDWR, S_IRUSR|S_IWUSR);  
  if (LOOKUP_handle_ < 0)
    errexit("Could not open/create database!\n");
  create_mutex(&LOOKUP_fileSem_);
  printf("count: %d\n",LOOKUP_filesStored_);
#if PRINT_STATUS
  printf("Done.\n");
#endif
}

void fullInitLookup() {
  if (YES == activeMigration())
    findLeastImportant();
  /* schedule cron job */
  cronPeriodicScan(NULL);
  addCronJob(&cronSyncLookup,
	     5*60, 5*60, NULL); /* flush every 5 minutes */
  /* schedule cron job */
  addCronJob(&cronReduceImportance,
	     6*60*60, 12*60*60,
	     NULL); /* decrease every 12h, but first time after 6h */
  /* schedule cron job */
  addCronJob(&cronPeriodicScan,
	     30*60,30*60,
	     NULL); /* every 30 minutes, scan for new content */
}

/**
 * Obtain the number of shared files.
 **/
unsigned int getSharedFileCount() {
  return LOOKUP_fileCount_;
}

/**
 * Check if the file the query is for is on the
 * local drive. If yes, send the reply and return YES,
 * otherwise return NO.
 * @param query the filename to look for
 * @param method a method pointer used for callback; this is the
 *               method to be called if the lookup succeeds. 
 *               must take three arguments: NULL, a pointer
 *               to the CONTENT_Block, and the query.
 * @return YES if success, else NO              
 **/
static int localLookup(HashCode160 * query,
		       int (* method)(HostIdentity*,CONTENT_Block*,
				       HashCode160*)) {
  CONTENT_Block buf[MAX_FSIZE];
  int i,j;
 
#if PRINT_QUERY
  HexName hex;
  hash2hex(query, &hex);

  print("QUERY: local lookup for %s.\n",
	&hex);
#else
#if PRINT_WARNINGS
  HexName hex;
#endif
#endif

  j = readContent(query, 
		  MAX_FSIZE * CONTENT_SIZE, 
		  buf);

  if (j <= 0) {
#if PRINT_QUERY
    print("QUERY: local lookup: %s not found!\n",
	  (char*) &hex);
#endif
    return NO;
  }
  /* data found on local node! */
  if ((j % CONTENT_SIZE) != 0) {
#if PRINT_WARNINGS
#if 0 == PRINT_QUERY
  hash2hex(query, &hex);
#endif
    print("Bad content file: %s\n",
	  (char*)&hex);
#endif
    return NO;
  }
#if PRINT_QUERY
  print("QUERY: local lookup succeded. Sending reply (%s)!\n",
	(char*)&hex);
#endif

  /* send multiple replies if applicable */
  j=j/CONTENT_SIZE;
  /*  method = callback; */
  srand((unsigned) time(0));
  i=(int)((float)j*rand()/(RAND_MAX)); /* select one of the entries at random */
  method(NULL, /* comes from local */
	 &buf[i], 
	 query);  
  return YES;
}

/* scan the list of on-demand shared files to
   initialize LOOUP_fileNames_ */
static void scanDatabaseList() {
  FileName fil;
  FILE * handle;
  int result;
  unsigned char line[65536];

  LOOKUP_fileCount_ = 0;
  fil = getSharedFileList();
  handle = fopen(fil, "a+");
  if (handle == NULL) {
#if PRINT_WARNINGS
    print("LOOKUP: List %s of directly shared filenames not available!\n",
	  fil);
#endif    
    return;
  }
  fseek(handle,0,SEEK_SET);
  result = 1;
  LOOKUP_fileCount_ = -1;
  while (result != EOF) {
    LOOPPRINT("scanDatabaseList (1)");
    result = fscanf(handle, "%65535s\n", line);
    LOOKUP_fileCount_++;
  }
  if (LOOKUP_fileCount_ == 0) {
    fclose(handle);
    return;
  }
  fseek(handle, 0, SEEK_SET);
  LOOKUP_fileNames_ = xmalloc(LOOKUP_fileCount_ * sizeof(FileName),
			      "scanDatabaseList: filenameList");
  LOOKUP_fileCount_ = 0;
  result = 1;
  while (result != EOF) {
    LOOPPRINT("scanDatabaseList (2)");
    result = fscanf(handle, "%65535s\n", line);
    if (result != EOF) {
      LOOKUP_fileNames_[LOOKUP_fileCount_] = xmalloc(strlen(line)+1,
						     "scanDatabaseList: filenamelist");
      memcpy(LOOKUP_fileNames_[LOOKUP_fileCount_],
	     line,
	     strlen(line)+1);
      LOOKUP_fileCount_++;
    }
  }
  fclose(handle);  
}

/**
 * Add a name to the list of filenames.
 * @returns the number of filenames in the index
 **/
int appendFilename(char * filename) {
  FileName fil;
  FILE * handle;
  int result;
  unsigned char line[65536];
  int scanf;

  LOOKUP_fileCount_ = 0;
  fil = getSharedFileList();
  handle = fopen(fil, "r+");
  if (handle == NULL) 
    errexit("LOOKUP: List %s of directly shared filenames not available!\n",
	    fil);
  filename = expandFileName(filename);   
  scanf = 1;
  result = -1;
  LOOKUP_fileCount_ = 0;
  while (scanf != EOF) {
    LOOPPRINT("appendFilename");
    scanf = fscanf(handle, "%65535s\n", line);
    if (scanf != EOF)
      line[scanf] = 0;
    if (strcmp(line,filename) == 0)
      result = LOOKUP_fileCount_+1;
    LOOKUP_fileCount_++;
  }
  if (result != -1) {
    fclose(handle);  
    xfree(filename, "appendFilename: expanded filename");  
    return result; /* already there! */
  }
  /* not there, append */
  fprintf(handle,"%s\n",filename);
  fclose(handle);  
  xfree(filename, "appendFilename: expanded filename");  
  scanDatabaseList();
  return LOOKUP_fileCount_; /* return index */
}

/**
 * Is space available on the drive? Returns the number of
 * blocks (1k) available. If there would be no space except
 * if less important content is discarded, this method 
 * *deletes* 1k of the less important content and returns 
 * 1. If there is only more important content, the
 * method returns 0.
 * @param importance how important is the content?
 **/
int spaceAvailable(int importance) {  
  if (LOOKUP_filesStored_ > getDiskQuota() * 1024) {
    if (importance+LOOKUP_age <= LOOKUP_leastImportance_)
      return 0; /* no */
    /* free lip entry */
    deleteEntry(&LOOKUP_lip_);
    findLeastImportant();
    return 1;
  } else {
    LOOKUP_filesStored_++; /* it is safe to assume that there will be one more file very soon... */
    return getDiskQuota() * 1024 - LOOKUP_filesStored_; /* this many blocks are free */
  }     
}

/**
 * Cron-job that makes sure the lookup-module is fine
 * by synchronizing the in-memory database with the drive.
 **/
static void cronSyncLookup(void * unused) {
#if PRINT_CRON
  print("CRON: enter cronSyncLookup\n");
#endif
#if HAVE_FDATASYNC
  assert(fdatasync(LOOKUP_handle_) == 0,
	 "fdatasync failed on database!");
#else
  assert(fsync(LOOKUP_handle_) == 0,
	 "fdatasync failed on database!");
#endif
#if PRINT_CRON
  print("CRON: exit cronSyncLookup\n");
#endif
}

/**
 * Cron-job that decreases the importance-level of all
 * files by 1. Runs 'not very often'.
 **/
static void cronReduceImportance(void * unused) {
 FileName fileName;
  int handle;
  
#if PRINT_CRON
  print("CRON: enter cronReduceImportance\n");
#endif
  fileName= getAgeFile();
  handle = open(fileName,O_CREAT,S_IWRITE);
  LOOKUP_age++;
  write(handle, 
       &LOOKUP_age,
       sizeof(int));
  close(handle);
#if PRINT_CRON
  print("CRON: exit cronReduceImportance\n");
#endif
}

/**
 * Delete an entry and the corresponding file on the
 * drive.
 * @param entry the double-hash of the item to remove (=filename)
 **/
static void deleteEntry(HashCode160 * entry) {
  int noll = 0;
  int index;
  HashCode160 tripleHash;
  int i;

  /* remove entry from database */
  hash(&entry,
       sizeof(HashCode160),
       &tripleHash);
  index = tripleHash.a & (CACHE_HASH_TABLE_SIZE-1);
  index *= sizeof(ContentIndex);
  MUTEX_LOCK(&LOOKUP_fileSem_);
  lseek(LOOKUP_handle_, index, SEEK_SET);
  /* fill database with zeros */
  for (i=0;i<(sizeof(ContentIndex)>>2);i++)
    write(LOOKUP_handle_, &noll, sizeof(int));
  MUTEX_UNLOCK(&LOOKUP_fileSem_);

  /* remove the file */
  unlinkFromDB(entry);
}

/**
 * Find the last important entry and
 * set the LOOKUP_lip_ and LOOKUP_leastImportance_
 * variables accordingly.
 **/
static void findLeastImportant() {
  ContentIndex entry;
  int j;
  int i;
  int k;
  int importances[LOOKUP_LIPS_COUNT];
 
  if (LOOKUP_filesStored_ < LOOKUP_LIPS_COUNT*LOOKUP_FACTOR)
    return;
  MUTEX_LOCK(&LOOKUP_fileSem_);
  while (LOOKUP_lips_count_ == 0) {
    LOOPPRINT("findLeastImportant");
    for (k=0;k<LOOKUP_LIPS_COUNT;k++)
      importances[k] = 0x7FFFFFFF;
    LOOKUP_lips_db_index_++;
    LOOKUP_lips_db_index_ = LOOKUP_lips_db_index_ % LOOKUP_LIPSPART_COUNT;
    lseek(LOOKUP_handle_, 
	  LOOKUP_lips_db_index_*(CACHE_HASH_TABLE_SIZE/LOOKUP_LIPSPART_COUNT)*sizeof(ContentIndex), 
	  SEEK_SET);
    for (i=LOOKUP_lips_db_index_*(CACHE_HASH_TABLE_SIZE/LOOKUP_LIPSPART_COUNT);
	 i<(LOOKUP_lips_db_index_+1)*(CACHE_HASH_TABLE_SIZE/LOOKUP_LIPSPART_COUNT);
	 i++) {
      j = read(LOOKUP_handle_, &entry, sizeof(ContentIndex));
      if (j == sizeof(ContentIndex)) {
	if (ntohl(entry.fileNameIndex) != -1) {
	  continue; /* on demand encoding or free */
	}
	for (k=0;k<LOOKUP_LIPS_COUNT;k++)
	  if (ntohl(entry.importance) < importances[k]) {
	    importances[k] = ntohl(entry.importance);
	    LOOKUP_lips_[k] = i;
	    break;
	  }
      }
    }
    for (k=0;k<LOOKUP_LIPS_COUNT;k++)
      if (importances[k] < 0x7FFFFFFF)
	LOOKUP_lips_count_++; /* found entry with low importance */
  }
  k = LOOKUP_lips_[--LOOKUP_lips_count_];
  lseek(LOOKUP_handle_, k*sizeof(ContentIndex), SEEK_SET);
  j = read(LOOKUP_handle_, &entry, sizeof(ContentIndex));
  if (j == sizeof(ContentIndex)) {
    LOOKUP_leastImportance_ = ntohl(entry.importance);
    memcpy(&LOOKUP_lip_,
	   &entry.doubleHash,
	   sizeof(HashCode160));      
  } /* else: warning: DB inconsistent? */
  MUTEX_UNLOCK(&LOOKUP_fileSem_);
}

/**
 * Lookup the file on the local machine. If found,
 * send it back.
 * @param query the triple-hash of the content
 * @param qp the policy (for us: priority)
 * @param callback procedure to call on the results,
 *        must take three arguments: NULL, a pointer
 *        to the CONTENT_Block, and the query.
 * @returns YES if found, NO if not.
 **/
int tripleHashLookup(HashCode160 * query,
		     QUERY_POLICY qp,
		     int (* callback)(HostIdentity*,CONTENT_Block*,
				       HashCode160*)) {
  ContentIndex ci;
  if (NO == findEntry(query,&ci)) {
    return NO;
  }
  ci.importance = htonl(ntohl(ci.importance)+
			(qp&QUERY_PRIORITY_BITMASK)); /* increase importance */
  writeEntry(&ci); /* write to disk! */
  resolveEntry(&ci, callback); /* send reply */
  return YES; /* report success */
}

/**
 * Given an entry in the cache, obtain the matching
 * message(s). Send them to the given host.
 **/ 
static void resolveEntry(ContentIndex * entry,
			 int (* method)(HostIdentity*,CONTENT_Block*,
					HashCode160*)) {
  CONTENT_Block result;
  CONTENT_Block crypt;
  FileName fn;
  int fileHandle;
  ssize_t blen;
  HashCode160 hc;

  /* data/content mechanism */
  if (ntohl(entry->fileNameIndex) == -1) {
    localLookup(&entry->doubleHash,
		method);
    return;
  }
  if (ntohl(entry->fileNameIndex) > LOOKUP_fileCount_) {
#if PRINT_WARNINGS
    print("Database inconsistent! (index points to invalid offset %d (>= %d))\n",
	  ntohl(entry->fileNameIndex), LOOKUP_fileCount_);
#endif
    return;
  }
  /* on-demand encoding mechanism */
  fn = LOOKUP_fileNames_[ntohl(entry->fileNameIndex)-1];
  fileHandle = open(fn, O_EXCL, S_IRUSR);
  if (fileHandle == -1) {
#if PRINT_WARNINGS
    print("Could not open file %s.\n",
	  fn);    
#endif
    return;
  }
  lseek(fileHandle, ntohl(entry->fileOffset), SEEK_SET);
  blen = read(fileHandle, 
	      &result,
	      sizeof(CONTENT_Block));
  close(fileHandle);
  hash(&result, 
       blen, 
       &hc);
  if (OK == encryptContent(&result,
			   &hc,
			   &crypt)) {
    /*    method = callback; */
    method(NULL, 
	   &crypt, 
	   &entry->doubleHash);  
  }
}

/**
 * Lookup the given Hash in the Hashtable-cache structure
 * (load blocks as needed).
 * @returns NO if not found, YES if found
 **/
static int findEntry(HashCode160 * query,
		     ContentIndex * entry) {
  unsigned int index;
  HashCode160 tripleHash;

  index = query->a & (CACHE_HASH_TABLE_SIZE-1);
  index *= sizeof(ContentIndex);
  MUTEX_LOCK(&LOOKUP_fileSem_);
  lseek(LOOKUP_handle_, index, SEEK_SET);
  entry->fileNameIndex = 0;
  read(LOOKUP_handle_, entry, sizeof(ContentIndex));
  MUTEX_UNLOCK(&LOOKUP_fileSem_);
  hash(&entry->doubleHash,
       sizeof(HashCode160),
       &tripleHash);
  if (entry->fileNameIndex == 0) {
    return NO; /* no collision, nothing there */
  }
  if (equalsHashCode160(query, &tripleHash)) 
    return YES;
  else {
    return findCollisionEntry(query, entry);  
  }
}

/**
 * Write the given entry to the database. 
 **/
void addEntry(ContentIndex * entry) {
  unsigned int index;
  int size;
  HashCode160 tripleHash;
  ContentIndex onDisk;

  hash(&entry->doubleHash,
       sizeof(HashCode160),
       &tripleHash);
  entry->importance = htonl(ntohl(entry->importance)+LOOKUP_age);
  index = tripleHash.a & (CACHE_HASH_TABLE_SIZE-1);
  index *= sizeof(ContentIndex);
  MUTEX_LOCK(&LOOKUP_fileSem_);
  lseek(LOOKUP_handle_, index, SEEK_SET);
  size = read(LOOKUP_handle_, &onDisk, sizeof(ContentIndex));
  MUTEX_UNLOCK(&LOOKUP_fileSem_);
  if ( (size == 0) || 
       (onDisk.fileNameIndex == 0) ) {
    writeEntry(entry);
    return; /* no collision, nothing there, just write */
  }
  if (equalsHashCode160(&onDisk.doubleHash, &entry->doubleHash)) {
    writeEntry(entry); /* update */
    return;
  }
  /* got to collision DB */
  addCollisionEntry(entry);
  return;  
}

/**
 * Write the given entry to the database.
 **/
static void writeEntry(ContentIndex * entry) {
  unsigned int index;
  HashCode160 tripleHash;
  hash(&entry->doubleHash,
       sizeof(HashCode160),
       &tripleHash);
  
  index = tripleHash.a & (CACHE_HASH_TABLE_SIZE-1);
  index *= sizeof(ContentIndex);
  MUTEX_LOCK(&LOOKUP_fileSem_);
  lseek(LOOKUP_handle_, index, SEEK_SET);
  write(LOOKUP_handle_, entry, sizeof(ContentIndex));
  MUTEX_UNLOCK(&LOOKUP_fileSem_);
}


/* end of lookup.c */
