/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * sessionkey - standard communication with another node.
 * encryptContent - given a data block and the key (i.e. hashcode)
 *                return the encrypted data block
 * decryptContent - given an ecrypted data block and a key, retrieve the
 *                useful information by decryption
 * @author Christian Grothoff
 * @author Ioana Patrascu
 * @file primitives/sessionkey.c
 **/

#include "config.h"
#include "gnettypes.h"

#define INITVALUE "GNUnet!!"

/**
 * Create a new SessionKey (for Blowfish)
 **/
void makeSessionkey(SESSIONKEY * key) {
  int i;
  for (i=0;i<SESSIONKEY_LEN;i++)
    key->key[i] = rand();
}

/**
 * Encrypt a block with the public key of another
 * host that uses the same cyper.
 * @param block the block to encrypt
 * @param len the size of the block
 * @param sessionkey the key used to encrypt
 * @param result the output parameter in which to store the encrypted result
 * @returns the size of the encrypted block, -1 for errors
 **/
int encryptBlock(void * block, 
		 BLOCK_LENGTH len,
		 SESSIONKEY * sessionkey,
		 BLOWFISHEncryptedData * result) {
  int outlen = 0;
  unsigned char iv[] = INITVALUE;
  EVP_CIPHER_CTX ctx;
  
  if ((block == NULL) || (sessionkey == NULL)) 
    return -1;  
  /* compute result size by adding block-length, always padded */
  EVP_EncryptInit(&ctx, EVP_bf_cfb(), sessionkey->key, iv);
#if SSL_MICRO >= 6
  if (0 == EVP_EncryptUpdate(&ctx, 
			     &result->encoding[0], 
			     &outlen,
			     block, len)) {
#ifdef PRINT_WARNINGS
    print("Sessionkey.c-encryptBlock: EVP_EncryptUpdate failed!\n");
#endif
    return -1;
  }
#else
  EVP_EncryptUpdate(&ctx, &result->encoding[0], &outlen, block, len);
#endif
  len = outlen; /* save bytes written so far */
  outlen = 0;
#if SSL_MICRO >= 6
  if (0 == EVP_EncryptFinal(&ctx, &result->encoding[0], &outlen)) {
#ifdef PRINT_WARNINGS
    print("Sessionkey.c-encryptBlock: EVP_EncryptFinal failed!\n");
#endif
    return -1;
  }
#else
  EVP_EncryptFinal(&ctx, &result->encoding[0], &outlen);
#endif
  outlen += len; /* add both updates together */
  EVP_CIPHER_CTX_cleanup(&ctx);
  return outlen;
}

/**
 * Decrypt a given block with the sessionkey.
 * @param sessionkey the key used to decrypt
 * @param block the data to decrypt, encoded as returned by encrypt
 * @param size the size of the block to decrypt
 * @param result address to store the result at
 * @return -1 on failure, size of decrypted block on success
 **/
int decryptBlock(SESSIONKEY * sessionkey, 
		 BLOWFISHEncryptedData * block,
		 BLOCK_LENGTH size,
		 void * result) {
  int outlen = 0;
  unsigned char iv[] = INITVALUE;
  EVP_CIPHER_CTX ctx;

  /* use blowfish-cfb */
  EVP_DecryptInit(&ctx, EVP_bf_cfb(), sessionkey->key, iv);
#if SSL_MICRO >= 6
  if (0 == EVP_DecryptUpdate(&ctx, result, &outlen,
			     block->encoding, size)) {
#ifdef PRINT_WARNINGS
    print("Sessionkey.c-decryptBlock: EVP_DecryptUpdate failed!\n");
#endif
    return -1;
  }
#else
  EVP_DecryptUpdate(&ctx, result, &outlen, block->encoding, size);
#endif
  size = outlen;
  outlen = 0; 
#if SSL_MICRO >= 6
  if (0 == EVP_DecryptFinal(&ctx, &((unsigned char*)result)[size], &outlen)) {
#ifdef PRINT_WARNINGS
    print("Sessionkey.c-decryptBlock: EVP_DecryptFinal failed!\n");
#endif
    return -1;
  }
#else
  EVP_DecryptFinal(&ctx, &((unsigned char*)result)[size], &outlen);
#endif
  return size+outlen;
}

/**
 * Encrypts a given data block
 *
 * @param data represents the data block
 * @param hashcode represents the key concatenated with the initial
 *        value used in the alg
 * @param result where to store the result (encrypted block)
 * @returns OK on success, SYSERR on error
 **/
int encryptContent(CONTENT_Block * data,
		   HashCode160 * hashcode,
		   CONTENT_Block * result){
  unsigned char iv[BLOWFISH_BLOCK_LENGTH];            /* initial value */
  EVP_CIPHER_CTX ctx;           /* context used in the encryption alg */
  int tmplen;
  int outlen = 0;  		/* output length */

  if ((data == NULL) || (hashcode == NULL) || (result == NULL)) {
#ifdef PRINT_WARNINGS
    print("Aborting encrypt content: NULL in arguments.\n");
#endif
    return SYSERR;
  }
  /* get key and init value from the hash code */
  memcpy(iv, &(((char *)hashcode)[BF_KEYSIZE]), BLOWFISH_BLOCK_LENGTH/2);
  memcpy(&iv[BLOWFISH_BLOCK_LENGTH/2], &(((char *)hashcode)[BF_KEYSIZE]),
         BLOWFISH_BLOCK_LENGTH/2);
  /* encrypt with hashcode */
  EVP_EncryptInit(&ctx, EVP_bf_cfb(), (unsigned char*) hashcode, iv);
#if SSL_MICRO >= 6
  if (0 == EVP_EncryptUpdate(&ctx, 
			     &result->content[0],
			     &outlen, 
			     &data->content[0], 
			     CONTENT_SIZE)) {
#ifdef PRINT_WARNINGS
    print("encrypt.c-encryptBlock: EVP_EncryptUpdate failed!\n");
#endif
    return SYSERR;
  }
#else
  EVP_EncryptUpdate(&ctx, &result->content[0], &outlen, &data->content[0],
  CONTENT_SIZE);
#endif
  /* since our blocks are multiples of 64, this should be
     unnecessary. But it can't hurt either and will
     prevent bugs if we ever change to a size that is not
     a multiple of 64 bit (very unlikely :-). */
#if SSL_MICRO >= 6
  if (0 == EVP_EncryptFinal(&ctx, 
			    &result->content[outlen], 
			    &tmplen)) {
#ifdef PRINT_WARNINGS
    print("encrypt.c-encryptBlock: EVP_EncryptFinal failed!\n");
#endif
    return SYSERR;
  }
#else
  EVP_EncryptFinal(&ctx, &result->content[outlen], &tmplen);
#endif
  EVP_CIPHER_CTX_cleanup(&ctx);
  /* return encrypted block */
  return OK;
}

/**
 * Decrypts a given data block
 *
 * @param data represents the data block
 * @param hashcode represents the key concatenated with the initial
 *        value used in the alg
 * @param result where to store the result (encrypted block)
 * @returns OK on success, SYSERR on error
 **/
int decryptContent(CONTENT_Block * data,
		   HashCode160 * hashcode,
		   CONTENT_Block * result){
  unsigned char iv[BLOWFISH_BLOCK_LENGTH]; /* initial value */
  EVP_CIPHER_CTX ctx; /* context used in the encryption alg */
  int tmplen;
  int outlen = 0;      /* output length */

  if ((data == NULL) || (hashcode == NULL) || (result == NULL)) {
#ifdef PRINT_WARNINGS
    print("Aborting decrypt content: NULL in arguments.\n");
#endif
    return SYSERR;
  }
  /* get key and init value from the hash code */
  memcpy(iv, 
	 &(((char *)hashcode)[BF_KEYSIZE]), 
	 BLOWFISH_BLOCK_LENGTH/2);
  memcpy(&iv[BLOWFISH_BLOCK_LENGTH/2], 
	 &(((char *)hashcode)[BF_KEYSIZE]),
         BLOWFISH_BLOCK_LENGTH/2);
  /* encrypt with hashcode */
  EVP_DecryptInit(&ctx, 
		  EVP_bf_cfb(), 
		  (unsigned char*) hashcode, 
		  iv);
#if SSL_MICRO >= 6
  if (0 == EVP_DecryptUpdate(&ctx, 
			     &result->content[0],
			     &outlen, 
			     &data->content[0], 
			     CONTENT_SIZE)) {
#ifdef PRINT_WARNINGS
    print("symcipher.c-decryptBlock: EVP_DecryptUpdate failed!\n");
#endif
    return SYSERR;
  }
#else
  EVP_DecryptUpdate(&ctx, &result->content[0], &outlen, &data->content[0],
  CONTENT_SIZE);
#endif
  /* since our blocks are multiples of 64, this should be
     unnecessary. But it can't hurt either and will
     prevent bugs if we ever change to a size that is not
     a multiple of 64 bit (very unlikely :-). */
#if SSL_MICRO >= 6
  if (0 == EVP_DecryptFinal(&ctx, 
			    &result->content[outlen], 
			    &tmplen)) {
#ifdef PRINT_WARNINGS
    print("symcipher.c-decryptBlock: EVP_DecryptFinal failed!\n");
#endif
    return SYSERR;
  }
#else
  EVP_DecryptFinal(&ctx, &result->content[outlen], &tmplen);
#endif
  EVP_CIPHER_CTX_cleanup(&ctx);
  /* return decrypted block */
  return OK;
}


/* end of sessionkey.c */
