/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * This is the main file for the gtk+ client.
 * @file src/gtkui/main.c
 * @author Christian Grothoff
 **/

#include "config.h"
#include <gtk/gtk.h>
#include <glib.h>
#include <unistd.h>
#include <sys/types.h>

#include "util/getopt.h"
#include "util/storage.h"
#include "gtkui/search.h"
#include "textui/requestmanager.h"
#include "textui/searchutil.h"
#include "configuration.h"

static GtkWidget * main_window_input_line;

/**
 * This method is called whenever the user clicks the
 * search button of the main window. data is NULL.
 **/
static void search(GtkWidget * widget,
		   gpointer data) {
  gchar * searchString;

  searchString 
    = gtk_entry_get_text(GTK_ENTRY(main_window_input_line));
  displaySearchWindow(searchString);
}

/* callback: window close: close the window */
static gint delete_event(GtkWidget * widget,
			 GdkEvent * event,
			 gpointer data) {
  return FALSE;
}

/* exit the application */
static void destroy(GtkWidget * widget,
		    gpointer data) {
  gtk_main_quit();
}

static void makeMainWindow() {
  GtkWidget * window;
  GtkWidget * button;
  GtkWidget * box;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
 
  gtk_window_set_title(GTK_WINDOW(window), "GNUnet: gtk+ GUI");
  box = gtk_hbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), box);
  gtk_signal_connect (GTK_OBJECT (window), 
		      "delete_event",
		      GTK_SIGNAL_FUNC (delete_event), NULL);
  gtk_signal_connect (GTK_OBJECT (window), 
		      "destroy",
		      GTK_SIGNAL_FUNC (destroy), NULL);
  gtk_container_set_border_width (GTK_CONTAINER (window), 10);
  /* search entry */
  main_window_input_line = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(box), 
		     main_window_input_line, TRUE, TRUE, 0);
  gtk_widget_show(main_window_input_line);
  /* search button */
  button = gtk_button_new_with_label ("Search");
  gtk_signal_connect (GTK_OBJECT (button), 
		      "clicked",
		      GTK_SIGNAL_FUNC (search), NULL);
  gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 0);
  gtk_widget_show(button);
  /* main window */
  gtk_widget_show(box);
  gtk_widget_show (window);
}

/**
 * Print a list of the options we offer.
 **/
static void printhelp() {
  printf("USAGE: gnunet-gtk [OPTIONS]\n");
  printf("-h, --help                    : print this page\n");
  printf("-c FILENAME, --config=FILENAME: load config file (defaults: %s)\n",
	 DEFAULT_CONFIG_FILE);
  printf("-v, --version                 : print the version number\n");
}


/**
 * Perform option parsing from the command line. 
 **/
static int parseCommandLine(int argc, 
		     char * argv[],
		     char ** configFile) {
  int shouldexit=0;
  int c;
  while (1) {
    int option_index=0;
    static struct option long_options[] = {
      { "config",  1, 0, 'c' },
      { "version", 0, 0, 'v' },
      { "help",    0, 0, 'h' },
      { 0,0,0,0 }
    };
    
    LOOPPRINT("parseCommandLine"); 
    c = getopt_long(argc,
		    argv, 
		    "vhc:", 
		    long_options, 
		    &option_index);
    
    if (c == -1) 
      break;  // No more flags to process
    
    switch(c) {
    case 'c': 
      *configFile = optarg; 
      print("Configfile specified: %s.\n",*configFile);
      break;
    case 'v': 
      print("GNUnet %s (%s) gnunet-gtk %s\n",
	    GNUNET_VERSION, GNUNET_BRANCH,
	    GNUNET_GTKUI_VERSION);
      shouldexit=1;
      break;
    case 'h': 
      printhelp(); 
      shouldexit=1;
      break;
    default:
      print("Unknown option %c. Aborting.\nUse --help to get a list of options.\n",
	    c);
      shouldexit=1;    
    } /* end of parsing commandline */
  }
  if (optind < argc) {
    print("Invalid arguments: ");
    while (optind < argc)
      print("%s ", argv[optind++]);
    print("\nExiting.\n");
    return 1;
  }
  return shouldexit;
}


int main(int argc,
         char * argv[]) {
  char * configfile;

  configfile = DEFAULT_CONFIG_FILE;
  parseCommandLine(argc, argv, &configfile);
  /* initialization */
  readConfig(configfile);
  initCron();
  initTCPIO();
  initKnownhosts();
  initSearchUtil("localhost",
		 getGNUnetPort());
  
  initRequestManager();
  startCron();

  g_thread_init(NULL);
  /*  gdk_threads_init(); */
  gtk_init(&argc, &argv);

  makeMainWindow();
  gdk_threads_enter();
  gtk_main ();
  gdk_threads_leave();
 
  return 0;
}
/* end of main.c */
