/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * This is the window displaying search results for
 * the gtk+ client.
 * @file src/gtkui/search.c
 * @author Christian Grothoff
 **/

#include "config.h"
#include <gtk/gtk.h>
#include <glib.h>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/types.h>
#include <pthread.h>
#include <netinet/in.h>

#include "util/storage.h"
#include "textui/searchutil.h"
#include "gtkui/search.h"
#include "gtkui/download.h"
#include "gtkui/saveas.h"

typedef struct {
  GtkWidget * search_window_result_box;
  int selectedRow;
  pthread_t searchThread;
  pthread_t receiveThread;
} ListModel;

/**
 * This method is called whenever the user clicks the
 * search button of the main window. data is NULL.
 **/
static void download(GtkWidget * widget,
		     gpointer data) {  
  ListModel * listModel;
  gchar *text[6];
  gint row;
  int i;

  listModel = data;
  row = listModel->selectedRow;
  if (row == -1) {
    print("Nothing selected!\n");
    return;
  }

  for (i=0;i<6;i++)
    gtk_clist_get_text(GTK_CLIST(listModel->search_window_result_box),
		       row, i, &text[i]);
  /*  print("Downloading %s - %s - %s - %s - %s\n",text[0],text[1],text[2],text[3],
      text[4]); */
  openSaveAs(text[0],text[1],text[2],text[3],text[4]);
  /* mind that here text** becomes freed, so we MUST copy before! */
  gtk_clist_remove(GTK_CLIST(listModel->search_window_result_box),
		   listModel->selectedRow);
  listModel->selectedRow = -1; 
}

static void select_row_callback(GtkWidget *widget,
				gint row,
				gint column,
				GdkEventButton *event,
				gpointer data) {
  ListModel * listModel;

  listModel = data;
  listModel->selectedRow = row;
}

static void unselect_row_callback(GtkWidget *widget,
				  gint row,
				  gint column,
				  GdkEventButton *event,
				  gpointer data) {
  ListModel * listModel;

  listModel = data;
  listModel->selectedRow = -1;
}

/**
 * Display results!
 **/
static void displayResult(RootNode * rootNode,
			  ListModel * model) {
  GtkWidget * search_window_result_box;
  gchar ** results;

  rootNode->description[MAX_DESC_LEN-1] = 0;
  rootNode->filename[MAX_FILENAME_LEN-1] = 0;
  rootNode->mimetype[MAX_MIMETYPE_LEN-1] = 0;

  search_window_result_box = model->search_window_result_box;
  results = xmalloc(6*sizeof(gchar *),
		    "displayResult: resultRow");
  results[0] = strdup(rootNode->description);
  results[1] = xmalloc(32,
		       "displayResult: filelength string");
  sprintf(results[1],"%d", 
	  ntohl(rootNode->file_length));
  results[2] = xmalloc(32,
		       "displayResult: crc string");
  sprintf(results[2],"%d", 
	  ntohl(rootNode->crc));
  results[3] = xmalloc(sizeof(HexName),
		       "displayResult: hexcode");
  hash2hex(&rootNode->hashCode,
	   (HexName*) results[3]);
  results[4] = strdup(rootNode->filename);
  results[5] = strdup(rootNode->mimetype);
  gdk_threads_enter();
  gtk_clist_append(GTK_CLIST(search_window_result_box),
		   results);
  gdk_threads_leave();
}

typedef struct {
  int num_keywords;
  TCP_Query_Request * messages;
  Semaphore * lock;
} _sendQueryArgument_;

/**
 * Main method of the sendQuery-threads..
 **/
static void sendQueries_(_sendQueryArgument_ * args) {
  int timeout = 0xFFFFFF;
  TCP_Query_Request * msg;

  msg = xmalloc(sizeof(TCP_Query_Request)*args->num_keywords,
		"sendQueries: message");
  memcpy(msg,
	 args->messages,
	 sizeof(TCP_Query_Request)*args->num_keywords);
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
  pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
  semaphore_up(args->lock);
  sendQueries(args->num_keywords,
	      timeout,
	      msg);
  xfree(msg,"sendQueries_: message"); /* fixme: memory leak on async cancel,
					register cleanup handler? */
  xfree(args,"sendQueries_: args");
}

typedef struct {
  int num_keywords;
  HashCode160 * keys;
  TCP_Query_Request * messages;
  void * displayResult;
  ListModel * model;
  Semaphore * lock;
} _receiveResultArgs_;

/**
 * Main method of the receiveResults-threads..
 **/
static void receiveResults_(_receiveResultArgs_ * args) {
  TCP_Query_Request * msg;

  msg = xmalloc(sizeof(TCP_Query_Request)*args->num_keywords,
		"receiveResults: messages");
  memcpy(msg,
	 args->messages,
	 sizeof(TCP_Query_Request)*args->num_keywords);

  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
  pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
  semaphore_up(args->lock);
  receiveResults(args->num_keywords,
		 args->keys,
		 msg,
		 args->displayResult,
		 args->model);
  xfree(msg,"receiveResults_: messages"); /* fixme: memory leak on async cancel,
					    register cleanup handler? */
  xfree(args,"receiveResults_: args");
}

/**
 * The main method of the search-thread.
 **/
void startSearchThreads(char * searchString,
			ListModel * model) {
  HashCode160 * keys;
  char ** keywords;
  int num_keywords;
  TCP_Query_Request * messages;
  _sendQueryArgument_ * sendArgs;
  _receiveResultArgs_ * receiveArgs;
  Semaphore * lock;
  int pos;
  int next;
  int idx;

  sendArgs = xmalloc(sizeof(_sendQueryArgument_),
		     "startSearchThreads: sargs");
  receiveArgs = xmalloc(sizeof(_receiveResultArgs_),
			"startSearchThreads: rargs");
  /*  fprintf(stderr,"Searching for %s!\n",
      searchString);*/

  num_keywords = 0;
  for (pos=strlen(searchString)-1;pos>=0;pos--)
    if (searchString[pos] == ' ') {
      num_keywords++;
      while ( (pos >= 0) &&
	      (searchString[pos] == ' ') ) 
	pos--;
      pos++;
    }
  if (searchString[strlen(searchString)-1] != ' ')
    num_keywords++;
  keywords = xmalloc(num_keywords * sizeof(char*),
		     "startSearchThreads: keywords");
  next=0;
  pos =0;
  idx =0;
  while (pos < strlen(searchString)) {
    while ( (searchString[next] != ' ') &&
	    (next < strlen(searchString)) )
      next++;
    if (next == pos) {
      next++;
      pos = next;
      continue; /* two spaces */
    }
    keywords[idx] = xmalloc(next-pos+1,
			    "startSearchThreads: allocating space for keyword");
    memcpy(keywords[idx],
	   &searchString[pos],
	   next-pos);
    keywords[idx][next-pos] = '\0';
    next++;
    idx++;
    pos = next;
  }

  num_keywords = parseKeywords(num_keywords,
			       keywords,
			       &keys);
  if (num_keywords <= 0) {
    return;
  }

  buildMessages(num_keywords,
		keys,
		&messages);
  lock = new_semaphore(0);
  sendArgs->num_keywords = num_keywords;
  sendArgs->messages = messages;
  sendArgs->lock = lock;
  pthread_create(&model->searchThread,
		 NULL,
		 (void * (*)(void *)) sendQueries_,
		 sendArgs);
  /*  fprintf(stderr,
	  "calling receiveResults with num %d\n",
	  num_keywords);*/
  receiveArgs->num_keywords = num_keywords;
  receiveArgs->keys =  keys;
  receiveArgs->messages=  messages;
  receiveArgs->displayResult = displayResult;
  receiveArgs->model =  model;
  receiveArgs->lock = lock;
  pthread_create(&model->receiveThread,
		 NULL,
		 (void * (*)(void *)) receiveResults_,
		 receiveArgs);
  /* wait for the memcpy of both threads */
  semaphore_down(lock);
  semaphore_down(lock);
  xfree(messages,"startSearchThreads: keywords");
}



/* callback: window close: close the window */
static gint delete_event(GtkWidget * widget,
			 GdkEvent * event,
			 gpointer data) {
  return FALSE;
}

/* close the window, stop the search thread */
static void destroy(GtkWidget * widget,
		    gpointer data) {
  ListModel * model;

  model = (ListModel*) data;
  pthread_cancel(model->searchThread);
  pthread_cancel(model->receiveThread);
  xfree(model,"destroy window: model");
}

void displaySearchWindow(gchar * title) {
  GtkWidget * window;
  GtkWidget * button;
  GtkWidget * box;
  GtkWidget * search_window_result_box;
  ListModel * model;
  static gchar * descriptions[] = {
    "Description",
    "Size",
    "CRC",
    "Key",
    "Filename",
    "Mimetype"};

  model = (ListModel*) xmalloc(sizeof(ListModel),
			       "displaySearchWindow: listModel");
  model->selectedRow = -1;
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), title);

  gtk_widget_set_usize(GTK_WIDGET(window), 800, 300);
  box = gtk_vbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), box);
  gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		      GTK_SIGNAL_FUNC (delete_event), NULL);
  gtk_container_set_border_width (GTK_CONTAINER (window), 10);
  /* result set box */

  search_window_result_box 
    = gtk_clist_new_with_titles(6, descriptions);
  model->search_window_result_box = search_window_result_box;

  gtk_clist_set_selection_mode
    (GTK_CLIST(search_window_result_box),
     GTK_SELECTION_MULTIPLE);
  /* for now: passive titles; we may want to sort by
     size later, though */
  gtk_clist_column_titles_passive
    (GTK_CLIST(search_window_result_box));

  /* description left, size right justification */
  gtk_clist_set_column_justification
    (GTK_CLIST(search_window_result_box),
     0,
     GTK_JUSTIFY_LEFT);
  gtk_clist_set_column_justification
    (GTK_CLIST(search_window_result_box),
     1,
     GTK_JUSTIFY_RIGHT);
  gtk_signal_connect(GTK_OBJECT(search_window_result_box),
		     "unselect_row",
		     GTK_SIGNAL_FUNC(unselect_row_callback),
		     (gpointer) model);
  gtk_signal_connect(GTK_OBJECT(search_window_result_box),
		     "select_row",
		     GTK_SIGNAL_FUNC(select_row_callback),
		     (gpointer) model);

    /* What however is important, is that we set the column widths as
     * they will never be right otherwise. Note that the columns are
     * numbered from 0 and up (to 5 in this case).
     */
  gtk_clist_set_column_width (GTK_CLIST(search_window_result_box), 
			      0, 500);
  gtk_clist_set_column_width (GTK_CLIST(search_window_result_box), 
			      1, 60);
  gtk_clist_set_column_width (GTK_CLIST(search_window_result_box), 
			      2, 60);
  gtk_clist_set_column_width (GTK_CLIST(search_window_result_box), 
			      3, 40);
  gtk_clist_set_column_width (GTK_CLIST(search_window_result_box), 
			      4, 200);
  gtk_clist_set_column_width (GTK_CLIST(search_window_result_box), 
			      5, 200);


  gtk_box_pack_start(GTK_BOX(box), 
		     search_window_result_box, 
		     TRUE, TRUE, 0);
  gtk_widget_show(search_window_result_box);  

  /* download button */
  button = gtk_button_new_with_label("Download");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (download), 
		      model);
  gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 0);
  gtk_widget_show(button);
  /* display window */
  gtk_widget_show(box);
  startSearchThreads(title,
		     model);
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (destroy), 
		      model);
  gtk_widget_show (window);
}

/* end of search.c */
