/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Values that define the behaviour of GNUnet that 
 * could be changed without doing much harm. 
 * Users can change these values to fine-tune the
 * behavior of their node. Some of these
 * values can not be changed without
 * making the node incompatible with other nodes.
 * Furthermore, most people don't need to worry,
 * and it would be overkill to put them into
 * gnunet.conf. 
 * @file include/gnettypes.h
 * @author Christian Grothoff
 * @author Tzvetan Horozov
 **/

#ifndef CONFIG_H
#define CONFIG_H

#ifdef __cplusplus
extern "C" {
#endif

#ifdef HAVE_CONFIG_H
#include "autodefs.h"
#endif

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif

/* indirection table size, defaults to 512 entries,
   reduce if you have very little memory, enlarge if you
   start to overflow often. MUST BE A POWER OF 2!!!*/
#define INDIRECTION_TABLE_SIZE 512

/**
 * Number of hosts that can wait for the same query at
 * any given point. Memory cost is
 * INDIRECTION_TABLE_SIZE * 20 * THIS_NUMBER, so it
 * is usually not really costly. Default: 16.
 **/
#define MAX_HOSTS_PER_QUERY 16


/* by which amount do we decrement the TTL for
   simple forwarding / indirection of the query; in seconds.
   Set somewhat in accordance to your network latency (above
   the time it'll take you to send a packet and get a reply). */
#define TTL_DECREMENT 5

/* How many client connections can a single node have
   waiting FOR THE SAME QUERY? 2 should be enough unless
   there are really multiple concurrent users looking for
   the same query */
#define MAX_TCP_PER_QUERY 2

/* Number of Entries in the on-disk database. Should be larger than the
   number of 1k blocks shared and must be a power of 2. The database-file
   will be 32bytes * CACHE_HASH_TABLE_SIZE. If the table is sparse, the
   database-file may occupy less actual space on the drive. 
   Default is 8388608, which corresponds to a 256 MB database file.
   This is sufficient to share about 8 GB of data (yet, if you get
   closer to this number, efficiency degrades and you may want to
   double the size of this table). You can increase this number up to
   2^31 (= 2 GB database, or 64 GB shared files). If you dedicate more
   space to GNUnet, we suggest running multiple nodes... */
#define CACHE_HASH_TABLE_SIZE 8388608  

/* THE OPTIONS BELOW SHOULD BE TREATED WITH EXTREME CAUTION. 
   THEY *WILL* BREAK COMPATIBILITY WITH ALL OTHER NODES IN THE
   NETWORK. THEY ARE HERE SUCH THAT DEVELOPERS CAN CHANGE
   CONTROVERSIAL SETTINGS. DON'T TOUCH IF YOU DON'T KNOW WHAT
   YOU'RE DOING. PLEASE DO NOT CONNECT TO THE MAIN GNET-NETWORK
   WITH THESE SETTINGS - OTHER NODES WILL IGNORE YOU ANYWAY... */


/**
 * The (maximum) size of each message. More is not allowed!
 **/
#define MESSAGE_SIZE 1472

/* max size of a packet - tuned for UDP/Ethernet Framesize (10 MBit) */
#define BUFSIZE (MESSAGE_SIZE-sizeof(MessagePack))

/* minimum size for an immediate flushing of the buffer;
   set to the largest number where still another query
   could be added to the buffer */
#define THRESHOLD (BUFSIZE - sizeof(QUERY_Message))

/**
 * Maximum number of replies for the same request/keyword.
 **/
#define MAX_REPLIES 16

/**
 * How many requests do we have pending (max)
 * Must be > 50 to make sure we trigger flushing buffers
 * at the node. Lower values should increase anonymity
 * and decrease performance (for the individual node
 * that makes this change).
 **/
#define MAX_REQUESTS 128

/**
 * How many UDP threads can we start (max count). 
 * Q: What is the system limit?
 **/
#define THREAD_COUNT 4

/**
 * Number of threads to start at the beginning
 **/
#define THREAD_MIN 4

/**
 * How many incoming packages do we have in the buffer
 * (max.). Must be >= THREAD_MIN to make sense.
 **/
#define QUEUE_LENGTH 8

/**
 * Default name of the configuration file.
 **/
#define DEFAULT_CONFIG_FILE "~/.gnunet/gnunet.conf"


/**
 * Version numbers...
 **/
#define GNUNET_VERSION "0.4.2"
#define GNUNET_BRANCH "beta"
#define GNUNET_TEXTUI_VERSION "2.0.2"
#define GNUNET_GTKUI_VERSION "0.9.6"

#ifdef __cplusplus
}
#endif

#endif
/* end of config.h */
