/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * GNUnet Debugging module. Configure here for which components you
 * want extra debugging output.
 *
 * @author Christian Grothoff
 * @file include/debugging.h
 **/

#ifndef DEBUGGING_H
#define DEBUGGING_H

#include "config.h"
#include "gnettypes.h"

/**
 * Where to print errors and warnings to?
 **/
#define PRINTTO stderr

/**
 * Is this a beta-release?
 **/
#define GNUNET_BETA 1  

#define LOOPPRINT(a)
/*#define LOOPPRINT(a) fprintf(stderr,"%s\n",a)   */

/* ********* things that we want on even
   for production releases (usually) **** */
/**
 * Print all warnings about detected errors in incoming
 * data, e.g. malformed packets (CRC).
 **/
#define PRINT_WARNINGS 1

/**
 * Should we print some status information on what's
 * going on during startup / downloads?
 **/
#define PRINT_STATUS 1


/* ***************** commonly turned-on checks
   for debugging and beta-versions *********** */

#if GNUNET_BETA

/**
 * Print entry-exit messages for each cron job.
 **/
#define PRINT_CRON 0

/**
 * Perform some additional run-time checks that
 * may be costly.
 **/
#define EXTRA_CHECKS 1 

/**
 * Should ASSERTS be checked?
 **/
#define CHECK_ASSERTS 1 

#else

/**
 * Perform some additional run-time checks that
 * may be costly.
 **/
#define EXTRA_CHECKS 0

/**
 * Should ASSERTS be checked?
 **/
#define CHECK_ASSERTS 0

#endif

/* ********* common functionality-debugging *********** */

/**
 * Print a dot every second to show that cron is live?
 **/
#define PRINT_CRONDOT 0

/**
 * Should we debug the UDP *networking* code?
 **/
#define PRINT_UDP 0

/**
 * Print all calls to xmalloc.
 **/
#define PRINT_MALLOC 0

/**
 * Print all calls to xfree.
 **/
#define PRINT_FREE 0

/**
 * Print all configuration actions.
 **/
#define PRINT_CONFIGURATION 0

/**
 * Print information about query processing.
 **/
#define PRINT_QUERY 0

/**
 * Print PING/PONG messages
 **/
#define PRINT_PING 0

/**
 * Debug connection module.
 **/
#define PRINT_CONNECTION 0

/**
 * Debug sessionkey events.
 **/
#define PRINT_SKEY 0

/**
 * Print information about HELO activities.
 **/
#define PRINT_HELO 0

/**
 * Print routing information.
 **/
#define PRINT_ROUTING 0

/**
 * Print content events.
 **/
#define PRINT_CONTENT 0

/**
 * Print policy decisions.
 **/
#define PRINT_POLICY 0

/**
 * Print trust related changes
 **/
#define PRINT_TRUST 0

/**
 * Print all important TCP actions.
 **/
#define PRINT_TCP 0

/**
 * Debug the request manager (textui).
 **/
#define DEBUG_RM 0

/**
 * Run without CRC checking
 **/
#define DISABLE_CRC_CHECK 0 


/* ********************** Types ***************** */

/* IP:Port as a string */
typedef struct {
  char ipport[4*4+5+1]; 
} IPPortString;

/* IP as a string */
typedef struct {
  char ip[4*4+1]; 
} IPString;

/* Sessionkey as a string */
typedef struct {
  char skey[5*SESSIONKEY_LEN+1]; 
} SKEYString;



/**
 * errexit - print an error message and exit
 * @param format the string describing the error message
 **/
void errexit(const char *format, ...);

/**
 * errexit - print an error message and exit
 * @param format the string describing the error message
 **/
void print(const char *format, ...);

/**
 * Display the contents of an incoming UDP message.
 **/
void showMessageEncrypted(MessagePack * msg);

/**
 * Display the contents of an incoming UDP message.
 **/
void showMessageDecrypted(MessagePack * msg);

/**
 * Print the connection address (IP and Port) encoded in the given
 * HostAddress to stderr.
 * @param conInfo the HostAddress to print.
 **/
void printAddress(HostAddress * conInfo,
		  IPPortString * result);
 
/**
 * Print the connection address (IP) encoded in the given
 * integer.
 * @param theIP the encoded IP
 **/
void printIP(int theIp,
	     IPString * result);

void printSessionKey(SESSIONKEY * key,
		     SKEYString * result);

/**
 * Assert something. Display message if it fails.
 **/
void assert(int condition,
	    char * message);

/* ************************ active DEBUG ************** */

/**
 * The following is a testing routine that will execute sending
 * of a query to a bunch of other hosts
 */
void DEBUG_broadcastQuery();

/**
 * Print the given packet in a human-readable form
 * (especially giving the parts of the packet).
 **/
void dumpPacket(unsigned char * message,
		BLOCK_LENGTH len);





#endif
/* end of debugging.h */
