/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * The most important types in GNUnet
 * @file include/gnettypes.h
 * @author Christian Grothoff
 * @author Tzvetan Horozov
 * @author Ioana Patrascu
 **/

#ifndef GNETTYPES_H
#define GNETTYPES_H

#include "config.h"
#include <openssl/conf.h>
#include <openssl/err.h>
#include <openssl/rsa.h>
#include <openssl/evp.h>
#include <openssl/bn.h>

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>

#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "util/crc32.h"
#include "util/semaphore.h"
#include "util/hostkey.h"
#include "util/storage.h"
#include "util/checksum.h"
#include "util/tcpio.h"

/* *********** CONSTANTS AND TYPES FOR THE ENCRYPTION BUSINESS *********** */


/** length of the sessionkey in bytes
   (128 BIT sessionkey) */
#define SESSIONKEY_LEN (128/8)

/** size of blowfish key in bytes */
#define BF_KEYSIZE 16

/** this BLOCK_LENGTH is 64 bit, blowfish */
#define BLOWFISH_BLOCK_LENGTH 8 

/* *********************** Strings - typed ************************** */

typedef unsigned char * IPAddress;

/* **************** Network protocol business ************************ */

/**
 * Constants defining the UDP (gnunetd-gnunetd) subprotocols (0 <= x <= 65535)
 **/

/** announcement of public key */
#define GNET_PROTO_HELO 0
/** session key exchange, session key is encrypted with hostkey */
#define GNET_PROTO_SKEY 1
/** PING **/
#define GNET_PROTO_PING 2
/** PONG (response to PING) **/
#define GNET_PROTO_PONG 3

/** query for content with Triple-hash */
#define GNET_PROTO_3QUERY 4
/** receive content */
#define GNET_PROTO_CONTENT 5

/** timestamp (until when is the packet valid) **/
#define GNET_PROTO_TIMESTAMP 6
/** sequence number (discard packet if sequence number
    is not higher than previously received number) */
#define GNET_PROTO_SEQUENCE 7
/** noise, used to fill packets to sizes >1k **/
#define GNET_PROTO_NOISE 8
/** termination of connection (other host is nice
    and tells us, there is NO requirement to do so!) **/
#define GNET_PROTO_HANGUP 9

/* number of defined UDP protocol numbers */
#define GNET_PROTO_MAX 10
/**
 * Constants defining the TCP subprotocols (0<= x <= 65535)
 * Used for gnunetd-client communication.
 **/
/* client to gnunetd: send query */
#define TCP_REQUEST_QUERY 0
/* gnunetd to client: here is your answer (content) */
#define TCP_REPLY_CONTENT 1
/* client to gnunetd: insert content (no index) */
#define TCP_REQUEST_INSERT 2
/* client to gnunetd: insert indexed content */
#define TCP_REQUEST_INDEX 3
/* client to gnunetd: add a file to the file list
   (gnunetd must send back the index!) */
#define TCP_REQUEST_LIST_FILE 4
/* gnunetd to client: this is the file-index */
#define TCP_REPLY_FILE_INDEX 5
/* client to gnunetd: request statistics */
#define TCP_REQUEST_STATISTICS 6
/* gnunetd to client: statistics */
#define TCP_REPLY_STATISTICS 7


/* number of defined TCP messages */
#define TCP_MAX 8

/* maximum size of a TCP message (must be bigger
   than the largest message that is actually exchanged). Can
   be increased between GNUnet versions if needed! */
#define TCP_MAXIMUM_MESSAGE_SIZE 1500


/** protocol number of TCP. Do NEVER change! */
#define TCP_PROTOCOL_NUMBER 6
/** protocol number of UDP. Do NEVER change! */
#define UDP_PROTOCOL_NUMBER 17

/** bit set in the packet size to indicate encryption */
#define ENCRYPTED_FLAG 0x8000
/** bit NOT set to define plaintext ** */
#define PLAINTEXT_FLAG 0x0000


/* type for symmetrically encrypted data */
typedef struct {
  unsigned char encoding[0] __attribute__ ((packed));
} BLOWFISHEncryptedData;

/**
 * Network-identity of a host (IP + Port + Protocol).
 * Currently, only UDP (17) is supported for the
 * protocol, but this naturally extends to other protocols
 * (6 will be the number for TCP, I'm considering
 * 25 for smtp, 80 for http, etc. -- and yes, I know that
 * portnumbers and protocol numbers are different beasts,
 * but we just need a number for GNUnet, and an illustrative
 * one is better than a random number... :-)
 **/
typedef struct {
  /* claimed IP of the sender, network byte order */  
  struct in_addr senderIP __attribute__ ((packed));
  /* claimed port of the sender, network byte order */
  unsigned short senderPort __attribute__ ((packed)); 
  /* protocol (UDP, TCP, etc), network byte order */
  unsigned short protocol __attribute__ ((packed)); 
} HostAddress __attribute__ ((packed));

/**
 * The identity of the host (basically the RIPE160 hashcode of
 * it's public key).
 **/
typedef struct {
  HashCode160 hashPubKey;
} HostIdentity __attribute__ ((packed));

/**
 * Type of the content index file entries.
 **/
typedef struct {
  /** the double-hash of this entry */
  HashCode160 doubleHash __attribute__ ((packed));
  /** the current rating of this content (network byte order)*/
  unsigned int importance __attribute__ ((packed));
  /** the filename, 0  for entry not used, -1 for data/content shared, > 0 for on-demand encoding (network byte order)*/
  unsigned int fileNameIndex __attribute__ ((packed)); 
  /** the offset in the file, offset for on-demand-encoding (network byte order) */
  unsigned int fileOffset __attribute__ ((packed));  
} ContentIndex __attribute__ ((packed));

/** BLOWFISHEncryptedData with enough space for anything we use */
typedef struct {
  unsigned char encoding[MESSAGE_SIZE] __attribute__ ((packed));
} BLOWFISHEncryptedDataMax;


/* *********** STRUCTS for the UDP (gnunetd-gnunetd) protocol *********** */

/**
 * Message-subpart header
 **/
typedef struct {
  /** size of this MESSAGE_PART (network byte order) */
  BLOCK_LENGTH size;
  /** type of the request, GNET_PROTO_XXX (network byte order) */
  unsigned short requestType; 
  /* this is followed by a requestType specific data block,
     consisting of size bytes. */
} MessagePart;

/**
 * Message-Packet header. 
 **/
typedef struct {
  /** size of the message, in bytes, including this header; max 65536-header (network byte order) */
  BLOCK_LENGTH size __attribute__ ((packed));
  /** reserved, must be 0 (network byte order) **/
  unsigned short reserved;
  /** CRC checksum of the packet  (network byte order)*/ 
  int checkSum __attribute__ ((packed));
  /** Where did the packet come from (IP, port) */  
  HostAddress sender __attribute__ ((packed));
  /** What is the identity of the sender (hash of public key) */
  HostIdentity senderIdentity __attribute__ ((packed));
  
  /* redundant: here is the size of all MESSAGE_PARTs together that follow (type: BLOCK_LENGTH) */
  /* this struct is followed by MESSAGE_PARTs - until size is reached */ 
  /* there is no "end of message" */
  MessagePart parts[0] __attribute__ ((packed));
} MessagePack __attribute__ ((packed));

/**
 * HELO.
 * A HELO body contains the current HostAddress,
 * the host identity (hash), the time how long the
 * HostAddress is valid, a signature signing the
 * information above and the public key of the host.
 * The hash of the public key must match the host
 * identity.
 **/
typedef struct {
  /** Whose identity follows? No, this is NOT a duplicate
      as a node may send us the identity of ANOTHER node! */
  HostAddress sender __attribute__ ((packed)); 
  HostIdentity senderIdentity __attribute__ ((packed)); 
  /** time this address expires  (network byte order) */ 
  time_t expirationTime __attribute__ ((packed));
  /** The signature **/
  Signature signature __attribute__ ((packed));
  /** The public key */
  PublicKey publicKey __attribute__ ((packed)); 
} HELO_Body __attribute__ ((packed));

/**
 * The complete HELO message (still only a part of a packet).
 **/
typedef struct {
  MessagePart header __attribute__ ((packed));
  HELO_Body body __attribute__ ((packed));
} HELO_Message __attribute__ ((packed));

/**
 * Format of a timestamp-message. The rest of the body
 * is only valid if the timestamp is greater than the
 * current time (in seconds after 1970...).
 * Used against replay attacks!
 **/
typedef struct {
  MessagePart header __attribute__ ((packed));
  /* timestamp  (network byte order)*/
  time_t timeStamp __attribute__ ((packed));
} TIMESTAMP_Message;

/**
 * Sequence number. If the sequence number is lower than
 * a previous number, the rest of the packet should be
 * ignored (replay). This will of course break if UDP packets
 * arrive out-of-order, but this is rare and we're best-effort.
 * This is used to defend against replay-attacks.
 **/
typedef struct {
  MessagePart header __attribute__ ((packed));
  /* sequence number, in network byte order */
  unsigned int sequenceNumber __attribute__ ((packed));
} SEQUENCE_Message;

/**
 * The other side has decided to terminate the connection.
 * This message MAY be send if the other node decides to 
 * be nice. It is not required. Mind that the message
 * contains for which host the termination is, such
 * that we don't hang up the wrong connection...
 * A node can also choose to ignore the HANGUP message,
 * though this is probably not going to help that node.
 * This message is used to prevent sending data to 
 * connections that were closed on the other side (can
 * happen anyway, so this is just an optimization between
 * well-behaved, non-malicious nodes that like each other).
 **/
typedef struct {
  MessagePart header __attribute__ ((packed));
  HostIdentity sender __attribute__ ((packed));
} HANGUP_Message;

/**
 * Ping message (test if address actually corresponds to
 * the advertised GNUnet host. The receiver responds with
 * exactly the same message, except that it is now a pong.
 * This message can be send in plaintext and without padding
 * and typically does make little sense (except keepalive)
 * for an encrypted (authenticated) tunnel. 
 * <br>
 * There is also no proof that the other side actually
 * has the acclaimed identity, the only thing that is
 * proved is that the other side can be reached via
 * the underlying protocol and that it is a GNUnet node.
 * <br>
 * The challenge prevents an inept adversary from sending
 * us a HELO and then an arbitrary PONG reply (adversary
 * must at least be able to sniff our outbound traffic).
 **/
typedef struct {
  MessagePart header __attribute__ ((packed));
  HostIdentity receiver __attribute__ ((packed));
  int challenge;
} PINGPONG_Message;

/**
 * The body of a sessionkey-message.
 **/
typedef struct {
  /** time when this key was created  (network byte order) */ 
  time_t creationTime __attribute__ ((packed)); 
  /** the encrypted session key */ 
  RSAEncryptedData key __attribute__ ((packed)); 
  /** Signature of the stuff above */
  Signature signature __attribute__ ((packed));
} SKEY_Body __attribute__ ((packed));

/**
 * Session key exchange.
 * The header is followed by an inlined SKS.
 **/
typedef struct {
  MessagePart header __attribute__ ((packed)); 
  SKEY_Body body __attribute__ ((packed));
} SKEY_Message __attribute__ ((packed));

/**
 * Request for content
 * The header is followed by the n hashcodes, each preceeded
 * by its length (as usual)
 **/
typedef struct {
  MessagePart header __attribute__ ((packed)); 
  /** how important is this request (network byte order) */
  int priority __attribute__ ((packed));         
  /** time to live in units of 5s (network byte order)  */
  int ttl __attribute__ ((packed));              
  /** to whom to return results? */
  HostIdentity returnTo __attribute__ ((packed));
  /** the file we're looking for */
  HashCode160 query __attribute__ ((packed)); 
} QUERY_Message __attribute__ ((packed));

/** 
 * type for session keys 
 **/
typedef struct {
  unsigned char key[SESSIONKEY_LEN];
} SESSIONKEY __attribute__ ((packed));

/**
 * Content (returned).
 **/
typedef struct {
  MessagePart header __attribute__ ((packed));   
  /** The double-hash */
  HashCode160 hash __attribute__ ((packed));
  /** The content */
  CONTENT_Block content __attribute__ ((packed));
} CONTENT_Message __attribute__ ((packed));

/**
 * Combined message with HELO and SKEY, with wrapper for
 * sendToHost.
 **/
typedef struct {
  /* length of this struct - 2 */
  HELO_Message helo __attribute__ ((packed));
  SKEY_Message sks __attribute__ ((packed));
} HELOSKEY __attribute__ ((packed));

/* ************************ TCP messages ********************** */
/* these messages are exchanged between gnunetd and the clients */

/**
 * Header for all TCP communications.
 **/
typedef struct {
  /** The length of the struct (in bytes, excluding the length field itself) **/
  TCP_HEADER size __attribute__ ((packed));
  /** The type of the message (TCPTYPE_XXXX) **/
  unsigned short tcpType __attribute__ ((packed));
} TYPED_TCP_HEADER __attribute__ ((packed));

/**
 * TCP communication: content send back by gnunetd
 **/
typedef struct {
  /** The TCP header (values: sizeof(TCP_Content_reply), TCP_REPLY_CONTENT) **/ 
  TYPED_TCP_HEADER header;

  /** The double-hash */
  HashCode160 hash __attribute__ ((packed));
  /** The content */
  CONTENT_Block content __attribute__ ((packed));
} TCP_Content_Reply __attribute__ ((packed));

/**
 * Structure for an incoming request messages from the local TCP link
 **/
typedef struct {
   /** The TCP header (values: sizeof(TCP_Query_request), TCP_REQUEST_QUERY) **/ 
  TYPED_TCP_HEADER header; 
  /** how important is this request (network byte order) */  
  unsigned int priority __attribute__ ((packed)); 
  /** time to live in seconds (network byte order) */
  int ttl __attribute__ ((packed));      
  HashCode160 query __attribute__ ((packed));
} TCP_Query_Request __attribute__ ((packed));

/**
 * Structure for an incoming request messages from the local TCP link
 * to add content to the node. 
 **/
typedef struct {
   /** The TCP header (values: sizeof(TCP_Content_Request), TCP_REQUEST_INSERT) **/ 
  TYPED_TCP_HEADER header; 
  /** The (initial) priority of the data  (network byte order)**/
  unsigned int importance;
  /** the key of the data to insert **/
  HashCode160 doubleHash;
  /** the data to insert **/
  CONTENT_Block content;  
} TCP_Insert_Request __attribute__ ((packed));

/**
 * Structure for an incoming request messages from the local TCP link
 * to add content to the INDEX of the node.
 **/
typedef struct {
   /** The TCP header (values: sizeof(TCP_Indexing_Request), TCP_REQUEST_INDEX) **/ 
  TYPED_TCP_HEADER header; 
  /* indexing information */
  ContentIndex contentIndex;
} TCP_Indexing_Request __attribute__ ((packed));

#define TCP_FILE_LIST_FILENAME 1024

/**
 * Structure for an incoming request messages from the local TCP link
 * to add a filename to the list of directly shared files
 **/
typedef struct {
   /** The TCP header (values: sizeof(TCP_File_List_Request), TCP_REQUEST_LIST_FILE) **/ 
  TYPED_TCP_HEADER header; 
  /** The length of the filename in characters (network byte order) **/
  unsigned short fileNameLength;
  /** reserved bits  (network byte order) **/
  unsigned short reserved;
  /** the filename (max. 1023 characters), padded with zeros **/
  char filename[TCP_FILE_LIST_FILENAME];
} TCP_File_List_Request __attribute__ ((packed));

/**
 * Structure for an incoming request messages from the local TCP link
 * to confirm the addition of a filename and to communicate the
 * index of that file. The length is communicated only implicitly
 * because the original name is padded (but the client should know
 * the length anyway).
 **/
typedef struct {
   /** The TCP header (values: sizeof(TCP_File_Index_Reply), TCP_REPLY_FILE_INDEX) **/ 
  TYPED_TCP_HEADER header; 
  /** The length of the filename in characters  (network byte order) **/
  unsigned int index;
  /** the filename that was inserted (max. 1023 characters), padded with zeros**/
  char filename[TCP_FILE_LIST_FILENAME];
} TCP_File_Index_Reply __attribute__ ((packed));

/* ******************** Policy Datastructures *********************** */

/** drop the query if & with this bitmask is 0 */
#define QUERY_DROPMASK 0x00FF0000
/** answer if local files match */
#define QUERY_ANSWER   0x00020000
/** forward the query, priority is encoded in QUERY_PRIORITY_BITMASK */
#define QUERY_FORWARD  0x00040000
/** forward the query with us as the originating node */
#define QUERY_INDIRECT 0x00080000
/** priority to use */
#define QUERY_PRIORITY_BITMASK 0x0000FFFF

/** type of the results of the polciy module */
typedef unsigned int QUERY_POLICY;

/* *********************** FileFormats ***************************** */

#define MAX_DESC_LEN     256
#define MAX_FILENAME_LEN 128
#define MAX_MIMETYPE_LEN 128
#define ROOT_MINOR_VERSION 0
#define ROOT_MAJOR_VERSION 0

/**
 * The structure of the root node - contains pertinent information
 * for the file (File length, checksum, hashcode of main indirection
 * node, description length, and description.
 **/
typedef struct {
  /* length of the file in network byte order (!) */
  size_t file_length __attribute__ ((packed));
  /* top-level crc (network byte order (!)) */
  Ulong crc __attribute__ ((packed));
  /* the hashcode of the topmost IBlock */
  HashCode160 hashCode __attribute__ ((packed));
  /* major format version, in network byte order */
  unsigned short major_formatVersion __attribute__ ((packed));
  /* minor format version, in network byte order */
  unsigned short minor_formatVersion __attribute__ ((packed));
  /** description of the contents **/
  char description[MAX_DESC_LEN] __attribute__ ((packed)); 
  /** suggested filename **/
  char filename[MAX_FILENAME_LEN] __attribute__ ((packed));
  /** mime-type (as claimed by insertion!) */
  char mimetype[MAX_MIMETYPE_LEN] __attribute__ ((packed));
  /** padding (zaro bytes?) */
  char padding[CONTENT_SIZE - (sizeof(size_t)+
			       sizeof(Ulong)+
			       sizeof(HashCode160)+
			       sizeof(unsigned short)+
			       sizeof(unsigned short)+
			       MAX_DESC_LEN+
			       MAX_FILENAME_LEN+
			       MAX_MIMETYPE_LEN)] __attribute__ ((packed));
} RootNode __attribute__ ((packed));

/* ************************ RUN-TIME Data Structures *********************** */

#define STAT_DOWN             0
#define STAT_WAITING_FOR_PING 1
#define STAT_WAITING_FOR_PONG 2
#define STAT_UP               3

/**
 * A slot in the output buffer that contains a sequence of
 * messages. Since GNUnet 0.4.x always starts with a sequence
 * message, this is made explicit by this struct for convenience.
 **/
typedef struct {
  SEQUENCE_Message sequence;
  unsigned char variable[BUFSIZE - sizeof(SEQUENCE_Message)];
} BufferSlot;

/**
 * Type of the connection table.
 **/
typedef struct {
  /** semaphore guarding access to this BE */
  Mutex sem;
  /** which host are we talking to here? */
  HostIdentity hostId;
  /** How much do we trust the host? signed
      because that makes many operations that go 
      negative easier. Of course, negative trust makes
      no sense. */
  int trust;
  /** the current session key */
  SESSIONKEY skey;
  /** at which time was the sessionkey created (by whichever party) */
  time_t created;
  /** is this host alive? bitmap marking the times of the last-active intervals */
  unsigned int isAlive; /* 1 true, 0 false, rotated!*/
  /** challenge in SKEY handshake, 0 if we send the SKEY */
  int challenge;
  /**  Status of the connection (STAT_XXX) **/
  unsigned int status;
  /** last sequence number received (!) on this connection (highest) */
  unsigned int lastSequenceNumberReceived;
  /** bit map indicating which of the 32 sequence numbers before the last
      were received (good for accepting out-of-order packets and
      estimating reliability of the connection) */
  unsigned int lastPacketsBitmap;
  /** priority for flushing/sending this buffer, max(priorities of packets in buffer) */
  unsigned int prio; /* what is the max. priority in this buffer */
  /** the buffer containing the data to be send to this host */
  BLOCK_LENGTH len;
  /** contents of the buffer */
  BufferSlot buffer;
} BufferEntry;



/**
 * Indirection table entry. Lists what we're looking for,
 * where to forward it, and how long to keep looking for it.
 **/
typedef struct {
  /* what are we waiting for? */
  HashCode160 hash;  
  /* lock for this bucket */
  Mutex lock;
  /* when can we forget about this entry? */
  time_t ttl;
  /* how much is this query worth? we may want to be a bit more
     precise here: the query may be worth X for giving credit to
     somebody who sends a reply to us and Y for forwarding to
     the originator;
     notice that the priority already changes after the first
     reply is received. */
  unsigned int priority;
  /* which replies have we already seen? */
  unsigned int seenIndex; 
  /* number of hosts waiting for this query */
  HashCode160 seen[MAX_REPLIES]; 
  /* how many hosts are waiting for an answer to this query */
  unsigned int hostsWaiting; 
  /* who are these hosts? */
  HostIdentity destination[MAX_HOSTS_PER_QUERY]; 
  /* local TCP sockets to send the reply to, NULL if none */
  GNUNET_TCP_SOCKET * tcpsocks[MAX_TCP_PER_QUERY]; 
} IndirectionTableEntry;


/* ************** Structs for GNUnet clients ********* */

/* a method that takes one argument which is
   a RequestEntry*, and a CONTENT_Block*. 
   How would you type that in C!? */
typedef void * Listener;

/**
 * Representation of a node in the tree (Block or IBlock).
 **/
typedef struct {

  /**
   * The parent-node. Either of type Node or RNode.
   **/
  void * parent;

  /**
   * Position in the tree (corresponding to the lowest
   * position index of its leftmost child)
   **/
  size_t pos;
	
  /**
   * This node's depth in the tree
   **/
  int depth;
		
  /**
   * The key of the node (to decrypt)
   **/
  HashCode160 key;
	
  /**
   * What is the request that maches this node?
   **/
  HashCode160 request;
	
  /**
   * The 1024 bytes of data associated with this node
   **/
  CONTENT_Block data;  

  /**
   * Has this block been resolved? 0: no, 1: downloaded, 2: transitively yes
   **/
  int resolved;
 
} Node;

/**
 * Inner node in the tree, represents IBlocks.
 **/
typedef struct {
  /**
   * This field must be first as INodes are sometimes
   * plainly cast to Nodes.
   **/
  Node node;

  /**
   * Child-nodes (may have NULL-entries).
   **/
  Node * childs[NODES_PER_INODE];
	
  /**
   * Checksum of the collected children of this node
   * (what we'd like to get)
   **/
  Ulong crc32;
  
} INode;

/**
 * The root of the tree.
 **/
typedef struct {
  /**
   * Dummy link to terminate the parent-linked list.
   * should always be NULL.
   **/
  void * nullParent;

  /**
   * Link to the display.
   **/
  void (*downloadModel)(size_t, size_t, void*);
  
  /**
   * The mmapped file.
   **/
  int fileHandle;

  /**
   * length of file to be retrieved. Used for
   * user-feedback.
   **/
  size_t fileLength;

  /**
   * Depth of the tree representing this file.
   **/
  int treeDepth;

  /**
   * The top-Inodes.
   **/
  Node * root;

  /**
   * The CRC of the root-node.
   **/
  Ulong crcRoot;

  /**
   * Total amount of data (LEAF NODES) downloaded 
   * so far for this rootnode
   **/
  size_t bytes_downloaded;

} RNode;


/**
 * Format of a request.
 **/
typedef struct {
  /**
   * Whom to call once it is received?
   **/
  Listener receiver;

  /**
   * The message send to GNUnet.
   **/
  TCP_Query_Request message;

  /**
   * Last time the query was send
   **/
  time_t lasttime;

  /**
   * Metadata to be kept around for the receiver.
   * Usually contains information like the position in
   * the file where to write the result.
   **/
  Node * metadata;

  /**
   * How long have we been actively trying this one?
   **/
  int tries;
} RequestEntry;


/* finally, make sure everybody has the debugging code */
#include "debugging.h"

#endif

/* end of gnettypes.h */
