/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * Code to maintain the list of currently known hosts
 * (in memory structure of data/hosts) and temporary blacklisting
 * information.
 * @author Christian Grothoff
 **/ 

#ifndef KNOWNHOSTS_H
#define KNOWNHOSTS_H

#include "config.h"
#include "gnettypes.h"

/**
 * Initialize the knownhosts module.
 **/
void initKnownhosts();

/**
 * Shutdown the knownhosts module.
 **/
void doneKnownhosts();

/**
 * Check if 2 hosts are the same (returns 1 if yes)
 * @param first the first host
 * @param second the second host
 * @returns 1 if the hosts are the same, 0 otherwise
 **/
int hostIdentityEquals(HostIdentity * first, 
		       HostIdentity * second);
 
/**
 * Add a host to the list.
 **/
void addHostToKnown(HostIdentity * identity);

/**
 * Delete a host from the list
 **/
void delHostFromKnown(HostIdentity * identity);

/**
 * Blacklist a host. This method is called if a host
 * failed to respond to a connection attempt.
 * @return OK on success SYSERR on error
 **/
int blacklistHost(HostIdentity * identity);

/**
 * Whitelist a host. This method is called if a host
 * successfully established a connection. It typically
 * resets the exponential backoff to the smallest value.
 * @return OK on success SYSERR on error
 **/
int whitelistHost(HostIdentity * identity);

/**
 * Call a method for each known host; not reentrant!
 * @param callback the method to call for each host, may be NULL
 * @param now the time to use for excluding hosts due to blacklisting, use 0 
 *        to go through all hosts.
 * @param data an argument to pass to the method
 * @return the number of known hosts matching
 **/
int forEachHost(void (*callback)(HostIdentity*, void *), 
		time_t now,
		void * data);
 
/**
 * Call this method periodically to scan data/hosts for new hosts.
 **/
void cronScanDirectoryDataHosts(void * unused);


#endif
/* end of knownhosts.h */

