/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * This module is responsible to lookup queries.
 * @file include/lookup.h
 * @author Christian Grothoff
 **/

#ifndef LOOKUP_H
#define LOOKUP_H

#include "config.h"
#include "gnettypes.h"
#include "lookup_collision.h"
#include "util/cron.h"
#include "util/storage.h"
#include "configuration.h"
#include "symcipher.h"

/**
 * Initialize the lookup module.
 **/
void initLookup();

/**
 * Initialize the lookup module (full init)
 **/
void fullInitLookup();

/**
 * Is space available on the drive? Returns the number of
 * blocks (1k) available. If there would be no space except
 * if less important content is discarded, this method 
 * *deletes* 1k of the less important content and returns 
 * 1. If there is only more important content, the
 * method returns 0.
 * @param importance how important is the content?
 **/
int spaceAvailable(int importance);

/**
 * Lookup the file on the local machine. If found,
 * send it back.
 * @param query the triple-hash of the content
 * @param qp the polciy (for us: priority)
 * @param callback procedure to call on the results,
 *        must take three arguments: NULL, a pointer
 *        to the CONTENT_Block, and the query.
 * @returns YES if found, NO if not.
 **/
int tripleHashLookup(HashCode160 * query,
		     QUERY_POLICY qp,
		     int (* callback)(HostIdentity*,CONTENT_Block*,
				       HashCode160*));

/**
 * Add an entry to the cache. The ->left and ->right
 * references are 0 when this method is called (new entry!)
 **/
void addEntry(ContentIndex * entry);

/**
 * Obtain the number of shared files.
 **/
unsigned int getSharedFileCount();

/**
 * Add a name to the list of filenames.
 **/
int appendFilename(char * filename);


#define LOOKUP_LIPS_COUNT 32
#define LOOKUP_FACTOR 128
#define LOOKUP_LIPSPART_COUNT ((CACHE_HASH_TABLE_SIZE/LOOKUP_FACTOR))

#endif
/* end of lookup.h */
