/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * This file provides methods to manipulate host credit
 * rankings. It should be merged with outbuffer soon.
 * @author Tzvetan Horozov
 * @author Christian Grothoff
 **/ 

#ifndef CONNECTION_H
#define CONNECTION_H

#include "config.h"
#include "gnettypes.h"
#include "symcipher.h"
#include "configuration.h"
#include "server/sessionkey.h"
#include "server/identity.h"
#include "server/routing.h"
#include "server/policy.h"
#include "util/cron.h"
#include "util/semaphore.h"
#include "util/checksum.h"

/* ********************** Exported globals ******************************* */

extern DirContents * CONNECTION_knownHosts;
extern int CONNECTION_currentMaxKnownHosts;
extern int CONNECTION_currentKnownHosts;
extern Mutex CONNECTION_knownHostsLock;
extern char * CONNECTION_hostsDirectory;
extern HostIdentity myIdentity;


/* ********************* Methods called from "node" ********************** */

/**
 * Initialize this module.
 **/
void initConnection();

/**
 * Call this method periodically to scan data/hosts for new hosts.
 **/
void cronScanDirectoryDataHosts(void * unused);

/**
 * For debugging.
 **/
void printConnectionBuffer();

/* ********************** Send-interface ****************************** */

/**
 * Compute the hashtable index of a host id.
 **/
unsigned int computeIndex(HostIdentity * hostId);

/**
 * Look for a host in the table. Also aquires the lock!
 * @returns the entry of the host in the table, NULL if not present
 **/
BufferEntry * lookForHost(HostIdentity * hostId);

/**
 * Force adding of a host to the buffer.
 **/
BufferEntry * addHost(HostIdentity * hostId);
 
/**
 * Send a message to "many" people., except ourselves! TTH (enqueue)
 * @param message the message to send
 * @param priority how important is the message? The higher, the more important
 **/
void broadcast(MessagePart * message,
	       unsigned int priority);

/**
 * Send a message to a specific host (reply, enqueue)
 * @param message the message to send (unencrypted!), first BLOCK_LENGTH_SIZE bytes give size
 * @param hostId the identity of the receiver
 * @param priority how important is the message?
 **/
void unicast(MessagePart * message,
	     HostIdentity * hostId,
	     unsigned int priority);

/**
 * We have gotten a query and decided to forward it
 * (or it may be our own query). This method must
 * decide to whom to send the query. 
 * <p>
 * Attention: if the returnTo-identity is not us, then
 * the receiver must also been given the identity of the
 * host we are forwarding (but not indirecting) for.
 * @param msg the message to forward.
 **/
void forwardQuery(QUERY_Message * msg);

/**
 * Perform an operation for all connected hosts.
 * The BufferEntry structure is passed to the method.
 * No synchronization or other checks are performed.
 * @param method the method to invoke
 * @param arg the second argument to the method
 **/ 
void forAllConnectedHosts(void (*method)(BufferEntry *, void*),
			  void * arg);


/**
 * Send a buffer; assumes that access to be is already 
 * synchronized.
 **/
void sendBuffer(BufferEntry * be);

/* ************************* encryption service ********************** */

/**
 * Decipher data comming in from a foreign host.
 * @param data the data to decrypt
 * @param size how big is data?
 * @param hostId the sender host that encrypted the data 
 * @param result where to store the decrypted data
 * @returns the size of the decrypted data, -1 on error
 **/
int decryptFromHost(BLOWFISHEncryptedData * data,
		    BLOCK_LENGTH size,
		    HostIdentity * hostId,
		    void * result);  

/**
 * Check the sequence number. Updates the sequence
 * number as a side-effect.
 * @returns OK if ok, SYSERR if not.
 **/
int checkSequenceNumber(HostIdentity * hostId,
			unsigned int sequenceNumber);

/**
 * We received a sign of life from this host.
 **/
void notifyActive(HostIdentity * hostId,
		  int challenge);


#endif
/* end of connection.h */

