/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * This module keeps track of host-identity - address (IP) translation.
 * Basically a mapping HostAddress <==> HostIdentity <==> Public Key 
 * with signature and expiration date for the HostAddress.
 * <p>
 * This file basically is all about the HELO_Message.
 * @file gnet/include/identity.h
 * @author Christian Grothoff
 * @author Tzvetan Horozov
 **/

#ifndef IDENTITY_H
#define IDENTITY_H

#include "config.h"
#include "gnettypes.h"
#include "util/returnvalues.h"
#include "util/checksum.h"
#include "util/storage.h"
#include "util/semaphore.h"
#include "server/keyservice.h"
#include "server/sessionkey.h"
#include "server/connection.h"

/** 
 * Initialize identity module. Requries configuration.
 **/
void initAddress();

/**
 * Bind a host identity (IP+port) to a hostId.
 * @param hostId the hostId (hash of publickey)
 * @param conInfo the connection information (IP+port) for the host
 **/
void bindAddress(HELO_Body * msg);

/**
 * Resolve identity (find matching connection)
 * @param hostId the host id
 * @param result the connection information that was bound to that identity or NULL if no information is available
 * @return SYSERR on failure, OK on success
 **/
int getConnectionInformation(HostIdentity *  hostId,
			     HostAddress * result);

/**
 * Obtain the identity information for the current node. DO NOT FREE!
 * @return SYSERR on error, OK on success
 **/
int getAddress(HostAddress * identity);

/**
 * Send a packet to a given host.
 * @param hostAddr the host address of the target host,
 *        if NULL, sendToHost will look it up
 * @param hostId the id of the target host
 * @param size the size of the content
 * @param content the actual data (after the header)
 * @param encryptedFlag is the data encrypted (subprotocol)
 * @param crc is the checksum of the unencrypted data (without header)
 **/
void sendToHost(HostAddress * hostAddr,
		HostIdentity * hostId,
		BLOCK_LENGTH size,
		void * content,
		int encryptedFlag,
                int crc);

/**
 * Obtain the public key of a known host.
 * @param hostId the host id
 * @param result where to store the result
 * @returns SYSERR on failure, OK on success
 **/
int identity2Key(HostIdentity *  hostId,
		 PublicKey * result);
 
/**
 * Create the signed HELO message for this host.
 * @param msg the address where to write the HELO message to
 **/
void createHELO(HELO_Message * msg);

/**
 * Verify that a HELO body is still valid.
 * @returns SYSERR if the message is invalid, OK if it is ok.
 **/
int verifyHELO(HELO_Body * msg);


#endif
/* end of identity.h */
