/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * Policy interface. This is the interface to the C part of the policy.
 * @author Christian Grothoff
 * @file include/policy.h
 **/

#ifndef POLICY_H
#define POLICY_H

#include "config.h"
#include "gnettypes.h"
#include "lookup.h"
#include "server/connection.h"
#include "util/statuscalls.h"

/**
 * A new packet is supposed to be send out. Should it be
 * dropped because the load is too high?
 * <p>
 * @param priority the highest priority of contents in the packet
 * @return OK if the packet should be handled, SYSERR if the packet should be dropped.
 **/
int outgoingCheck(int priority);

/**
 * A new packet is coming in. The question is, if we should
 * have a look at it at all (incl. decryption). The sender
 * is known. 
 * <p>
 * The policy must decide if we should drop the packet. The
 * reason may be that we don't trust the host or that the load
 * is to high (or both or a combination). 
 * <p>
 * @param senderIdentity the identity of the sender
 * @return OK if the packet should be handled, SYSERR if the packet should be dropped.
 **/
int incomingCheck(HostIdentity * senderIdentity);

/**
 * A query has been received. The question is, if it should be forwarded
 * and if with which priority. Routing decisions(to whom) are to be taken elsewhere.
 * <p>
 * @param senderIdentity the host *forwarding* (or originating, we don't know) us the query
 * @param priority the priority the query had when it came in, may be an arbitrary number if the 
 *        sender is malicious! Cap by trustlevel first!
 * @return binary encoding: QUERY_XXXX constants
 **/
QUERY_POLICY evaluateQuery(HostIdentity * senderIdentity, unsigned int priority);


/**
 * Some content dropped by. We may want to store it locally, or not.
 * The policy can do with it whatever it sees fit.
 * @param senderIdentity the sender of the content
 * @param content the data
 * @param preference how good was the data?
 **/
void evaluateContent(HostIdentity * senderIdentity,
		     CONTENT_Block * content,
		     HashCode160 * hash,
		     int preference);


#endif
/* end of policy.h */
