/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * sessionkey - standard communication with another node.
 * encrypt/decrypt - for the content using the hashcode
 * @author Christian Grothoff
 * @author Ioana Patrascu
 * @file include/symcipher.h
 **/

#ifndef SYMCIPHER_H
#define SYMCIPHER_H

#include "config.h"
#include <stdlib.h>
#include <string.h>

#include <openssl/blowfish.h>
#include <openssl/evp.h>

#include "configuration.h"
#include "gnettypes.h"


/**
 * Create a new Session key.
 **/
void makeSessionkey(SESSIONKEY * key);

/**
 * Encrypt a block with the public key of another
 * host that uses the same cyper.
 * @param block the block to encrypt
 * @param len the size of the block
 * @param sessionkey the key used to encrypt
 * @returns the size of the encrypted block, -1 for errors
 **/
int encryptBlock(void * block, 
		 BLOCK_LENGTH len,
		 SESSIONKEY * sessionkey,
		 BLOWFISHEncryptedData * result);

/**
 * Decrypt a given block with the sessionkey.
 * @param sessionkey the key used to decrypt
 * @param block the data to decrypt, encoded as returned by encrypt
 * @param size how big is the block?
 * @param result address to store the result at
 * @return -1 on failure, size of decrypted block on success
 **/
int decryptBlock(SESSIONKEY * sessionkey, 
		 BLOWFISHEncryptedData * block,
		 BLOCK_LENGTH size,
		 void * result);

/**
 * Encrypts a given data block
 *
 * @param data represents the data block
 * @param size how big is data?
 * @param hashcode represents the key concatenated with the initial
 *        value used in the alg
 * @param result where to store the result (encrypted block)
 * @returns OK on success, SYSERR on error
 **/
int encryptContent(CONTENT_Block * data,
		   HashCode160 * hashcode,
		   CONTENT_Block * result);


/**
 * Decrypts a given data block
 *
 * @param data represents the data block
 * @param size how big is data?
 * @param hashcode represents the key concatenated with the initial
 *        value used in the alg
 * @param result where to store the result (decrypted block)
 * @returns OK on success, SYSERR on error
 **/
int decryptContent(CONTENT_Block * data,
		   HashCode160 * hashcode,
		   CONTENT_Block * result);



/* end of sessionkey.h */
#endif
