/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Hashing methods.
 * @author Christian Grothoff
 * @file include/util/hashing.h
 **/

#ifndef HASHING_H
#define HASHING_H

#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <openssl/ripemd.h>

/**
 * A 160-bit hashcode
 **/
typedef struct {
  int a __attribute__ ((packed));
  int b __attribute__ ((packed));
  int c __attribute__ ((packed));
  int d __attribute__ ((packed));
  int e __attribute__ ((packed));
} HashCode160 __attribute__ ((packed));

/**
 * Hash2Hex: filename, consisting of 32 bytes [0-9A-Z] plus
 * null-termination.
 **/
typedef struct {
  unsigned char data[sizeof(HashCode160)*2+1];
} HexName;

/**
 * Compute the distance between 2 hashcodes.
 * The computation must be fast, not involve
 * a.a or a.e (they're used elsewhere), and
 * be somewhat consistent. And of course, the
 * result should be a positive number.
 **/
int distanceHashCode160(HashCode160 * a, 
			HashCode160 * b);
 
/**
 * compare two hashcodes.
 **/
int equalsHashCode160(HashCode160 * a, 
		      HashCode160 * b);
 


/**
 * Convert (hash) block to hex (= filename)
 * @param block the sequence to convert
 * @param result where to store thestring (\0-terminated), hex-encoding
 **/
void hash2hex(HashCode160 * block,
	      HexName * result);

/**
 * Convert hex (filename) to the hostIdentity
 * @param hex the filename
 * @param hash is set to the correspoinding host identity
 **/
void hex2hash(HexName * hex,
	      HashCode160 * hash);

/**
 * Hash block of given size.
 * @param block the data to hash, length is given as a second argument
 * @param ret pointer to where to write the hashcode
 **/
void hash(void * block,
	  int size,
	  HashCode160 * ret);
 
#endif

/* end of hashing.h */
