/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * Hostkey - management of the node's main identity.
 * @author Christian Grothoff
 * @file include/util/hostkey.h
 **/

/**
 * We currently do not handle encryption of data
 * that can not be done in a single call to the
 * RSA methods (read: large chunks of data).
 * We should never need that, as we can use
 * the hash for larger pieces of data for signing,
 * and for encryption, we only need to encode sessionkeys!
 **/

#ifndef HOSTKEY_H
#define HOSTKEY_H

#include "config.h"

#include <stdlib.h>
#include <string.h>

#include <openssl/rsa.h>
#include <openssl/evp.h>
#include <openssl/bn.h>

#define HOSTKEY RSA *

/**
 * Length of RSA encrypted data (2048 bit)
 **/
#define RSA_ENC_LEN 256

/**
 * Length of an RSA KEY (d,e,len), 2048 bit (=256 octests) key d, 2 byte e
 **/
#define RSA_KEY_LEN 258

/** type used for length of blocks; except for whole files */
typedef unsigned short BLOCK_LENGTH;
/** size (in bytes) of BLOCK_LENGTH */
#define BLOCK_LENGTH_SIZE sizeof(BLOCK_LENGTH)

typedef struct {
  BLOCK_LENGTH len; /*  in big-endian! */
  BLOCK_LENGTH sizen;/*  in big-endian! */
  BLOCK_LENGTH sizee;/*  in big-endian! */
  BLOCK_LENGTH sized;/*  in big-endian! */
  BLOCK_LENGTH sizep;/*  in big-endian! */
  BLOCK_LENGTH sizeq;/*  in big-endian! */
  BLOCK_LENGTH sizedmp1;/*  in big-endian! */
  BLOCK_LENGTH sizedmq1;/*  in big-endian! */
  unsigned char key[0];
} HostKeyEncoded;

typedef struct {
  unsigned char sig[RSA_ENC_LEN]; 
} Signature;

typedef struct {
  BLOCK_LENGTH len __attribute__ ((packed)); /*  in big-endian, must be RSA_KEY_LEN+2 */
  BLOCK_LENGTH sizen __attribute__ ((packed));  /*  in big-endian! */ 
  unsigned char key[RSA_KEY_LEN] __attribute__ ((packed));
  unsigned short padding __attribute ((packed)); /* padding (must be 0) */
} PublicKey __attribute__ ((packed));

typedef struct {
  unsigned char encoding[RSA_ENC_LEN] __attribute__ ((packed));
} RSAEncryptedData;


/**
 * Initialize Random number generator.
 **/
void initRAND(); 

/**
 * create a new hostkey. Callee must free return value.
 **/
HOSTKEY makeHostkey(); 

/**
 * Free memory occupied by hostkey
 * @param hostkey pointer to the memory to free
 **/
void freeHostkey(HOSTKEY hostkey); 

/**
 * Extract the public key of the host.
 * @param result where to write the result.
 **/
void getPublicKey(HOSTKEY hostkey,
		  PublicKey * result);

/**
 * Encode the private key in a format suitable for
 * storing it into a file.
 * @param hostkey the hostkey to use
 * @returns encoding of the private key.
 **/
HostKeyEncoded * encodeHostkey(HOSTKEY hostkey);

/**
 * Decode the private key from the file-format back
 * to the "normal", internal, RSA format.
 * @param encoded the encoded hostkey
 * @returns the decoded hostkey
 **/
HOSTKEY decodeHostkey(HostKeyEncoded * encoding);

/**
 * Encrypt a block with the public key of another
 * host that uses the same cyper.
 * @param block the block to encrypt
 * @param size the size of block
 * @param publicKey the encoded public key used to encrypt
 * @param target where to store the encrypted block
 * @returns SYSERR on error, OK if ok
 **/
int encryptHostkey(void * block, 
		   BLOCK_LENGTH size,
		   PublicKey * publicKey,
		   RSAEncryptedData * target);

/**
 * Decrypt a given block with the hostkey. 
 * @param hostkey the hostkey to use
 * @param block the data to decrypt, encoded as returned by encrypt, not consumed
 * @param result pointer to a location where the result can be stored
 * @param max the maximum number of bits to store for the result, if
 *        the decrypted block is bigger, an error is returned
 * @returns the size of the decrypted block, -1 on error
 **/
int decryptHostkey(HOSTKEY hostkey, 
		   RSAEncryptedData * block,
		   void * result,
		   unsigned int max);

/**
 * Sign a given block.
 * @param block the data to sign, first BLOCK_LENGTH_SIZE bytes give length
 * @param size how many bytes to sign
 * @param result where to write the signature
 * @return SYSERR on error, OK on success
 **/
int sign(HOSTKEY hostkey, 
	 BLOCK_LENGTH size,
	 void * block,
	 Signature * result);

/**
 * Verify signature.
 * @param block the signed data
 * @param len the length of the block 
 * @param sig signature
 * @param publicKey public key of the signer
 * @returns OK if ok, SYSERR if invalid
 **/
int verifySig(void * block,
	      BLOCK_LENGTH len,
	      Signature * sig,	      
	      PublicKey * publicKey);



/* end of hostkey.h */
#endif
