/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * Storage - management of the node's files.
 * @author Christian Grothoff
 * @file include/util/storage.h
 **/

#ifndef STORAGE_H
#define STORAGE_H

#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include "util/hashing.h"
#ifndef SOMEBSD
#include <wordexp.h>
#endif

/* size of the blocks we slice file data into (and indirection node size) */
#define CONTENT_SIZE 1024

/* number of children per INode, <=(CONTENT_SIZE-4)/20 */
#define NODES_PER_INODE 51

/* maximum number of matching files per keyword, default: 1024 */
#ifndef SOMEBSD
    #define MAX_FSIZE 1024
#else
    #define MAX_FSIZE 512
#endif

/**
 * A CONTENT_SIZE block.
 **/
typedef struct {
  unsigned char content[CONTENT_SIZE]; 
} CONTENT_Block __attribute__ ((packed));

typedef unsigned char * FileName;

/**
 * Structure to represent n HexNames obtained from
 * scanning a directory.
 **/
typedef struct {
  HexName files[0];
} DirContents;

/**
 * Assert that fil corresponds to a filename
 * (of a file that exists and that is not a directory).
 * @returns 1 if yes, 0 if not (will print an error
 * message in that case, too).
 **/
int assertIsFile(FileName fil);

/**
 * Build a filename from directory an fil parts.
 * @param fil the name of the file, will NOT be deallocated!
 * @returns the full file name, expanded with wordexp.
 **/
FileName expandFileName(FileName fil);

/**
 * implementation of "mkdir -p"
 * @param dir the directory to create
 **/
void mkdirp(char * dir);

/**
 * Build a filename from directory and filename, completing like the shell does
 * @param dir the name of the directory, may contain ~/ or other shell stuff. Will 
 *        NOT be freed!
 * @param fil the name of the file, will NOT be deallocated anymore!
 * @param result where to store the full file name (must be large enough!)
 **/
void buildFileName(char * dir,
		   HexName * fil,
		   FileName result);

/**
 * Read the contents of a binary file into a buffer.
 * @param fileName the name of the file, not freed,
 *        must already be expanded!
 * @param len the maximum number of bytes to read
 * @param result the buffer to write the result to
 * @return the number of bytes read on success, -1 on failure
 **/ 
int readFile(FileName fileName,
	     int  len,
	     void * result);

/**
 * Write a buffer to a file.
 * @param fileName the name of the file, NOT freed!
 * @param buffer the data to write
 * @param n number of bytes to write
 * @param mode the mode for file permissions
 **/ 
void writeFile(FileName fileName, 
	       void * buffer,
	       int n,
	       char *mode);

/**
 * Scan a directory for files. The name of the directory
 * must be expanded first (!).
 * @param dirName the name of the directory
 * @param callback the method to call for each file
 * @param data argument to pass to callback
 * @return the number of files found, -1 on error
 **/
int scanDirectory(FileName dirName,
		  void (*callback)(HexName *, void *),
		  void * data);

#endif

/* end of storage.h */
