/*
     This file is part of GNUnet

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * Generic TCP code. This module is used to receive or send records (!) 
 * from a TCP stream. The code automatically attempts to re-connect if
 * the other side closes the connection.<br>
 * The code can be used on the server- or the client side, just in case
 * of the server the reconnect can of course not be used. The TCP stream
 * is broken into records of maximum length MAX_TCP_RECORD_SIZE, 
 * each preceeded by an
 * unsigned short giving the length of the following record.<p>
 * 
 * @author Christian Grothoff
 * @file include/util/tcpio.h
 **/

#ifndef TCPIO_H
#define TCPIO_H

#include "config.h"

#include "returnvalues.h"

/* The value here must be bigger than the largest record transferred,
   it must also be smaller (or equal) to 64k (-2byte) */
#define MAX_TCP_RECORD_SIZE 1500


#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <signal.h>
#include <netdb.h>

typedef unsigned short TCP_HEADER;

typedef struct {
  /* the socket handle, -1 if invalid / not life */
  int socket;
  /* the following is the IP for the remote host for client-sockets,
     as returned by gethostbyname("hostname");
     server sockets should use NULL */
  struct hostent * ip_info;
  /* the port number, in host byte order */
  unsigned short port;
} GNUNET_TCP_SOCKET;

typedef struct {
  /* the size of the data to send/the result, including the 2 byte header */
  TCP_HEADER size __attribute__ ((packed));
  /* the data */
  char data[MAX_TCP_RECORD_SIZE] __attribute__ ((packed));
} TCP_SOCKET_BUFFER __attribute__ ((packed));

/**
 * Initialize TCPIO module.
 **/
void initTCPIO();
 
/**
 * Shutdown TCPIO module.
 **/
void doneTCPIO();

/**
 * Initialize a GNUnet client socket.
 * @param port the portnumber in host byte order
 * @param hostname the name of the host to connect to
 * @param result the SOCKET (filled in)
 * @return OK if successful, SYSERR on failure
 **/
int initGNUnetClientSocket(unsigned short port,
			   char * hostname,
			   GNUNET_TCP_SOCKET * result);

/**
 * Initialize a GNUnet server socket.
 * @param sock the open socket
 * @param result the SOCKET (filled in)
 * @return OK (always successful)
 **/
int initGNUnetServerSocket(int socket,
			   GNUNET_TCP_SOCKET * result);

/**
 * Check if a socket is open. Will ALWAYS return 'true'
 * for a valid client socket (even if the connection is
 * closed), but will return false for a closed server socket.
 * @return 1 if open, 0 if closed
 **/
int isOpenConnection(GNUNET_TCP_SOCKET * sock);


/**
 * Read from a GNUnet TCP socket.
 * @param sock the socket
 * @param buffer the buffer to write data to
 * @return OK if the read was successful, SYSERR if the socket
 *         was closed by the other side (if the socket is a
 *         client socket and is used again, tcpio will attempt
 *         to re-establish the connection [temporary error]).
 **/
int readFromSocket(GNUNET_TCP_SOCKET * sock,
		   TCP_SOCKET_BUFFER * buffer);

/**
 * Write to a GNUnet TCP socket.
 * @param sock the socket to write to
 * @param buffer the buffer to write
 * @return OK if the write was sucessful, otherwise SYSERR.
 **/
int writeToSocket(GNUNET_TCP_SOCKET * sock,
		  TCP_SOCKET_BUFFER * buffer);

/**
 * Close a GNUnet TCP socket for now (use to temporarily close
 * a TCP connection that will probably not be used for a long
 * time; the socket will still be auto-reopened by the
 * readFromSocket/writeToSocket methods if it is a client-socket).
 **/
void closeSocketTemporarily(GNUNET_TCP_SOCKET * sock);

/**
 * Destroy a socket for good. If you use this socket afterwards,
 * you must first invoke initializeSocket, otherwise the operation
 * will fail.
 **/
void destroySocket(GNUNET_TCP_SOCKET * sock);



#endif
