/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * GNUNET MAIN. This is the server-process that must run on each
 * and every GNUNET node. The code in here initializes the subsystems
 * and then dispatches incomming UDP packets using multiple threads
 * to the handler(s).
 *
 * @file server/gnet.c
 * @author Christian Grothoff
 * @author Larry Waldo
 * @author Tzvetan Horozov
 **/

#include "config.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/stat.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <pthread.h>
 
#include "configuration.h"
#include "util/ipcheck.h"
#include "util/semaphore.h"
#include "util/statuscalls.h"
#include "util/contentdatabase.h"
#include "server/startup.h"
#include "server/handler.h"
#include "server/pingpong.h"
#include "server/keyservice.h" 
#include "server/heloexchange.h"
#include "server/trustservice.h"
#include "knownhosts.h"
#include "lookup.h"
#include "statistics.h"

/* threads and the shared buffers */
static pthread_t GNET_threads_[THREAD_COUNT];

static char GNET_buffers[QUEUE_LENGTH][MESSAGE_SIZE];
static int GNET_bufferQueue_[QUEUE_LENGTH];
static int GNET_bq_firstFree_;
static int GNET_bq_lastFree_;
static int GNET_bq_firstFull_;

static Semaphore * GNET_bufferQueueRead_;
static Semaphore * GNET_bufferQueueWrite_;   
static Mutex GNET_globalLock_;


/**
 * This is the main loop of each thread.
 * It loops *forever* waiting for incomming packets
 * in the packet queue. Then it calls 
 * "handle" (defined in handler.c) on the packet.
 **/
static void threadMain(int id) {
  int current;
  int needed = 1;
  
  while (needed) {
    LOOPPRINT("threadMain");
    semaphore_down(GNET_bufferQueueRead_);
    /* handle buffer entry */
    /* sync with other handlers to get buffer */
    MUTEX_LOCK(&GNET_globalLock_);
    current = GNET_bufferQueue_[GNET_bq_firstFull_++];
    if (GNET_bq_firstFull_ == QUEUE_LENGTH)
      GNET_bq_firstFull_ = 0;
    MUTEX_UNLOCK(&GNET_globalLock_);
    /* end of sync */

    /* handle buffer - now out of sync */    
    handle(id,(MessagePack*)GNET_buffers[current]);

    /* free buffer */
    MUTEX_LOCK(&GNET_globalLock_);
    GNET_bufferQueue_[GNET_bq_lastFree_++] = current;
    if (GNET_bq_lastFree_ == QUEUE_LENGTH)
      GNET_bq_lastFree_ = 0;
    MUTEX_UNLOCK(&GNET_globalLock_);
    /* buffer is back in pool. */
    /* handling done - open buffer for write */    
    semaphore_up(GNET_bufferQueueWrite_);    
  }
} /* end of threadMain */

#if PRINT_CRONDOT
static void printDot(void * unused) {
  print(".");
}
#endif

/**
 * Listen on the given socket and distribute the packets
 * to the UDP handler.
 * This method never returns.
 **/
static void listenAndDistribute(int sock) {
  int current;
  struct sockaddr_in incoming;
  socklen_t addrlen = sizeof(incoming);  
  int size;

  bzero(&incoming,sizeof(struct sockaddr_in));
  while (1) {
    LOOPPRINT("main (1)");
    semaphore_down(GNET_bufferQueueWrite_);
    /* aquire buffer */
    MUTEX_LOCK(&GNET_globalLock_);
    if (GNET_bq_firstFree_ == QUEUE_LENGTH)
      GNET_bq_firstFree_ = 0;
    current = GNET_bufferQueue_[GNET_bq_firstFree_++];    
    MUTEX_UNLOCK(&GNET_globalLock_);
#if PRINT_UDP > 1
    print("UDP: Receiving from socket %d, buffer %d.\n",
	  sock, current);
#endif    
    size = recvfrom(sock, GNET_buffers[current], MESSAGE_SIZE, 0,
		    (struct sockaddr * )&incoming, &addrlen);
    GNUNET_STATISTICS.octets_total_udp_in+=size;
#if PRINT_UDP > 1
    print("Received %d bytes from %d.%d.%d.%d.\n",
	  size,
	  ((*(int*)&incoming.sin_addr) >> 0) & 255, 
	  ((*(int*)&incoming.sin_addr) >> 8) & 255, 
	  ((*(int*)&incoming.sin_addr) >>16) & 255, 
	  ((*(int*)&incoming.sin_addr) >>24) & 255);
#endif
    /* quick test of the packet, if failed, repeat! */
    while ( (size != ((ntohs(((MessagePack*)GNET_buffers[current])->size)) & 
		      (ENCRYPTED_FLAG-1))) ||
	    (size < sizeof(MessagePack)) ) {
      LOOPPRINT("main (2)");
      
#if PRINT_WARNINGS
      print("Received invalid packet (%d != %d) from %d.%d.%d.%d. Discarded.\n",
	    size,
	    (ntohs(((MessagePack*)GNET_buffers[current])->size))
	    & (ENCRYPTED_FLAG-1),
	    ((*(int*)&incoming.sin_addr) >> 0) & 255, 
	    ((*(int*)&incoming.sin_addr) >> 8) & 255, 
	    ((*(int*)&incoming.sin_addr) >>16) & 255, 
	    ((*(int*)&incoming.sin_addr) >>24) & 255);
#endif     
      size = recvfrom(sock, 
		      GNET_buffers[current], 
		      MESSAGE_SIZE, 
		      0,
		      (struct sockaddr * )&incoming, 
		      &addrlen);
      GNUNET_STATISTICS.octets_total_udp_in+=size;
#if PRINT_UDP > 1
    print("Received %d bytes from %d.%d.%d.%d.\n",
	  size,
	  ((*(int*)&incoming) >> 0) & 255, 
	  ((*(int*)&incoming) >> 8) & 255, 
	  ((*(int*)&incoming) >>16) & 255, 
	  ((*(int*)&incoming) >>24) & 255);
#endif
    }
#if PRINT_UDP 
    print("UDP: Received %d bytes from socket %d, buffer %d.\n",
	  size, sock, current);
#endif    
   /* finished with list */
    semaphore_up(GNET_bufferQueueRead_);
    bzero(&incoming,
	  sizeof(struct sockaddr_in));
  }
}

/**
 * The main method of GNUnet. And here is how it works:
 * 1) initialize subsystems 
 *    (configuration, semaphores, key library, output queue)
 * 2) initialize synchronization mechanism for the threads
 * 3) initialize threads, they execute "threadMain"
 *    and query the input queue with incomming packets
 * 4) initialize network (UDP listen)
 * 5) forever: get UDP packet, put it in the queue for the threads
 * 6) [not yet reachable]: close socket on exit
 **/
int main(int argc, char * argv[]) {
  int i;
  int sock;
  FileName configFile = DEFAULT_CONFIG_FILE;

  /* initialize subsystems */
  if (parseCommandLine(argc, argv, (char**)&configFile))
    return 0; /* parse error */

  readConfig(configFile);

  checkCompiler();
  initStatistics();
  initContentDatabase(getContentDirectory());
  initCron();
  initTrustService();
  initTCPIO();
  initStatusCalls(getNetInterfaces(),
		  getMaxNetBPSTotal());
  initAddress();
  initKnownhosts();
  initKeyService();
  initPingPong();
   /* initialize UDP network */
  sock = passivesock(htons(getGNUnetPort()));

  initConnection(); 
  initLookup();
  fullInitLookup();
  initRouting();
  startTCPListener();
  initCronJobs();
  
  /* initialize sync mechanisms for UDP threads */
  GNET_bufferQueueRead_ = new_semaphore(0);
  GNET_bufferQueueWrite_ = new_semaphore(QUEUE_LENGTH);
  create_mutex(&GNET_globalLock_);
  for (i=0;i<QUEUE_LENGTH;i++)
    GNET_bufferQueue_[i]=i;
  GNET_bq_firstFree_ = 0;
  GNET_bq_lastFree_ = 0;
  GNET_bq_firstFull_ = 0;
  
  /* create UDP threads */
  for (i=0;i<THREAD_MIN;i++) 
    pthread_create(&GNET_threads_[i], NULL,
		   (void * (*)(void *)) &threadMain,
		   (void *) &i);  


  /* start control thread */
#if PRINT_CRONDOT
  addCronJob(&printDot,
	     1, 1, NULL);
#endif  
  startCron();

  /* listen & distribute UDP packets to threads */
  listenAndDistribute(sock);
  /* shutdown - this can never happen ... */
  close(sock);
  
  doneTCPIO();
  freeConfig();
  doneStatistics();
  return 0;
} 

/* You have reached the end of GNUnet. You can shutdown your
   computer and get a live now. */
