/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Main handler for incoming packets.
 * @author Christian Grothoff
 * @file server/handler.c
 **/

#include "config.h"
#include "statistics.h"
#include "server/handler.h"
#include "server/connection.h"
#include "server/routing.h"
#include "server/pingpong.h"
#include "server/heloexchange.h"
#include "util/ipcheck.h"
#include <stdio.h>

#define VERIFY_OK_ENCRYPTED   1
#define VERIFY_OK_PLAINTEXT   0
#define VERIFY_REJECT_POLICY -1
#define VERIFY_REJECT_CRC    -2
#define VERIFY_REJECT_NOKEY  -3
#define VERIFY_REJECT_MALF   -4

/**
 * Verify that this message is correct (CRC, sym. encoding).
 * @param pack the message that was received and that should be checked
 * @return VERIFY_XXX depending on the result
 **/
static int verify(MessagePack * pack);


/**
 * Take the HELO message and add the public key to the
 * list of known hosts. Also bind the identity to the
 * connection information provided.
 * @param msg the HELO message
 **/
static int handleHELO(HELO_Message * msg);

/**
 * Signed session key exchange.
 * @param pack the header of the message (packet)
 * @param msg the part that contains the new Sessionkey
 **/
static void handleSKEY(MessagePack * pack, 
		       SKEY_Message * msg);

/**
 * Handle triple-hash query for content. Depending on how we like
 * the sender, lookup or even forward.
 * @param pack the header of the message (packet)
 * @param msg the part that contains the query
 **/
static void handle3QUERY(MessagePack * pack,
			 QUERY_Message * msg);

/**
 * Receive content, do something with it!
 * There are 3 basic possiblilities. Either our
 * node did the request and we should send the result
 * to gproxy via TCP, or the content was requested by
 * another node and we forwarded the request (and thus
 * we now have to fwd the reply) or 3rd somebody just
 * send us some content we did NOT ask for - and we
 * can choose to store it or just discard it.
 * @param pack the header of the message (packet)
 * @param msg the part that contains the content
 **/
static void handleCONTENT(MessagePack * pack, CONTENT_Message * msg);

/**
 * Handle a PING. If it was for us, send back a pong.
 **/
static void handlePING(MessagePack * pack,
		       PINGPONG_Message * msg,
		       int encrypted);

/**
 * Handle a PONG. 
 **/
static void handlePONG(MessagePack * pack,
		       PINGPONG_Message * msg,
		       int encrypted);

/**
 * Handle a HANGUP. 
 **/
static void handleHANGUP(MessagePack * pack,
			 HANGUP_Message * msg);



/**
 * The actual main method of gnet: message dispatch/handling.
 * And here is how it works:
 * 1) check integrity of the message (verify checksum)
 * 2) split message into submessages (eventually only one), and
 *    for each submessage, call the appropriate specific handler.
 *    If the message type is unknown, this part is skipped.
 * @param pack the message that was received. caller frees it on return
 * @param threadId number of the current thread
 **/
void handle(int threadId, MessagePack * pack) {
  BLOCK_LENGTH pos; /* position in the message */
  int veri; /* result from verify */
  MessagePart * part; /* part of the message */
  BLOCK_LENGTH total_len; /* length of message */
  time_t now;
#if PRINT_WARNINGS
  HexName hostName;
  IPPortString ipPort;
#endif

  if (YES == checkIPListed(getNetBlacklist(),
			   pack->sender.senderIP)) {
#if PRINT_WARNINGS > 3
    print("UDP: sender address is blacklisted, dropping\n");
#endif
    return; /* drop packet from blacklisted IP */
  }
   /* verify packet, also decrypts! */  
  veri = verify(pack);
#if PRINT_UDP > 3
  dumpPacket(&((unsigned char*)&pack->parts[0])[0],
	     ntohs(pack->size) - sizeof(MessagePack));
#endif
  switch (veri) {
  case VERIFY_REJECT_MALF:
#if PRINT_POLICY
    print("UDP: received malformed message, dropped\n");
#endif
   return; 
  case VERIFY_REJECT_POLICY:
#if PRINT_POLICY
    print("UDP: verify: reject by policy\n");
#endif
    return; /* denied by policy */
  case VERIFY_REJECT_NOKEY:
#if PRINT_WARNINGS
    hash2hex(&pack->senderIdentity.hashPubKey, &hostName);
    printAddress(&pack->sender, 
		 &ipPort);
    print("No session key established with host %s-%s.\n",
	  &hostName, &ipPort);    
#endif
    return; 
  case VERIFY_REJECT_CRC:
#if PRINT_WARNINGS > 2
    hash2hex(&pack->senderIdentity.hashPubKey, &hostName);
    printAddress(&pack->sender, 
		 &ipPort);
    print("CRC error. Dropping packet from %s-%s.\n",
	  &hostName, &ipPort);    

    printConnectionBuffer();
#endif
    return; /* CRC error, abort */
  default:
    break; /* continue... */
  } /* end switch */
  total_len = ntohs(pack->size) - sizeof(MessagePack);
  pos = 0;
  /* split into submessages, call subhandlers */
  if (veri == VERIFY_OK_PLAINTEXT) {  /* message was not encrypted! */
    while (pos < total_len) {
      LOOPPRINT("handle (1)");
      part = (MessagePart*) &((unsigned char*)&pack->parts[0])[pos];
      pos += ntohs(part->size);
      if ((pos > total_len)) {
#if PRINT_WARNINGS
#if PRINT_UDP
	print("handler.c: received invalid packet\n");
	print("Header size is %d\n.",sizeof(MessagePack));
	print("Invalid packet format (pos: %d - total len: %d)- dropping remaining parts (part size: %d part type: %d).\n",
	      pos, total_len,
	      ntohs(part->size),
	      ntohs(part->requestType));
	print("Packet contained (%d):\n",veri);
	dumpPacket((unsigned char*) &pack->parts[0], 
		   total_len);
#endif
#endif
	return;
      }      
#if PRINT_UDP
      print("UDP: Handling message part (unencrypted) with size %d and type %d.\n",
	    ntohs(part->size), 
	    ntohs(part->requestType));
#endif
      if (ntohs(part->requestType) < GNET_PROTO_MAX)
	GNUNET_STATISTICS.udp_in_counts[ntohs(part->requestType)]++;
      switch (ntohs(part->requestType)) { 	
      case GNET_PROTO_HELO:
	handleHELO((HELO_Message*)part);
	break;
      case GNET_PROTO_SKEY:  
	handleSKEY(pack, (SKEY_Message*)part);
	break;
      case GNET_PROTO_PING:
	handlePING(pack, (PINGPONG_Message*)part, NO);
	break;
      case GNET_PROTO_PONG:
	handlePONG(pack, (PINGPONG_Message*)part, NO);
	break;
      /* This would be the place to add more handlers for NON-encrypted 
	 messages (none planned yet, though) */
      default:
#if PRINT_WARNINGS
	print("Unknown subprotocol (NOT encrypted): %d. Part ignored.\n",
	      ntohs(part->requestType));
#endif
	break;
      }    
    } /* for all parts */
  } else {
#if PRINT_UDP > 42
    dumpPacket(&((unsigned char*)&pack->parts[0])[pos],
	       total_len-pos);
#endif
    /* assert(veri == VERIFY_OK_ENCRYPTED) */
    /* message was encrypted */
    while (pos < total_len) {
      LOOPPRINT("handle (2)");
      part = (MessagePart*) &((unsigned char*)&pack->parts[0])[pos];
      pos += ntohs(part->size);
      if (pos > total_len) {
#if PRINT_WARNINGS
#if PRINT_UDP
	print("Invalid e-packet format (%d - %d) - dropping remaining parts.\n",
	      pos, total_len);
	print("Packet contained (%d):\n",veri);
	dumpPacket((unsigned char*)&pack->parts[0], 
		   total_len);
#endif
#endif
	return;
      }      
      if (ntohs(part->requestType) < GNET_PROTO_MAX)
	GNUNET_STATISTICS.udp_in_counts[ntohs(part->requestType)]++;
      switch (ntohs(part->requestType)) {		
      case GNET_PROTO_HELO:
	handleHELO((HELO_Message*)part); 
	break;
      case GNET_PROTO_SKEY:
	handleSKEY(pack, (SKEY_Message*)part);
	break;
      case GNET_PROTO_PING:
	handlePING(pack, (PINGPONG_Message*)part, YES);
	break;
      case GNET_PROTO_PONG:
	handlePONG(pack, (PINGPONG_Message*)part, YES);
	break;
      case GNET_PROTO_3QUERY:
	handle3QUERY(pack, (QUERY_Message*)part);
	break;
      case GNET_PROTO_CONTENT:
	handleCONTENT(pack, (CONTENT_Message*)part);
	break;       
     case GNET_PROTO_TIMESTAMP:
	time(&now);
	if (ntohl(((TIMESTAMP_Message*)part)->timeStamp) < now) {
	  /* packet has timed out, drop the rest of the packet */
#if PRINT_WARNINGS
	  print("WARNING: Invalid timestamp, dropping rest of packet\n");
#endif
	  pos = total_len; 
	}	
	break;
      case GNET_PROTO_SEQUENCE:
	if (SYSERR == checkSequenceNumber
	    (&pack->senderIdentity,
	     ntohl(((SEQUENCE_Message*)part)->sequenceNumber))) {
	  /* invalid number, drop the rest of the packet */
	  pos = total_len; 
	}
	break;
      case GNET_PROTO_NOISE:
	/* nothing to do, just ignore */
	break;
      case GNET_PROTO_HANGUP:
	handleHANGUP(pack, (HANGUP_Message*)part);
	break;
       
      /* This would be the place to add more handlers for ENCRYPTED messages 
	 (planned: credit trading?, routing information?
	 please always discuss on gnunet-developers first! ) */
      default:
#if PRINT_WARNINGS
	print("Unknown subprotocol (encrypted): %d - part ignored!\n",
	      ntohs(part->requestType));
#endif
	break;
      }    
    } /* for all parts */
  } /* end of if-encrypted */
} /* end of handle */

/**
 * Verify that this message is correct (CRC, sym. encoding).
 * How it works:
 * 0) check policy: do we currently accept anything from that source?
 * 1) IF highest bit in size is set, decrypt:
 *    decrypt packet with key from local database
 *    (senderPort + senderIP => symmetric key)
 *    --- if not encrypted: return 2 at the end!
 *    Important: also the checksum is encoded!
 * 2) check CRC checksum for decrypted packet (rest!)
 *    If wrong, return 0.
 * @param pack the message that was received and that should be checked
 * @return VERIFY_XXX depending on the result
 **/
static int verify(MessagePack * pack) {
  int returnValue = VERIFY_OK_PLAINTEXT;
  unsigned char decr[MESSAGE_SIZE];
  int size;
#if PRINT_WARNINGS
  HexName hostName;
#endif
  
  /* check size */
  size = ntohs(pack->size) & (~ENCRYPTED_FLAG);
  if ((size < sizeof(MessagePack)) || (size > MESSAGE_SIZE) )
    return VERIFY_REJECT_MALF;

  /* check policy (is sender host blocked? is load ok?) */
  if (SYSERR == incomingCheck(&pack->senderIdentity)) {
    return VERIFY_REJECT_POLICY; /* error: packet denied by policy */
  }
  /* decrypt packet */
  if ((ntohs(pack->size) & ENCRYPTED_FLAG) > 0) {          
    pack->size = htons(size);
    /* decrypt msg */
    size = decryptFromHost((BLOWFISHEncryptedData*)&pack->parts[0],
			   size - sizeof(MessagePack),
			   &pack->senderIdentity,
			   decr);
    if (size < 0) {
#if PRINT_WARNINGS
      hash2hex(&pack->senderIdentity.hashPubKey, &hostName);
      print("Can not decrypt packet from host %s: no session key? (%d).\n",
	    &hostName,
	    size);
#endif
      return VERIFY_REJECT_NOKEY;
    }
    memcpy(&pack->parts[0],
	   decr, size);
    returnValue = VERIFY_OK_ENCRYPTED;
  } else
    size -= sizeof(MessagePack);
    
  /* Check CRC vs. information in header */
  assert(size >= 0,
	 "very bad: size negative! This should be impossible.");
#if DISABLE_CRC_CHECK == 0
  if (ntohl(pack->checkSum) != crc32N(&pack->parts[0],size)) {
    /* invalid CRC */
#if PRINT_WARNINGS 
    print("Computed CRC over %d bytes\n",
	  size);
    printConnectionBuffer();
    print("CRC ERROR: %d (pack) %d (msg)\n", 
	  ntohl(pack->checkSum), 
	  crc32N(&pack->parts[0],size));
#endif
    return VERIFY_REJECT_CRC; 
  }
#endif
  return returnValue; 
}


/* ******************* Methods to handle the different requests ************* */

/**
 * Take the HELO message and add the public key to the
 * list of known hosts. Also bind the identity to the
 * connection information provided.
 * @return OK if the processing went find, SYSERR if not
 **/
static int handleHELO(HELO_Message * msg) {
#if PRINT_HELO
  IPPortString ipPort;
#endif

  if (ntohs(msg->header.size) != sizeof(HELO_Message))
    return SYSERR;
 
#if PRINT_HELO
  printAddress(&msg->body.sender,
	       &ipPort);
  print("handleHELO: Handling HELO from %s.\n",
	&ipPort);    
#endif
  receivedHELO(&msg->body);
  return OK;
}

/**
 * Signed session key exchange.
 **/
void handleSKEY(MessagePack * pack, SKEY_Message * msg) {
#if PRINT_SKEY
  IPPortString ipPort;
#endif
  
  if (ntohs(msg->header.size) != sizeof(SKEY_Message))
    return;
#if PRINT_SKEY
  printAddress(&pack->sender,
	       &ipPort);
  print("SKEY: Handling SKEY from %s.\n",
	&ipPort);    
#endif  
  acceptSessionKey((HostIdentity *) &pack->senderIdentity, 
		   msg);  
}

/**
 * Handle query for content. Depending on how we like
 * the sender, lookup or even forward.
 **/
static void handle3QUERY(MessagePack * pack, 
			 QUERY_Message * msg) {
  QUERY_POLICY qp;
#if PRINT_QUERY
  IPPortString ipPort;
  HexName query;
#endif
  
  if (ntohs(msg->header.size) != sizeof(QUERY_Message))
    return;
  if (((signed int)ntohl(msg->ttl)) <= 0)
    return;
  notifyActive(&pack->senderIdentity, -1);
 
#if PRINT_QUERY
  hash2hex(&msg->query, &query);
  printAddress(&pack->sender,&ipPort);
  print("UDP: Handling 3QUERY for %s from %s with priority %u and ttl %d.\n",
	&query, 
	&ipPort,
	ntohl(msg->priority), 
	ntohl(msg->ttl));
#endif
 /* first: query policy */
  qp = evaluateQuery(&pack->senderIdentity,
		     ntohl(msg->priority));  
  if ((qp & QUERY_DROPMASK) == 0)
    return; /* straight drop. */
  /* if ok: do query */
  exec3Query(qp, msg, NULL);
}
 
/**
 * Handle a PING. If it was for us, send back a pong.
 **/
static void handlePING(MessagePack * pack,
		       PINGPONG_Message * msg,
		       int encrypted) {
  if (ntohs(msg->header.size) != sizeof(PINGPONG_Message) )
    return;
  if (!hostIdentityEquals(&ROUTING_localhost_,
			  &msg->receiver))
    return; /* not for us */
#if PRINT_PING
  print("PING: received ping (%d)\n",
	encrypted);
#endif
  if (YES == encrypted)
    notifyActive(&pack->senderIdentity,
		 0);
  msg->header.requestType = htons(GNET_PROTO_PONG);
  if (NO == encrypted) {
    GNUNET_STATISTICS.udp_out_counts[GNET_PROTO_PONG]++;
    sendToHost(&pack->sender,
	       &pack->senderIdentity,
	       sizeof(PINGPONG_Message),
	       msg,
	       PLAINTEXT_FLAG,
	       crc32N(msg, sizeof(PINGPONG_Message)));
  } else {
    unicast(&msg->header,
	    &pack->senderIdentity,
	    5);	    
  }
}

/**
 * Handle a PONG.
 **/
static void handlePONG(MessagePack * pack,
		       PINGPONG_Message * msg,
		       int encrypted) {
  if (ntohs(msg->header.size) != sizeof(PINGPONG_Message) )
    return;
#if PRINT_PING
  print("PING: received pong (%d)\n",
	encrypted);
#endif
  if (YES == encrypted)
    notifyActive(&pack->senderIdentity,
		 msg->challenge);
  else
    pongReceived(&msg->receiver,
		 msg->challenge);
}

/**
 * Handle a HANGUP.
 **/
static void handleHANGUP(MessagePack * pack,
			 HANGUP_Message * msg) {
  BufferEntry * be;

  if (ntohs(msg->header.size) != sizeof(HANGUP_Message) )
    return;
  if (!hostIdentityEquals(&pack->senderIdentity,
			  &msg->sender))
    return;
  be = lookForHost(&msg->sender);
  if (be != NULL)
    shutdownConnection(be);
  MUTEX_UNLOCK(&be->sem);  
}

/**
 * Receive content, do something with it!
 * There are 3 basic possiblilities. Either our
 * node did the request and we should send the result
 * to gproxy via TCP, or the content was requested by
 * another node and we forwarded the request (and thus
 * we now have to fwd the reply) or 3rd somebody just
 * send us some content we did NOT ask for - and we
 * can choose to store it or just discard it.
 * <p>
 * handleCONTENT leaves the possibilities gproxy and
 * indirection to "useContent". The policy module
 * is considered for local storage.
 **/
static void handleCONTENT(MessagePack * pack, 
			  CONTENT_Message * msg) {  
  int prio;

#if PRINT_CONTENT
  HexName cont;
#endif

  if (ntohs(msg->header.size) != sizeof(CONTENT_Message))
    return;
#if PRINT_CONTENT
  hash2hex(&msg->hash, 
	   &cont);
  print("UDP: Handling CONTENT %s.\n",
	&cont);    
#endif
  notifyActive(&pack->senderIdentity, -1);
  prio = useContent(&pack->senderIdentity,
		    &msg->content, 
		    &msg->hash);
  if (pack != NULL) /* never evaulate loopback-content! */
    evaluateContent(&pack->senderIdentity,
		    &msg->content,
		    &msg->hash, 
		    prio);
}


/* end of handler.c */
