/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Generic cron-jobs. Currently contains only HELO
 * exchanges.
 * @author Christian Grothoff
 * @file server/heloexchange.c
 **/

#include "config.h"
#include "server/heloexchange.h"
#include "util/ipcheck.h"
#include "knownhosts.h"
#include <unistd.h>
#include <math.h>
#include <signal.h>
#include "statistics.h"
#include "server/pingpong.h"

#define TCP_HTTP_PORT 80
#define HTTP_URL "http://"
#define GET_COMMAND "GET %s\n"

/* ************* internal Methods **************** */

/**
 * Tell everybody we are there...
 **/
static void broadcastHELO(void * unused);
 
/**
 * Forward HELOs from all known hosts to all known hosts.
 **/
static void forwardHELO(void * unused);

/* ******************** CODE ********************* */

/**
 * Catch sigpipe on send, and do nothing
 **/
#ifndef LINUX
static void catcher(int sig) {
#if PRINT_INFO
  fprintf(stderr, "signal %d caught\n", sig);
#endif
  signal(sig, catcher);
}
#endif

/**
 * initialize a few cron jobs. Must be called after
 * initcron (!).
 **/
void initCronJobs() {
  addCronJob(&broadcastHELO,
	     30,60*5,
	     NULL); /* every five minutes */
  addCronJob(&downloadHostlist,
	     15,0,
	     NULL); /* after 15s */
  if (YES == isHELOEnabled())
    addCronJob(&forwardHELO,
	       60, 7*60,
	       NULL); /* seven minutes: exchange */
#if PRINT_HELO
  else
    print("HELO forwarding disabled!\n");
#endif
}


/**
 * We have received a HELO. Verify (signature, integrity,
 * ping-pong) and store identity if ok.
 **/
void receivedHELO(HELO_Body * body) {
  HELO_Body * copy;
  HostAddress address;
  PublicKey pubKey;
  
  if (YES == checkIPListed(getNetBlacklist(),
			   body->sender.senderIP)) {
#if PRINT_WARNINGS
    print("Received HELO from blacklisted host. Dropping.\n");
#endif
    return; /* we don't like this IP (blacklisted) */  
  }
  if (SYSERR == verifyHELO(body)) {
#if PRINT_WARNINGS
    print("HELO message received invalid. Dropping.\n");
#endif
    return; /* message invalid */  
  }
  if ( (OK == identity2Key(&body->senderIdentity,
			   &pubKey)) &&
       (OK == getConnectionInformation(&body->senderIdentity,
				       &address) ) ) {
    if ( (0 == memcmp(&address, &body->sender, sizeof(HostAddress))) &&
	 (0 == memcmp(&pubKey, &body->publicKey, sizeof(PublicKey))) ) {
      /* ok, we've seen this one exactly like this before (at most the
	 TTL has changed); thus we can 'trust' it without playing
	 ping-pong */
      bindAddress(body);
      return;
    }
  }
  copy = xmalloc(sizeof(HELO_Body),
		 "malloc for pingAction");
  memcpy(copy, body,
	 sizeof(HELO_Body));
  pingAction(&body->sender,
	     &body->senderIdentity,
	     (void (*)(void*))&bindAddress, 
	     copy);
}

/**
 * Download hostlist from the web.
 **/
void downloadHostlist(void * unused) {
  unsigned short port;
  char * url;
  char * hostname;
  char * filename;
  int curpos;
  struct hostent *ip_info;
  struct sockaddr_in soaddr;
  int sock;
  int ret;
  char * command;
  HELO_Body body;
  time_t start;

#if PRINT_CRON
  print("CRON: enter downloadHostlist\n");
#endif
  port = TCP_HTTP_PORT;
  url = getHostlistURL();
  if (url == NULL) {
#if PRINT_CRON
    print("CRON: exit downloadHostlist (error: URL not specified)\n");
#endif
    return;
  }
  url = strdup(url);
  curpos = 0;
  if (0 != strncmp(HTTP_URL, url, strlen(HTTP_URL)) ) {
#if PRINT_WARNINGS
    print("WARNING: invalid URL %s (must begin with %s)\n", url, HTTP_URL);
#endif
#if PRINT_CRON
    print("CRON: exit downloadHostlist (error: not http)\n");
#endif
    return;
  }
  curpos = strlen(HTTP_URL);
  hostname = &url[curpos];
  while ( (curpos < strlen(url)) &&
	  (url[curpos] != '/') )
    curpos++;
  if (curpos == strlen(url))
    filename = strdup("/");
  else {
    filename = strdup(&url[curpos]);
    url[curpos] = '\0'; /* terminator for hostname */
  }
  ip_info = gethostbyname(hostname);
  if (ip_info == NULL) {
#if PRINT_WARNINGS
    print("WARNING: could not download hostlist, host %s unknown\n", 
	  hostname);
#endif
    free(filename);
    free(url);
#if PRINT_CRON
    print("CRON: exit downloadHostlist (error: can not resolve hostname)\n");
#endif
    return;
  }
  sock = socket(AF_INET, SOCK_STREAM, 0);
  if (sock < 0) {
#if PRINT_WARNINGS
    print("WARNING: could not open socket for hostlist download\n");
    perror("socket");
#endif
    free(filename);
    free(url);
#if PRINT_CRON
    print("CRON: exit downloadHostlist (error: can not open socket)\n");
#endif
    return;
  }
  soaddr.sin_family = AF_INET;
  soaddr.sin_addr.s_addr = ((struct in_addr*)(ip_info->h_addr))->s_addr;
  soaddr.sin_port=htons(TCP_HTTP_PORT);
  if (connect (sock, (struct sockaddr*)&soaddr, sizeof(soaddr)) < 0) {
#if PRINT_WARNINGS
    print("WARNING: failed so send HTTP request to host %s (%d - %d)\n",
	  hostname,
	  curpos, sock);
    perror("send");
#endif
    free(filename);
    free(url);
#if PRINT_CRON
    print("CRON: exit downloadHostlist (error: can not connect)\n");
#endif
    return;
  }
  
  command = malloc(strlen(filename) + strlen(GET_COMMAND));
  sprintf(command, GET_COMMAND, filename);
  free(filename);
  curpos = strlen(command)+1;
#ifndef LINUX
  if ( SIG_ERR == signal(SIGPIPE, SIG_IGN)) {
    if (SIG_ERR == signal(SIGPIPE, catcher))
      fprintf(stderr, "WARNING: could not install handler for SIGPIPE!\n"\
                      "Attempting to continue anyway.");
 }
  curpos = send(sock, command, curpos, 0);
#else
  curpos = send(sock, command, curpos, MSG_NOSIGNAL);
#endif
  if (curpos != strlen(command)+1) {
#if PRINT_WARNINGS
    print("WARNING: failed so send HTTP request %s to host %s (%d - %d)\n",
	  command, hostname,
	  curpos, sock);
    perror("send");
#endif
    free(url);
    free(command);
#if PRINT_CRON
    print("CRON: exit downloadHostlist (error)\n");
#endif
    return;
  }
  free(url);
  free(command);
#if PRINT_HELO
  print("Receiving hostlist...");
#endif
  time(&start);
  while (1) {
    if (start + 30 < time(NULL))
      break; /* exit after 30s */
    curpos = 0;
    while (curpos < sizeof(HELO_Body)) {
      if (start + 30 < time(NULL))
	break; /* exit after 30s */
      ret = recv(sock, 
		 &((char*)&body)[curpos],
		 sizeof(HELO_Body)-curpos,
		 MSG_DONTWAIT);
      if (ret == 0)
	break;
      if (ret < 0) 
	if (errno != EAGAIN)
	  break;
	else
	  ret = 0;
      curpos += ret;
    }
    if (curpos != sizeof(HELO_Body))
      break;
#if PRINT_HELO
      print("!");
#endif
    receivedHELO(&body);
  }
#if PRINT_HELO
  print("\n");
#endif
  close(sock);  
#if PRINT_CRON
    print("CRON: exit downloadHostlist (success, %d seconds before timeout)\n",
	  start + 30 - time(NULL));
#endif
}

typedef struct {
  /* the CRC of the message */
  int crc;
  /* the HELO message */
  HELO_Message m;
  /* send the HELO in 1 out of n cases */
  int n;
} SendData;

static void broadcastHelper(HostIdentity * hi,
			    SendData * sd) {
  if (randomi(sd->n) == 0) {
    GNUNET_STATISTICS.udp_out_counts[GNET_PROTO_HELO]++;
    sendToHost(NULL,
	       hi,
	       sizeof(HELO_Message),
	       &sd->m.header,
	       PLAINTEXT_FLAG,
	       sd->crc);
  }
}


/**
 * Tell a couple of random hosts on the currentKnownHost list 
 * that we are exist...
 **/
static void broadcastHELO(void * unused) {
  int j;
  SendData sd;

#if PRINT_CRON
  print("CRON: enter broadcastHELO\n");
#endif
  j = forEachHost(NULL, 0, NULL); /* just count */
  if (j <= 1)
    return; /* no point in trying... */
  sd.n = j/((int) log(j+1)+1); /* send the HELO to log(known host) hosts */
  createHELO(&sd.m);
  bindAddress(&sd.m.body);
  sd.crc = crc32N(&sd.m, 
		  sizeof(HELO_Message));

  forEachHost((void (*)(HostIdentity*, void *))&broadcastHelper,
	      0, /* ignore blacklisting */
	      &sd);
#if PRINT_CRON
  print("CRON: exit broadcastHELO\n");
#endif
}

/**
 * Forward HELOs from all known hosts to all connected hosts.
 **/
static void forwardHELOHelper(HostIdentity * identity,
			      int * count) {
  HELO_Message fi;
  FileName fn;
  int i;
  time_t now;
  int ran;
  HexName hexName;

#if PRINT_HELO > 1
  printf("FORWARDING HELOs %d\n",
	 KNOWNHOSTS_currentKnownHosts);
#endif
  time(&now);
  fi.header.size = htons(sizeof(HELO_Message));
  fi.header.requestType = htons(GNET_PROTO_HELO); 
  fn = xmalloc(strlen((char*)getHostsDirectory())+sizeof(HexName)+1,
	       "forwardHELOHelper: filename");

  hash2hex(&identity->hashPubKey,
	   &hexName);
  buildFileName(getHostsDirectory(),
		&hexName,
		fn);
  bzero(&fi.body, sizeof(HELO_Body));
  if (sizeof(HELO_Body) != readFile(fn, 
				    sizeof(HELO_Body), 
				    &fi.body)) {
#if PRINT_WARNINGS
    print("WARNING: malformed HELO stored under %s\n",
	  fn);
#endif
    return;
  }      
  /* do not forward expired HELOs */
  if (ntohl(fi.body.expirationTime) < now) {
    /* remove HELOs that expired over 1 week ago */ 
    if (ntohl(fi.body.expirationTime) < now - 60*60*24*7) {
#if PRINT_HELO
      print("HELO-EXCHANGE: Removing ancient file: %s (from %d now is %d)\n",
	    fn, ntohl(fi.body.expirationTime), now);
#endif
      unlink(fn);
      delHostFromKnown(identity);
    }
    return;
  }
  /* remove HELOs for blacklisted IPs */ 
  if (YES == (checkIPListed(getNetBlacklist(),
			    fi.body.sender.senderIP))) {
#if PRINT_HELO > 2
    printf("HELOEXCHANGE: Removing blacklisted file: %s\n",
	   fn);
#endif
    unlink(fn); 
    delHostFromKnown(identity);
    return;
  }
  /* everything else: forward with a certain probability */
  ran = randomi(*count);
  if (ran <= log(*count+1)) 
    broadcast(&fi.header, 0);
  xfree(fn,"forwardHelo: filename");
}

/**
 * Forward HELOs from all known hosts to all connected hosts.
 **/
static void forwardHELO(void * unused) {
  int count;

#if PRINT_CRON
  print("CRON: enter forwardHELO\n");
#endif
  count = forEachHost(NULL, 0, NULL);
  forEachHost((void (*)(HostIdentity*, void *))&forwardHELOHelper,
	      0, /* ignore blacklisting */
	      &count);
#if PRINT_CRON
  print("CRON: exit forwardHELO\n");
#endif
}

/* end of heloexchange.c */
