/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * This module keeps track of host-identity - address (IP) translation.
 * Basically a mapping HostAddress <==> HostIdentity <==> Public Key
 * with signature and expiration date for the HostAddress.
 * <p>
 * This file basically is all about the HELO_Message.
 * @file server/identity.c
 * @author Christian Grothoff
 * @author Tzvetan Horozov
 **/

#include "config.h"
#include "server/identity.h"
#include "util/ipcheck.h"
#include "statistics.h"

#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <sys/types.h>
#include <netdb.h>
#include <net/if.h>
#include <stdlib.h>
#ifdef SOLARIS
#include <sys/sockio.h>
#endif


static HostAddress myAddress;

static char * networkIdDirectory;

static void cronRefreshAddress(void * unused) {
#if PRINT_CRON
    print("CRON: enter cronRefreshAddress\n");
#endif
   getAddress(&myAddress);  
#if PRINT_CRON
    print("CRON: exit cronRefreshAddress\n");
#endif
 }

/** 
 * Initialize identity module. Requires configuration.
 **/
void initAddress() {
  networkIdDirectory 
    = getHostsDirectory();  
  if (SYSERR == getAddress(&myAddress)) {
    fprintf(stderr,
	    "Could not find IP for this host. Please provide the IP in the configuration file.\n");
    exit(-1);
  }
  addCronJob(&cronRefreshAddress, 120, 120,
	     NULL);
}

/**
 * Bind a host identity (IP+port) to a hostId.
 * @param msg the verified (!) HELO message
 **/
void bindAddress(HELO_Body * msg) {
  HexName fil;
  FileName fn;
  HELO_Body oldMsg;
#if PRINT_HELO
  HexName hostName;
  IPPortString ipPort;
#endif

  hash2hex(&msg->senderIdentity.hashPubKey,
	   &fil);
  fn = (FileName) xmalloc(strlen((char*)networkIdDirectory) + 
			  sizeof(HexName) + 1,
			  "bindAddress: filename");
  buildFileName(networkIdDirectory,
		&fil,
		fn);    
  writeFile(fn,
	    msg, 
	    sizeof(HELO_Body),
	    "644");    
  if (sizeof(HELO_Body) == readFile(fn,
				    sizeof(HELO_Body),
				    &oldMsg)) {
    if (ntohl(oldMsg.expirationTime) > ntohl(msg->expirationTime)) 
      return; /* have newer HELO in stock */    
  }
  xfree(fn, "bindAddress: filename");
  cronScanDirectoryDataHosts(NULL);
  
#if PRINT_HELO
  hash2hex(&msg->senderIdentity.hashPubKey,
	   &hostName);
  printAddress(&msg->sender, 
	       &ipPort);
  print("HELO: Bound identity %s to %s\n",
	&hostName, &ipPort);
#endif 
}

/**
 * Create the signed HELO message for this host.
 * @param msg the address where to write the HELO message to
 **/
void createHELO(HELO_Message * msg) {
  PublicKey * myKey;
  time_t expires;
  
 /* allocate memory for HELO */
  myKey = getPublicHostkey();
  /* write signed header */
  time(&expires);
  expires += 60 * 60 * getHeloExpiration(); 
  msg->body.expirationTime = htonl(expires);
  getHostIdentity(myKey, 
		  &msg->body.senderIdentity);
  memcpy(&msg->body.sender,
	 &myAddress,
	 sizeof(HostAddress));

  /* sign sender, senderIdentity and expires */
  if (SYSERR == signData(&msg->body.sender,
			 sizeof(HostAddress)+sizeof(HostIdentity)+sizeof(time_t),
			 &msg->body.signature)) {
    print("ERROR: signData failed!\n Trying to continue.");
  }
  /* write unsigned header */
  msg->header.size = htons(sizeof(HELO_Message));
  msg->header.requestType = htons(GNET_PROTO_HELO);
  /* add public key after signature */
  memcpy(&msg->body.publicKey,
	 myKey,
	 sizeof(PublicKey));
  /* check created HELO */
#if EXTRA_CHECKS
  if (verifyHELO(&msg->body) != OK) 
    errexit("Oops, HELO creation failed!\nExiting...\n");
#endif
}

/**
 * Obtain the public key of a known host.
 * @param hostId the host id
 * @param result where to store the result
 * @returns SYSERR on failure, OK on success
 **/
int identity2Key(HostIdentity *  hostId,
		 PublicKey * result) {  
  HexName fil;
  FileName fn;
  HELO_Body buffer;
  
  hash2hex(&hostId->hashPubKey,
	   &fil);
  fn = (FileName) xmalloc(strlen((char*)networkIdDirectory) + 
			  sizeof(HexName) + 1,
			  "identity2key: filename");
  buildFileName(networkIdDirectory,
		&fil,
		fn);
  if (-1 == readFile(fn, 
		     sizeof(HELO_Body), 
		     &buffer)) {
    xfree(fn,
	  "identity2key: filename (1)");
    return SYSERR;  
  }
  xfree(fn,
	"identity2key: filename (2)");
  memcpy(result,
  	 &buffer.publicKey,
  	 sizeof(PublicKey));	   
  return OK;
}

/**
 * Resolve identity (find matching connection)
 * @param hostId the host id
 * @param result the connection information that was bound to that identity or NULL if no information is available
 * @return SYSERR on failure, OK on success
 **/
int getConnectionInformation(HostIdentity *  hostId,
			     HostAddress * result) {
  HexName fil;
  FileName fn;
  HELO_Body buffer;
  
  hash2hex(&hostId->hashPubKey,
	   &fil);
  fn = (FileName) xmalloc(strlen((char*)networkIdDirectory) + 
			  sizeof(HexName) + 1,
			  "getConnectionInformation: filename");
  buildFileName(networkIdDirectory,
		&fil,
		fn);

  if (-1 == readFile(fn, 
		     sizeof(HELO_Body), 
		     &buffer)) {
#if PRINT_WARNINGS
    print("No connection information for host available (%s)!\n",
	  fn);
#endif
    xfree(fn,
	  "getConnectionInformation: filename (1)");
    return SYSERR;
  }
  xfree(fn,
	"getConnectionInformation: filename (1)");
  memcpy(result, 
	 &buffer.sender,
	 sizeof(HostAddress));
  return OK;
}

/**
 * Obtain the identity information for the current node
 * (connection information), conInfo.
 * @return SYSERR on failure, OK on success
 **/
static int getAddressFromHostname(HostAddress * identity) {
  char * hostname;
  struct hostent * ip;
#if PRINT_HELO
  IPString ipStringDbg;
#endif  

  hostname = getenv("HOSTNAME");
  if (hostname == NULL) {
    fprintf(stderr,
	    "Environment variable HOSTNAME not set!\n");
    return SYSERR;
  }
  ip = gethostbyname(hostname);
  if (ip == NULL) {
#if PRINT_WARNINGS
    print("Couldn't find server '%s' (h_errno=%d)",
	  hostname, h_errno);
#endif
    return SYSERR;
  }
  assert(ip->h_addrtype == AF_INET,
    "getAddress: h_addrtype is not AF_INET!?");
  identity->senderPort = htons(getGNUnetPort());
  memcpy(&identity->senderIP,
	 &((struct in_addr*)ip->h_addr_list[0])->s_addr,
	 sizeof(struct in_addr));
#if PRINT_HELO
  printIP(*(int*)&identity->senderIP,
          &ipStringDbg);
  print("HELO: getAddress (old): my address is %s\n",
        &ipStringDbg);
#endif
  return OK;
}

#define MAX_INTERFACES 16
static int getAddressFromIOCTL(HostAddress * identity) {
  struct ifreq ifr[MAX_INTERFACES];
  struct ifconf ifc;
  int sockfd,ifCount;
  int i;

  sockfd = socket(PF_INET, SOCK_DGRAM, 0);
  if (sockfd == -1)
    return SYSERR;
  ifc.ifc_len = sizeof(ifr);
  ifc.ifc_buf = (char*)&ifr;
  
  if (ioctl(sockfd,SIOCGIFCONF,&ifc)==-1) {
    print("Could not obtain IP with ioctl\n");
    close(sockfd);
    return SYSERR;
  }
  ifCount=ifc.ifc_len/sizeof(struct ifreq);
  
  for(i=0;i<ifCount;i++){
    LOOPPRINT("getAddressFromIOCTL (1)"); 
    if (ioctl(sockfd, SIOCGIFADDR, &ifr[i]) != 0)
       continue;

    if (ioctl(sockfd, SIOCGIFFLAGS, &ifr[i]) != 0)
       continue;

    if (!(ifr[i].ifr_flags & IFF_UP))
       continue;

    if (strcmp((char*) getNetInterfaces(),
	       (char*) ifr[i].ifr_name) != 0)
      continue;

    identity->senderPort = htons(getGNUnetPort());
    memcpy(&identity->senderIP,
	   &(((struct sockaddr_in *)&ifr[i].ifr_addr)->sin_addr),
	   sizeof(struct in_addr));
    close(sockfd);
    return OK;
  }

  for(i=0;i<ifCount;i++){
    LOOPPRINT("getAddressFromIOCTL (2)"); 
    if (ioctl(sockfd, SIOCGIFADDR, &ifr[i]) != 0)
       continue;

    if (ioctl(sockfd, SIOCGIFFLAGS, &ifr[i]) != 0)
       continue;

    if (!(ifr[i].ifr_flags & IFF_UP))
       continue;

    if (strncmp("lo", (char*) ifr[i].ifr_name, 2) == 0)
      continue;

    identity->senderPort = htons(getGNUnetPort());
    memcpy(&identity->senderIP,
	   &(((struct sockaddr_in *)&ifr[i].ifr_addr)->sin_addr),
	   sizeof(struct in_addr));
    close(sockfd);
    return OK;
  }

  close(sockfd);
  print("Could not obtain IP for interface %s with ioctl\n",
	getNetInterfaces());
  return SYSERR;
}

/**
 * Obtain the identity information for the current node. DO NOT FREE!
 * @return SYSERR on error, OK on success
 **/
int getAddress(HostAddress * identity){
  IPAddress ipString;
  struct in_addr sockInfo;
#if PRINT_HELO
  IPString ipStringDbg;
#endif
  int retval;
  struct hostent *ip; /* for the lookup of the IP in gnunet.conf */

  identity->protocol = htons(UDP_PROTOCOL_NUMBER);
  retval = SYSERR;
  ipString = getIP();
  identity->senderPort = htons(getGNUnetPort());
  if (ipString == NULL) {
    if (OK == getAddressFromIOCTL(identity))
      retval = OK;
    else 
      retval = getAddressFromHostname(identity);
  } else {
    /*
     * inet_aton(ipString,&sockInfo);  
     * memcpy(&identity->senderIP,((struct in_addr*)&sockInfo),
     * 	      sizeof(struct in_addr));
     * retval = OK;
     */
    ip = gethostbyname (ipString);
    if (ip == NULL) {
      fprintf (stderr, "Couldn't resolve '%s' (h_errno=%d)",
	       ipString, h_errno);
      retval = SYSERR;
    }
    else {
      assert (ip->h_addrtype == AF_INET,
	      "getAddress: h_addrtype is not AF_INET!?");
      memcpy (&identity->senderIP,
	      &((struct in_addr*) ip->h_addr_list[0])->s_addr,
	      sizeof(struct in_addr));
#if PRINT_HELO
      print ("Resolved '%s' to '%s'\n", ipString, 
	     inet_ntoa (identity->senderIP));
#endif
      retval = OK;
    }
  }
  if (retval == OK) {
    if (YES == checkIPListed(getNetBlacklist(),
			     identity->senderIP)) 
      errexit("FATAL: my own IP address is blacklisted in my local configuration!\n");      
  }
#if PRINT_HELO
  printIP((*(int*)&identity->senderIP),
	  &ipStringDbg);
  print("HELO: Identity address (new): %s\n",
	&ipStringDbg);
#endif
  return retval;
}

/**
 * Send a packet to a given host. This method does the actual transfer.
 * @param conInfoStar the HostAddress of the receiver, if NULL,
 *        sendToHost will look it up
 * @param hostId the id of the target host
 * @param size the size of the content
 * @param content the actual data (after the header). Encryption is
 *        NOT performed by this method! The header (MessagePack) is
 *        added, the content should consist of all message-parts.
 * @param encryptedFlag is the data encrypted (subprotocol), 
 *        or-ed with the size of the packet in the header
 * @param crc is the checksum of the unencrypted data (minus the header)
 **/
void sendToHost(HostAddress * conInfoStar,
		HostIdentity * hostId,
		BLOCK_LENGTH size,
		void * content,
		int encryptedFlag,		
                int crc) {
  MessagePack * pack;
  int sock;
  HostAddress conInfo;
  struct sockaddr_in sin;	/* an Internet endpoint address	*/
#if PRINT_WARNINGS
  HexName hostName;
#endif
#if PRINT_UDP
  IPString ipString;
#endif
  pack = (MessagePack *) xmalloc(sizeof(MessagePack) + size,
				 "sendToHost: messagePack");
  pack->reserved = 0;
  memcpy(&pack->parts,
	 content,
	 size);
  pack->size = htons((size+sizeof(MessagePack)) | encryptedFlag);
  memcpy(&pack->sender,
	 &myAddress,
	 sizeof(HostAddress));
  getHostIdentity(getPublicHostkey(),
		  &pack->senderIdentity);
  pack->checkSum = htonl(crc); 
  if (conInfoStar == NULL) {
    conInfoStar = &conInfo;
    if (SYSERR == getConnectionInformation(hostId, &conInfo)) {
#if PRINT_WARNINGS
      hash2hex(&hostId->hashPubKey,
	       &hostName);
      print("Could not get connection information for host %s! Message not sent!\n",
	    &hostName);
#endif
      xfree(pack,"sendToHost: pack (1)");
      return; /* could not send: no con info!*/
    }
  }
  /* finally: UDP send! */  
  memset(&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = conInfoStar->senderPort;
  memcpy(&sin.sin_addr, &conInfoStar->senderIP,
	 sizeof(struct in_addr));
  sock = socket(PF_INET, SOCK_DGRAM, UDP_PROTOCOL_NUMBER);
  if (connect(sock, 
	      (struct sockaddr *)&sin, 
	      sizeof(sin)) == 0) {
#if PRINT_UDP
    printIP(*(int*)&sin.sin_addr,
	    &ipString);
    print("UDP: Sending %d bytes to %s:%d.\n",
	  size+sizeof(MessagePack), 
	  &ipString,
	  ntohs(conInfoStar->senderPort));
#endif
    GNUNET_STATISTICS.octets_total_udp_out += size+sizeof(MessagePack);
    send(sock, pack, 
	 size+sizeof(MessagePack), 0);
  }
  close(sock);
  xfree(pack,
	"sendToHost: pack (2)");
}

/**
 * Verify that a HELO message is still valid. 
 * @returns SYSERR if the message is invalid, OK if it is ok.
 **/
int verifyHELO(HELO_Body * msg) {
  HashCode160 hashValue;

  /* check that the HELO has not expired */
  if (ntohl(msg->expirationTime) < time(NULL)) {
#if PRINT_WARNINGS
    print("Expired HELO-message received. Dropped!\n");
#endif
    return SYSERR;
  }  
  /* check that the signature is valid */
  if (SYSERR == verifySig(&msg->sender,
			  sizeof(HostAddress)+sizeof(HostIdentity)+sizeof(time_t),
			  &msg->signature,
			  &msg->publicKey)) {
#if PRINT_WARNINGS
    print("Invalid signature for HELO!\n");
#endif
    return SYSERR;
  }  

  /* check that the senderIdentity is really the hash of the public key */
  hash(&msg->publicKey,
       sizeof(PublicKey),
       &hashValue);
  if (memcmp(&hashValue,
	     &msg->senderIdentity,
	     sizeof(HostIdentity)) != 0) {
#if PRINT_WARNINGS
    print("HELO Message sender identity does not match public key supplied.\n");
#endif
    return SYSERR;
  }
  return OK;
}


/* end of identity.c */
