/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * General key management (high level). This wraps
 * around hostkey and configuration and storage providing
 * simple access to the database of keys.
 * @author Christian Grothoff
 * @file server/keyservice.c
 **/

#include "config.h"
#include "server/keyservice.h"

/**
 * The identity of THIS node.
 **/
HostIdentity myIdentity;

/**
 * The SECRET hostkey. Keep local, never export outside of this
 * module and hostkey.c!
 **/
static HOSTKEY hostkey;

/**
 * The public hostkey
 **/
static PublicKey * publicKey;

/**
 * Initialize KeyService. Configuration must be 
 * initialized at this point. You must call this
 * method first!
 **/
void initKeyService() {
  FileName hostkeyfile;
  HostKeyEncoded * encHostkey;
  HELO_Message helo;
  BLOCK_LENGTH len;
  int res;

  initRAND();
  hostkeyfile = getHostKeyFilePriv();
  res = readFile(hostkeyfile, 
		 BLOCK_LENGTH_SIZE, 
		 &len);
  if (res == BLOCK_LENGTH_SIZE) {
    encHostkey = (HostKeyEncoded*) xmalloc(ntohs(len),
					   "initKeyservice: hostkeyencoded");    
    if (ntohs(len) != 
	readFile(hostkeyfile, ntohs(len), encHostkey)) {       
      xfree(encHostkey,"initKeyService: hostkeyencoded");
      encHostkey = NULL;
    }
  } else
    encHostkey = NULL;
  if (encHostkey == NULL) { /* make new hostkey */
    print("Creating new hostkey (this may take a while)... ");
    hostkey = makeHostkey();
    encHostkey = encodeHostkey(hostkey);
    writeFile(hostkeyfile, 
	      encHostkey, 
	      ntohs(encHostkey->len),
	      "600");
    print("Done.\n");
   } else {
    hostkey = decodeHostkey(encHostkey);
  }
  publicKey = xmalloc(sizeof(PublicKey),"initKeyservice: encoded public key");
  getPublicKey(hostkey, publicKey);
  getHostIdentity(publicKey,&myIdentity);  
  createHELO(&helo);
  bindAddress(&helo.body);
  xfree(encHostkey,"initKeyservice: encoded hostkey (2)"); /* ok on encodeHostKey path! */
}

/**
 * Get the public key of the host
 * @return reference to the public key. Do not free it!
 **/
PublicKey * getPublicHostkey() {
  return publicKey;
}

/**
 * Obtain identity from publicHostkey.
 * @param pubKey the public key of the host
 * @param result address where to write the identity of the node
 **/
void getHostIdentity(PublicKey * pubKey,
		     HostIdentity * result) {
  hash(pubKey,
       sizeof(PublicKey),
       &result->hashPubKey);
}

/** 
 * Sign arbitrary data. ALWAYS use only on data we generated
 * entirely! 
 * @return SYSERR on error, OK on success
 **/
int signData(void * data,
	     BLOCK_LENGTH size,
	     Signature * result) {
  return sign(hostkey, 
	      size, 
	      data, 
	      result);
}

/**
 * Decrypt a given block with the hostkey. 
 * @param block the data to decrypt, encoded as returned by encrypt, not consumed
 * @param result pointer to a location where the result can be stored
 * @param max the maximum number of bits to store for the result, if
 *        the decrypted block is bigger, an error is returned
 * @returns the size of the decrypted block, -1 on error
 **/
int decryptData(RSAEncryptedData * block,
		void * result,
		unsigned int max) {
  return decryptHostkey(hostkey, 
			block, 
			result, 
			max);
}



/* end of keyservice */
