/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Pingpong hander. Pings a host and triggers an action if a
 * reply is received.
 * @author Christian Grothoff
 * @file server/pingpong.c
 **/

#include "util/cron.h"
#include "server/identity.h"
#include "statistics.h"

#define MAX_PING_PONG 16
/* if we don't receive a reply within 5 minutes, there is probably nobody
   there... */
#define MAX_DELAY 300


typedef struct {
  HostIdentity receiverIdentity;
  int challenge;
  time_t expires;
  void (*method) (void*);
  void * data;
} PingPongEntry;

static PingPongEntry * pingPongs;
Mutex pingPongLock;

void initPingPong() {
  int i;
  time_t now;
  
  time(&now);
  create_mutex(&pingPongLock);
  pingPongs = (PingPongEntry*) xmalloc(sizeof(PingPongEntry)*MAX_PING_PONG,
				       "ping pong buffer");
  for (i=0;i<MAX_PING_PONG;i++) {
    pingPongs[i].expires = now - 1;
    pingPongs[i].data = NULL;
  }
}

void donePingPong() {
  xfree(pingPongs,
	"ping pong buffer");
}

/**
 * Ping a host an call a method if a reply comes back.
 * @param receiver the IP address to use for the ping
 * @param receiverIdentity the identity to fill into the ping
 * @param method the method to call if a PONG comes back
 * @param an argument to pass to the method.
 **/
void pingAction(HostAddress * receiver,
		HostIdentity * receiverIdentity,
		void (*method)(void *),
		void * data) {
  int i;
  time_t now;
  PingPongEntry * entry;
  PINGPONG_Message msg;

#if PRINT_PING
  print("PING: sending ping\n");
#endif
  MUTEX_LOCK(&pingPongLock);   
  time(&now);
  for (i=0;i<MAX_PING_PONG;i++) 
    if (now > pingPongs[i].expires)
      break; /* found a slot! */
  if (i < MAX_PING_PONG) {
    entry = &pingPongs[i];
    entry->expires = now + MAX_DELAY;
    entry->method = method;
    if (entry->data != NULL)
      xfree(entry->data,
	    "ping pong free");
    entry->data = data;
    memcpy(&entry->receiverIdentity,
	   receiverIdentity,
	   sizeof(HostIdentity));
    msg.header.size = htons(sizeof(PINGPONG_Message));
    msg.header.requestType = htons(GNET_PROTO_PING);
    GNUNET_STATISTICS.udp_out_counts[GNET_PROTO_PING]++;
    memcpy(&msg.receiver,
	   receiverIdentity,
	   sizeof(HostIdentity));
    msg.challenge = entry->challenge = rand();
    sendToHost(receiver,
	       receiverIdentity,
	       sizeof(PINGPONG_Message),
	       &msg,
	       PLAINTEXT_FLAG,
	       crc32N(&msg, sizeof(PINGPONG_Message)));
  }
  MUTEX_UNLOCK(&pingPongLock);     
}
		
void pongReceived(HostIdentity * responderIdentity,
		  int challenge) {
  int i;
  time_t now;
  PingPongEntry * entry;
#if PRINT_PING
  int success = NO;
#endif  
  
  MUTEX_LOCK(&pingPongLock);   
  for (i=0;i<MAX_PING_PONG;i++) {
    entry = &pingPongs[i];
    if ( (challenge == entry->challenge) &&
	 hostIdentityEquals(responderIdentity,
			    &entry->receiverIdentity) ) {
      time(&now);
      if (now < pingPongs[i].expires) {
#if PRINT_PING
	print("PING: received pong, triggering action\n");
	success = YES;
#endif
	entry->method(entry->data);
	entry->expires = now - 1;
	if (pingPongs[i].data != NULL) {
	  xfree(entry->data,
		"ping pong free");
    	  entry->data = NULL;
	}
      }
    }
  }
#if PRINT_PING
  if (NO == success)
    print("PING: no handler found\n");
#endif
  MUTEX_UNLOCK(&pingPongLock);   
}
