/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Policy implementation.
 * @author Christian Grothoff
 * @file server/policy.c
 **/

#include "config.h"
#include "server/policy.h"
#include "server/routing.h"
#include "util/contentdatabase.h"
#include "server/trustservice.h"

/**
 * A new packet is supposed to be send out. Should it be
 * dropped because the load is too high?
 * <p>
 * @param priority the highest priority of contents in the packet
 * @return OK if the packet should be handled, SYSERR if the packet should be dropped.
 **/
int outgoingCheck(int priority) {
  unsigned int maxNetLoad=getMaxNetLoad();
  unsigned int curNetLoad = networkUsage();

  priority++;
  if (curNetLoad>maxNetLoad*(priority+1)) {
#if PRINT_POLICY
    print("POLICY: network load to high, dropping outgoing.\n");
#endif
    return SYSERR;
  }
  return OK; /* no policy */
}

/**
 * A new packet is coming in. The question is, if we should
 * have a look at it at all (incl. decryption). The sender
 * is known. 
 * <p>
 * The policy must decide if we should drop the packet. The
 * reason may be that we don't trust the host or that the load
 * is to high (or both or a combination). 
 * <p>
 * @param senderIdentity the identity of the sender
 * @return OK if the packet should be handled, SYSERR if the packet should be dropped.
 **/
int incomingCheck(HostIdentity * senderIdentity) {
  unsigned int maxLoad=getMaxLoad();
  unsigned int curLoad = cpuUsage();

  if (curLoad>maxLoad*4) {
#if PRINT_POLICY
    print("POLICY: CPU load to high (%d > %d), dropping incoming packet.\n",
	  curLoad, maxLoad);
#endif
    return SYSERR;
  }
  return OK; /* no policy */
}

/**
 * A query has been received. The question is, if it should be forwarded
 * and if with which priority. Routing decisions(to whom) are to be taken elsewhere.
 * <p>
 * @param senderIdentity the host *forwarding* (or originating, we don't know) us the query
 * @param priority the priority the query had when it came in, may be an arbitrary number if the 
 *        sender is malicious! Cap by trustlevel first!
 * @return binary encoding: QUERY_XXXX constants
 **/
QUERY_POLICY evaluateQuery(HostIdentity * senderIdentity, 
			   unsigned int priority) {
  unsigned int maxNetLoad=getMaxNetLoad();
  unsigned int maxLoad=getMaxLoad();
  unsigned int curNetLoad = networkUsage();
  unsigned int curLoad = cpuUsage();
  if ( (maxNetLoad > curNetLoad*2) &&
       (maxLoad > curLoad*2) )
    return 1 /* minimum priority */ |
      QUERY_ANSWER | QUERY_FORWARD | QUERY_INDIRECT;
  priority = 1-changeHostCredit(senderIdentity, -priority);
  if (maxNetLoad*2*(priority+1)>curNetLoad*3)
    return ((priority) & QUERY_DROPMASK) |
      QUERY_ANSWER | QUERY_FORWARD | QUERY_INDIRECT;
  else if (maxLoad*2*(priority+1)>(curLoad*3))
    return ((priority) & QUERY_DROPMASK) |
      QUERY_ANSWER | QUERY_FORWARD;
  else if (maxNetLoad*(priority+1)>(curLoad))
    return ((priority) & QUERY_DROPMASK) |
      QUERY_ANSWER;
  else
    return 0; /* drop entirely */
}

/**
 * Some content dropped by. We may want to store it locally, or not.
 * The policy can do with it whatever it sees fit.
 *
 * The current primitive implementation stores content that flys
 * by until the drive is full. If the drive is full, it looks at
 * the priority of the query and discards less important content
 * to make room if there is content that is less important. Otherwise
 * the content is just dropped.
 * 
 * If we are busy, we also drop the content - no questions asked.
 * The processing may be too expensive == there'll be other content :)
 *
 * @param senderIdentity the sender of the content
 * @param content the data
 * @param hc the hashcode of the data
 * @param preference how good was the data?
 **/
void evaluateContent(HostIdentity * senderIdentity,
		     CONTENT_Block * content,
		     HashCode160 * hc,
		     int preference) {
  ContentIndex newEntry;
  HashCode160 tripleHash;
  int distance;
  int j;
  
  if (NO == activeMigration()) 
    return; /* we're not supposed to migrate, drop! */
  if (getMaxLoad() < cpuUsage()) 
    return; /* busy: drop */
  

  hash(hc, sizeof(HashCode160), &tripleHash);
  distance = distanceHashCode160(&tripleHash,
				 &ROUTING_localhost_.hashPubKey);
  /* compute 'log' of distance */
  j=16;
  while (distance > 0) {
    distance = distance>>1;
    j--;
  }
  preference*=j; /* multiply preference with proximity */
  if (!(spaceAvailable(preference)))
    return;
  writeContent(hc, content);
  memcpy(&newEntry.doubleHash,
	 hc,
	 sizeof(HashCode160));
  newEntry.importance = htonl(preference);
  newEntry.fileNameIndex = htonl(-1);
  newEntry.fileOffset = htonl(-1);
  addEntry(&newEntry);
}


/* end of policy.c */

