/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Routing implementation.
 *
 * @author Christian Grothoff
 * @file server/routing.c
 **/

#include "config.h"
#include "statistics.h"
#include "server/routing.h"
#include "server/trustservice.h"

static IndirectionTableEntry * ROUTING_indTable_;

HostIdentity ROUTING_localhost_;

/* ************* internal Methods **************** */

/**
 * Compute the hashtable index of a host id.
 **/
static unsigned int computeRoutingIndex(HashCode160 * query) {
  unsigned int res = (((unsigned int)query->a) & 
		      ((unsigned int)(INDIRECTION_TABLE_SIZE - 1)));
#if EXTRA_CHECKS
  if (res >= INDIRECTION_TABLE_SIZE)
    errexit("FATAL: INDIRECTION_TABLE_SIZE not power of 2? (%d)\n",
	    INDIRECTION_TABLE_SIZE);
#endif
  return res;
}

/**
 * Find out, if this query is already pending.
 *
 * @param msg the Query to look for
 * @param sock the TCP socket that is asking (NULL for remote 
 *        queries)
 * @return YES if the query was already send out (for this socket),
 *         NO if not
 **/
static int isPending(QUERY_Message * msg,
		     GNUNET_TCP_SOCKET * sock) {
  IndirectionTableEntry * ite;
  time_t now;
#if PRINT_ROUTING > 1
  HexName hex;
#endif

  time(&now);
  ite = &ROUTING_indTable_[computeRoutingIndex(&msg->query)];
  MUTEX_LOCK(&ite->lock); 

  /* check if the entry is already timed out */
  if (ite->ttl < now) {
    MUTEX_UNLOCK(&ite->lock); 
#if PRINT_ROUTING > 1
    print("QUERY: ttl of new query %d is higher than old ttl %d.\n",
	  ntohl(msg->ttl), ite->ttl - now);
#endif
    return NO; /* if there, the ttl is smaller and we should re-send */
  }
  /* check if it is actually about the same question */
  if (equalsHashCode160(&msg->query,
			&ite->hash)) {
#if PRINT_ROUTING > 1 
    hash2hex(&ite->hash, &hex);
    print("QUERY: Query %s is pending (with ttl old %d and new %d).\n",
	  &hex,
	  ite->ttl-now, ntohl(msg->ttl));
#endif
    MUTEX_UNLOCK(&ite->lock); 
    return YES; /* already there, ttl larger */    
  } 
#if PRINT_ROUTING 
  else
    print("ROUTING: collision in routing table\n");
#endif

  MUTEX_UNLOCK(&ite->lock); 
#if PRINT_ROUTING > 1
  print("QUERY: Query is brand new with ttl %d.\n",
	ntohl(msg->ttl));
#endif
  return NO;
}

/**
 * We're going to forward an external request with indirection. Thus
 * this node must keep track of the request for a while to route
 * results. This method should not call forwardQuery, the caller will.
 *
 * @param msg the query we're going to forward
 * @param sock the tcp socket that a reply will eventually go to, 
 *        NULL if only non-local delivery.
 * @return OK if that can be done, SYSERR if the routing table is full,
 *         the query has timed out or the query was already forwarded
 **/
static int indirectingQuery(QUERY_Message * msg,
			    GNUNET_TCP_SOCKET  * sock) {
  int j;
  int present;
  IndirectionTableEntry * ite;
  time_t now;
  int result;
#if PRINT_ROUTING
  HexName hex;
  HexName hexHost;
#endif
 
  time(&now);
  if (ntohl(msg->ttl) <= 0) /* expired? */
    return SYSERR;
#if PRINT_ROUTING
  hash2hex(&msg->query, &hex);
  hash2hex(&msg->returnTo.hashPubKey, &hexHost);
  print("ROUTING: Indirecting query %s from node %s.\n",
	(char*)&hex,
	(char*)&hexHost);
#endif

  ite = &ROUTING_indTable_[computeRoutingIndex(&msg->query)];
  MUTEX_LOCK(&ite->lock);
  result = SYSERR;
  /* case 1: the same query is already live */
  if ( equalsHashCode160(&msg->query,
			 &ite->hash) &&
       (ite->ttl > now) ) {
    /* we may have this host already in the waiting list,
       we definitely have the query pending. Decide between
       doing nothing and appending host to queue */      
    present = NO;
    if (hostIdentityEquals(&msg->returnTo,
			   &ROUTING_localhost_))
      present = YES;
    else
      for (j=0;j<ite->hostsWaiting;j++) 
	if (hostIdentityEquals(&ite->destination[j],
			       &msg->returnTo)) 
	  present = YES;
    if ( (present == NO) &&
	 (ite->hostsWaiting < MAX_HOSTS_PER_QUERY) ) {
      memcpy(&ite->destination[ite->hostsWaiting++],
	     &msg->returnTo,
	     sizeof(HostIdentity));
      result = OK;
    }
    /* same pattern, just now for tcp socket */
    if (sock != NULL) {
      present = NO;
      for (j=0;j<MAX_TCP_PER_QUERY;j++)
	if (ite->tcpsocks[j] == sock)
	  present = YES;
      if (present == NO)
	for (j=0;j<MAX_TCP_PER_QUERY;j++)
	  if (ite->tcpsocks[j] == NULL) {
	    ite->tcpsocks[j] = sock;
	    result = OK;
	    break;
	  }
#if PRINT_WARNINGS
      if (present == YES) {
	print("ROUTING: You may want to increase MAX_TCP_PER_QUERY.\nThe current limit %d was hit.\n",
	      MAX_TCP_PER_QUERY);
	print("Sockets waiting: ");
	for (j=0;j<MAX_TCP_PER_QUERY;j++)
	  if (ite->tcpsocks[j] != NULL) 
	    print("%d ", ite->tcpsocks[j]->socket);
	  else
	    print("?? ");
	print("\nUnserved socket: %d\n",
	      sock->socket);
      }
#endif    
    } /* sock != NULL */
    MUTEX_UNLOCK(&ite->lock); 
    return result;
  }
  /* we had no or a different query before. If the old
     query has a smaller tll we keep it, otherwise we
     take the new one. */
  if ( (ite->ttl < now + ntohl(msg->ttl)) &&
       (ite->ttl > now) ) {
    /* life query with smaller tll has this slot,
       we can't keep track of this indirection. */
    MUTEX_UNLOCK(&ite->lock); 
#if PRINT_WARNINGS > 2
    print("ROUTING: You may want to increase INDIRECTION_TABLE_SIZE.\nThe current limit %d was hit.\n",
	  INDIRECTION_TABLE_SIZE);
#endif    
    return SYSERR;
  }
  /* Ok, this entry is empty or requires us to keep track
     longer; copy the new query over */
  ite->ttl = ntohl(msg->ttl)+now;
  ite->priority = ntohl(msg->priority);  
  if (sock == NULL) {
    ite->hostsWaiting = 1; 
    memcpy(&ite->destination[0],
	   &msg->returnTo,
	   sizeof(HostIdentity));
  } else {
    ite->hostsWaiting = 0;
    assert(hostIdentityEquals(&msg->returnTo,
			      &ROUTING_localhost_),
	   "Query that is to be send back to TCP should have the local node as the originator!");
  }
  memcpy(&ite->hash,
	 &msg->query,
	 sizeof(HashCode160));
  ite->seenIndex = 0;
  ite->tcpsocks[0] = sock; /* NULL or not, either way, this is it */
  for (j=1;j<MAX_TCP_PER_QUERY;j++)
    ite->tcpsocks[j] = NULL;
  MUTEX_UNLOCK(&ite->lock); 
  return OK;
}

/**
 * Send a reply to a host. Distinguishes between
 * local and remote delivery, converts the reply
 * into the appropriate format and sends it out.
 *
 * @param ite the matching slot in the indirection table
 * @param content the data to send (1k)
 * @param hash the hashcode to send with the reply
 **/
static void sendReply(IndirectionTableEntry * ite,
		      CONTENT_Block * content,
		      HashCode160 * hash) {
  TCP_Content_Reply reply;
  CONTENT_Message message;
  int j;
  int doTcp;

#if PRINT_ROUTING
  HexName query;

  hash2hex(hash, &query);  
  print("ROUTING: sending reply for query %s\n",
	&query);
  
#endif
  
  doTcp = NO;
  for (j=0;j<MAX_TCP_PER_QUERY;j++) 
    if (ite->tcpsocks[j] != NULL) 
      doTcp = YES;		
  if (doTcp == YES) {
    reply.header.size = htons(sizeof(TCP_Content_Reply));
    reply.header.tcpType = htons(TCP_REPLY_CONTENT);
    memcpy(&reply.content,
	   content,
	   CONTENT_SIZE);
    memcpy(&reply.hash,
	   hash,
	   sizeof(HashCode160));    
    for (j=0;j<MAX_TCP_PER_QUERY;j++) 
      if (ite->tcpsocks[j] != NULL) {
	GNUNET_STATISTICS.tcp_out_counts[TCP_REPLY_CONTENT]++;
	writeToSocket(ite->tcpsocks[j],
		      (TCP_SOCKET_BUFFER*)&reply.header);
      }
  }
  if (ite->hostsWaiting == 0)
    return; /* that was it */

  /* else: prepare sending via UDP/connection module */  
  memcpy(&message.content,
	 content,
	 CONTENT_SIZE);
  memcpy(&message.hash,
	 hash,
	 sizeof(HashCode160));    
  message.header.size 
    = htons(sizeof(CONTENT_Message));
  message.header.requestType 
    = htons(GNET_PROTO_CONTENT);
  for (j=0;j<ite->hostsWaiting;j++)
    unicast(&message.header, 
	    &ite->destination[j], 
	    ite->priority);
}
 

/* ***************** public methods ****************** */

/**
 * Initialize routing module (initializes indirection table)
 **/
void initRouting() {
  unsigned int i;
  time_t now;

  time(&now);
  ROUTING_indTable_ 
    = xmalloc(sizeof(IndirectionTableEntry)*
	      INDIRECTION_TABLE_SIZE,
	      "initRouting: indirectionTable");
  for (i=0;i<INDIRECTION_TABLE_SIZE;i++) {
    LOOPPRINT("initRouting"); 
    ROUTING_indTable_[i].ttl = now-1; /* expired / free */  
    ROUTING_indTable_[i].hostsWaiting = 0; /* expired / free */  
    create_mutex(&ROUTING_indTable_[i].lock);    
  }
  getHostIdentity(getPublicHostkey(),
		  &ROUTING_localhost_);
}

/**
 * Print the routing table.
 **/
void printRoutingTable() {
  int i;
  IndirectionTableEntry * ite;
  HexName h1;
  HexName h2;
  time_t now;

  time(&now);
  print("Routing TABLE:\n");
  for (i=0;i<INDIRECTION_TABLE_SIZE;i++) {
    LOOPPRINT("printRoutingTable"); 
    ite = &ROUTING_indTable_[i];  
    MUTEX_LOCK(&ite->lock); 
    hash2hex(&ite->hash, &h1);
    hash2hex(&ite->destination[0].hashPubKey, 
	     &h2);
    if (ite->ttl >= now)
      print("%d: hash %s, destination[0] %s, ttl %d, hostsWaiting %d prio %d seenIndex: %d\n ", 
	    i,
	    &h1, &h2,
	    ite->ttl-now,
	    ite->hostsWaiting,
	    ite->priority,
	    ite->seenIndex);
    MUTEX_UNLOCK(&ite->lock);  
  }
}

/**
 * Execute the triple-hash query. This method does NOT check the QUERY
 * Policy, it is used either after the policy was checked or if the
 * request is trusted (especially needed if comes from our client).
 *
 * @param qp the polciy (priorty, etc) for the query
 * @param msg the query message (with host identity for the reply)
 * @param sock the TCP socket to send the answer to if it is
 *        a query from the local host, otherwise NULL.
 **/
void exec3Query(QUERY_POLICY qp, 
		QUERY_Message * msg,
		GNUNET_TCP_SOCKET * sock) {
  if ((qp & QUERY_FORWARD) == 0) 
    return; /* drop on the floor: policy */
  msg->ttl = htonl(ntohl(msg->ttl)-TTL_DECREMENT);
  if (((signed int)ntohl(msg->ttl)) <= 0) 
    return; /* drop on the floor: timed out! */
  if (YES == isPending(msg, sock)) {
    indirectingQuery(msg, sock); 
#if ROUTING
    print("ROUTING: Query is pending, not forwarded, no local lookup\n");
#endif
    return;  /* we've already forwarded this query */
  }

  /* forward query */
  if ((qp & QUERY_INDIRECT) > 0) { 
    /* forward query & replace returnTo with our identity, 
       also keep track of the original returnTo for the case
       we get a reply! */
    if (OK == indirectingQuery(msg, sock)) {
      /* set reply-to to myself */
      getHostIdentity(getPublicHostkey(),
		      &msg->returnTo);
      tripleHashLookup(&msg->query,
		       qp,
		       &useContent);
      if ( (qp & QUERY_PRIORITY_BITMASK) < ntohl(msg->priority) )
	msg->priority = htonl(qp & QUERY_PRIORITY_BITMASK);
   } else
     msg->priority = 0; /* indirection table full, no indirection, no credit, so we don't care */
  } else 
    msg->priority = 0; /* we're not even indirecting, so we can't earn credit, so we don't care */
  forwardQuery(msg); 
}

/**
 * Content has arrived. We must decide if we want to a) forward it to
 * our clients b) indirect it to other nodes. The routing module
 * should know what to do.
 *
 * @return how good this content was (priority of the request)
 **/
int useContent(HostIdentity * hostId,
	       CONTENT_Block * content,
	       HashCode160 * hc) {
  unsigned int i;
  HashCode160 tripleHash;
  HashCode160 collisionHC;
  IndirectionTableEntry * ite;
  int prio = -1;
  time_t now;
  int isNew;

#if PRINT_CONTENT
  HexName hex;
  hash2hex(hc, &hex);
  print("CONTENT: Using Content %s.\n",
	&hex);
#endif
#if PRINT_ROUTING > 2
  print("Received content. Routing table: \n");
  printRoutingTable();
#endif

  hash(hc,
       sizeof(HashCode160),
       &tripleHash);
  hash(content,
       sizeof(CONTENT_Block),
       &collisionHC);
  time(&now);
  ite = &ROUTING_indTable_[computeRoutingIndex(&tripleHash)];
  MUTEX_LOCK(&ite->lock);
 
  if ( (ite->ttl < now) ||
       (!equalsHashCode160(&ite->hash,
			   &tripleHash)) ) {	
    MUTEX_UNLOCK(&ite->lock);
#if PRINT_CONTENT
    print("CONTENT: no matching query pending\n");
#endif    
    return 0; /* no indirection pending: was useless */
  }
  isNew = YES;
  for (i=0;i<ite->seenIndex;i++) {
    LOOPPRINT("useContent"); 
    if (equalsHashCode160(&collisionHC,
			  &ite->seen[i])) 
      isNew = NO;
  }
 /* new reply, adjust credits! */
  if (hostId != NULL) /* if we are the sender, hostId will be NULL */
    changeHostCredit(hostId, ite->priority);
  prio = ite->priority;
  ite->priority = 1; /* reduce priority for further replies,
			because we didn't get paid for those... */
  if (isNew == YES) {
#if PRINT_CONTENT
  hash2hex(&ite->hash,
	   &hex);
  print("CONTENT: Indirecting new content matching triple query %s.\n",
	&hex);
#endif	
    sendReply(ite,
	      content,
	      hc); 
    if (ite->seenIndex == MAX_REPLIES)
      ite->ttl = now-1; /* mark as 'timed out' */
    else /* mark as already forwarded */
      memcpy(&ite->seen[ite->seenIndex++],
	     &collisionHC,
	     sizeof(HashCode160));
  }
  MUTEX_UNLOCK(&ite->lock);
  return prio;
}

/**
 * TCP connection is shut down, cancel all
 * replies to that client.
 **/
void cancelTCP(GNUNET_TCP_SOCKET * sock) {
  int i;
  int j;
  IndirectionTableEntry * ite;

  for (i=0;i<INDIRECTION_TABLE_SIZE;i++) {
    LOOPPRINT("printRoutingTable"); 
    ite = &ROUTING_indTable_[i];  
    MUTEX_LOCK(&ite->lock); 
    for (j=0;j<MAX_TCP_PER_QUERY;j++)
      if (ite->tcpsocks[j] == sock)
	ite->tcpsocks[j] = NULL;
    MUTEX_UNLOCK(&ite->lock);  
  }
}
 


/* end of routing.c */
