/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Sessionkey management
 * @author Christian Grothoff
 * @file server/sessionkey.c
 **/

#include "config.h"
#include "gnettypes.h"
#include "statistics.h"
#include "server/sessionkey.h"
#include "server/connection.h"


typedef struct {
  /* length of the block (SESSIONKEY_LEN), in network byte order */
  BLOCK_LENGTH len __attribute__ ((packed));
  /* reserved, must be 0 (for now) */
  unsigned short reserved __attribute__ ((packed));
  SESSIONKEY skey __attribute__ ((packed));
} SessionKeyEncoded;


/**
 * @return SYSERR if invalid, OK if valid
 **/
static int verifySKS(HostIdentity * hostId,
		     SKEY_Message * sks) {
  PublicKey pubKey;
  HashCode160 keyHash;
#if PRINT_WARNINGS
  HexName hostName;
#endif
  
  if ( (sks == NULL) ||
       (hostId == NULL)) {
#if PRINT_WARNINGS
    hash2hex(&hostId->hashPubKey, &hostName);
    print("VerifySKS: invalid sessionkey suggestion for host %s\n",
	  &hostName);
#endif
    return SYSERR;
  }
  
  if (SYSERR == identity2Key(hostId, &pubKey)) {
#if PRINT_WARNINGS
    hash2hex(&hostId->hashPubKey, &hostName);
    print("verifySKS: host %s for sessionkey exchange not known\n",
	  &hostName);
#endif
    return SYSERR;
  }
  /* verify signature */
  hash(&sks->body, 
       sizeof(RSAEncryptedData) + sizeof(time_t),
       &keyHash);
  if (!verifySig(&keyHash,
		 sizeof(HashCode160),
		 &sks->body.signature, 
		 &pubKey)) {
#if PRINT_WARNINGS
    print("verifySKS: received SessionKey with bad signature!\n");
#endif
    return SYSERR; /*reject!*/
  }
  return OK; /* ok */
}

static void checkAndPing(BufferEntry * be) {
  PINGPONG_Message msg;

#if PRINT_CRON
  print("CRON: enter checkAndPing\n");
#endif
   if (be->status != STAT_UP) {
    msg.header.size = htons(sizeof(PINGPONG_Message));
    msg.header.requestType = htons(GNET_PROTO_PING);
    memcpy(&msg.receiver,
	   &be->hostId,
	   sizeof(HostIdentity));
    msg.challenge = be->challenge;
    unicast(&msg.header,
	    &be->hostId,
	    5);    
    MUTEX_LOCK(&be->sem);
    sendBuffer(be);
    MUTEX_UNLOCK(&be->sem);
  }
#if PRINT_CRON
  print("CRON: exit checkAndPing\n");
#endif
}

static void checkAndGiveup(BufferEntry * be) {
#if PRINT_CRON
  print("CRON: enter checkAndGiveup\n");
#endif
  MUTEX_LOCK(&be->sem);
  if (be->status != STAT_UP)
    be->status = STAT_DOWN;
  MUTEX_UNLOCK(&be->sem);
#if PRINT_CRON
  print("CRON: exit checkAndGiveup\n");
#endif
}

/**
 * Force creation of a new Session key for the given host. 
 * @param hostId the identity of the other host 
 * @param sk the SESSIONKEY to use
 * @param created the timestamp to use
 * @param ret the address where to write the signed session key, first BLOCK_LENGTH_SIZE byte give length
 * @return OK on success, SYSERR on failure
 **/
static int makeSessionKeySigned(HostIdentity * hostId,
				SESSIONKEY * sk,
				time_t created,
				SKEY_Message * ret) {
  SessionKeyEncoded enc;
  HashCode160 keyHash;
  HostIdentity myId;
  PublicKey otherPubKey;
#if EXTRA_CHECKS
  HexName hostName;
#endif

  /* create and encrypt sessionkey */
  if (SYSERR == identity2Key(hostId, &otherPubKey)) {
#if PRINT_WARNINGS
    print("makeSesionKeySigned: cannot encrypt sessionkey, other host not known!\n");  
#endif 
    return SYSERR; /* other host not known */  
  }
  enc.len = htons(SESSIONKEY_LEN);
  enc.reserved = 0;
  memcpy(&enc.skey,
	 sk,
	 sizeof(SESSIONKEY));

  if (SYSERR == encryptHostkey(&enc,
			       sizeof(SessionKeyEncoded),
			       &otherPubKey,
			       &ret->body.key)) {
#if PRINT_WARNINGS
    print("makeSesionKeySigned: encrypt failed!");
#endif
    return SYSERR; /* encrypt failed */
  }
  /* compute hash and sign hash */
  ret->body.creationTime = htonl(created);
  hash(&ret->body, 
       sizeof(RSAEncryptedData) + sizeof(time_t),
       &keyHash);
  if (SYSERR == signData(&keyHash, 
			 (BLOCK_LENGTH)sizeof(HashCode160),
			 &ret->body.signature)) {
    print("ERROR: signData failed\nTrying to continue.\n");
  }

  /* complete header */
  ret->header.size = htons(sizeof(SKEY_Message));
  ret->header.requestType = htons(GNET_PROTO_SKEY);
  /* verify signature/SKS */
  getHostIdentity(getPublicHostkey(),
		  &myId); 
#if EXTRA_CHECKS
  hash2hex(&hostId->hashPubKey, &hostName);
  if (SYSERR == verifySKS(&myId, ret)) {
    errexit("MakeSessionKey: generated sessionkey for host %s failed verification! exiting!\n",
	    &hostName);
  }
#endif
  return OK;
} 

/**
 * Shutdown the connection.
 * Send a HANGUP message to the other side and
 * mark the sessionkey as dead.
 **/
void shutdownConnection(BufferEntry * be) {
  if (be->status != STAT_UP)
    return; /* nothing to do */
  /* FIXME: send HANGUP */
  be->status = STAT_DOWN;
}

/**
 * Perform a session key exchange for entry be.
 * First sends a HELO and then the new SKEY
 * (in one unencrypted packet). When called, the 
 * semaphore of at the given index must already be down
 **/
void exchangeKey(BufferEntry * be) {
  HELOSKEY m;
  int crc;

#if PRINT_SKEY
  HexName hex;
  hash2hex(&be->hostId.hashPubKey, &hex);
  print("SKEY: Beginning of exchange key with %s\n",
	&hex);
#endif
  assert(be->status == STAT_DOWN,
	 "status must be down for session key exchange!");
  makeSessionkey(&be->skey);
  time(&be->created); /* set creation time for session key! */
  if (SYSERR == makeSessionKeySigned(&be->hostId,
				     &be->skey,
				     be->created,
				     &m.sks))
    return;
  be->isAlive = 0;
  be->status = STAT_WAITING_FOR_PING;
  be->challenge = 0;
  be->lastSequenceNumberReceived = 0;
  be->lastPacketsBitmap = (unsigned int) -1;
  /* send tmp to target */
  createHELO(&m.helo);
  /* get crc of data */
  crc = crc32N(&m, sizeof(HELOSKEY));
  GNUNET_STATISTICS.udp_out_counts[GNET_PROTO_HELO]++;
  GNUNET_STATISTICS.udp_out_counts[GNET_PROTO_SKEY]++;
  
  sendToHost(NULL,
	     &be->hostId,
	     sizeof(HELOSKEY),
	     &m, 
	     PLAINTEXT_FLAG, 
	     crc);
}

/**
 * Force adding of a host to the buffer.
 * @param created the time at which this key was created
 * @return OK if it went well, SYSERR if not.
 **/
static int addHostWithSKEY(HostIdentity * hostId,
			   SESSIONKEY * skey,
			   time_t created) {
  BufferEntry * be;
#if PRINT_SKEY
  HexName hex;
#endif
#if PRINT_CONNECTION > 1
#if 0 ==  PRINT_SKEY
  HexName hex;
#endif
#endif
    
  be = lookForHost(hostId);
  if (be == NULL) {
    cronScanDirectoryDataHosts(NULL); /* maybe the host is very new */
    be = addHost(hostId);
  }
  if (be == NULL) {
#if PRINT_SKEY
    hash2hex(&hostId->hashPubKey, &hex);
    print("SKEY: Session key exchange denied, host %s unknown!\n",
	  &hex);
#endif
    return SYSERR;
  }
  /* here we should *also* consider the trust-levels in the future! */
  if ( ( (!hostIdentityEquals(&be->hostId,
			      hostId)) &&
	 (be->status != STAT_DOWN)) ||
       (be->created > created) ) {
#if PRINT_SKEY
    print("Session key exchange denied, another host is in the table and active!\n");
#endif
    MUTEX_UNLOCK(&be->sem);
    return SYSERR;
  }
#if PRINT_CONNECTION > 1
  hash2hex(&hostId->hashPubKey, &hex);
  print("CONNECTION: adding host %s with sessionkey to table.\n",
	&hex);
#endif 
  memcpy(&be->hostId,
	 hostId,
	 sizeof(HostIdentity));	 
  memcpy(&be->skey,
	 skey,
	 SESSIONKEY_LEN);
  be->created = created;
  be->status = STAT_WAITING_FOR_PONG;
  be->lastSequenceNumberReceived = 0;
  be->lastPacketsBitmap = (unsigned int) -1;
  be->buffer.sequence.header.size
    = htons(sizeof(SEQUENCE_Message));
  be->buffer.sequence.header.requestType
    = htons(GNET_PROTO_SEQUENCE);
  be->buffer.sequence.sequenceNumber
    = htonl(1);
  be->len = sizeof(SEQUENCE_Message);
  MUTEX_UNLOCK(&be->sem);
  delCronJob((void (*)(void*))&checkAndPing,
	     0, be);
  delCronJob((void (*)(void*))&checkAndGiveup,
	     0, be);
  addCronJob((void (*)(void*))&checkAndGiveup,
	     120, 0, be);
  return OK;
}

/**
 * Accept a session-key that has been sent by another host.
 * The other host must be known (public key)
 * @param hostId the identity of the sender host
 * @param sessionkeySigned the session key that was "negotiated"
 **/
void acceptSessionKey(HostIdentity * hostId,
		      SKEY_Message * sessionkeySigned) {
  SessionKeyEncoded key;
  BufferEntry * be;
#if PRINT_SKEY
  HexName hostName;
#endif
#if PRINT_WARNINGS
  HostAddress conInfo;
  IPPortString ipPort;
#if 0 == PRINT_SKEY
  HexName hostName;
#endif
#endif

#if PRINT_SKEY
  hash2hex(&hostId->hashPubKey,
	    &hostName);
  print("SKEY: Accepting sessionkey from host %s.\n",
	&hostName);
#endif
  if (SYSERR == verifySKS(hostId, sessionkeySigned)) {
#if PRINT_WARNINGS > 1
    print("WARNING: session key failed verification, ignored!\n");
#endif
    return;  /* rejected */
  }

  /* prepare file */
  if ((sizeof(SessionKeyEncoded) != 
       decryptData(&sessionkeySigned->body.key,
		   &key,
		   sizeof(SessionKeyEncoded))) ||
      (ntohs(key.len) != SESSIONKEY_LEN) ||
      (ntohs(key.reserved != 0)) ) {
#if PRINT_WARNINGS 
#if 0 == PRINT_SKEY
    hash2hex(&hostId->hashPubKey,
	     &hostName);
#endif
    if (OK == getConnectionInformation(hostId,
				       &conInfo)) {
      printAddress(&conInfo,
		   &ipPort);
      print("SKEY rejected from host %s (address %s): key.len = %d\n",
	    &hostName,
	    &ipPort,
	    key.len);
    } else
      print("SKEY rejected from host %s (address unknown): key.len = %d\n",
	    &hostName,
	    key.len);
#endif    
    return;
  }
  if (SYSERR == addHostWithSKEY(hostId, &key.skey, 
				ntohl(sessionkeySigned->body.creationTime))) {
#if PRINT_WARNINGS > 1
    print("WARNING: key exchange failed: addHostWithSKEY returned SYSERR\n");
#endif
    return;
  }
  be = lookForHost(hostId);
  be->challenge = rand();
#if PRINT_SKEY
  print("SKEY: sending ping with challenge %d\n",
	be->challenge);
#endif
  MUTEX_UNLOCK(&be->sem);
  checkAndPing(be);
  delCronJob((void (*)(void*))&checkAndPing,
	     0, be);
  delCronJob((void (*)(void*))&checkAndGiveup,
	     0, be);
  addCronJob((void (*)(void*))&checkAndPing,
	     90, 0, be);
  addCronJob((void (*)(void*))&checkAndPing,
	     180, 0, be);
  addCronJob((void (*)(void*))&checkAndGiveup,
	     240, 0, be);
}

/* end of sessionkey.c */
