/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Helper methods for the startup of gnet. Do 
 * some checks and commandline parsing.
 *
 * @file server/startup.c
 * @author Christian Grothoff
 **/

#include "config.h"
#include "server/startup.h"

#include <sys/types.h>
#include <unistd.h>
#include <pwd.h>
#include "getopt.h"

/**
 * Check if the compiler did a decent job aligning the structs...
 **/
void checkCompiler() {
  assert(sizeof(HELO_Body) == 552,
	 "sizeof HELO BODY wrong!");
  assert(sizeof(MessagePack) == 36,
	 "sizeof MessagePack wrong!");
  assert(sizeof(MessagePart) == 4,
	 "sizeof MessagePart wrong!");
  assert(sizeof(RootNode) == CONTENT_SIZE,
	 "sizeof RootNode wrong!");
}


/**
 * Print a list of the options we offer.
 **/
void printhelp() {
  printf("USAGE: gnunetd [OPTIONS]\n");
  printf("-h, --help                    : print this page\n");
  printf("-c FILENAME, --config=FILENAME: load config file (defaults: %s)\n",
	 DEFAULT_CONFIG_FILE);
  printf("-v, --version                 : print the version number\n");
  printf("-u USER, --user=USER          : run as user USER\n");
}


/**
 * Perform option parsing from the command line. 
 **/
int parseCommandLine(int argc, 
		     char * argv[],
		     char ** configFile) {
  int shouldexit=0;
  int c;
  struct passwd * pws;
  while (1) {
    int option_index=0;
    static struct option long_options[] = {
      { "config",  1, 0, 'c' },
      { "version", 0, 0, 'v' },
      { "help",    0, 0, 'h' },
      { "user",    1, 0, 'u' },
      { 0,0,0,0 }
    };
    
    LOOPPRINT("parseCommandLine"); 
    c = getopt_long(argc,
		    argv, 
		    "vhc:u:", 
		    long_options, 
		    &option_index);
    
    if (c == -1) 
      break;  // No more flags to process
    
    switch(c) {
    case 'c': 
      *configFile = optarg; 
      print("Configfile specified: %s.\n",*configFile);
      break;
    case 'v': 
      print("GNUnet %s (%s)\n",
	    GNUNET_VERSION, GNUNET_BRANCH);
      shouldexit=1;
      break;
    case 'h': 
      printhelp(); 
      shouldexit=1;
      break;
    case 'u':
      pws = getpwnam(optarg);
      if (pws == NULL) {
        print("User %s not known, can not change UID to it. Will continue...",
              optarg);
        break;
      }
      if ( (0 != setregid(pws->pw_gid, pws->pw_gid)) || 
	   (0 != setreuid(pws->pw_uid, pws->pw_uid)) )
	print("Can not change user/group to %s. Will continue...",
	      optarg);
      break;
    default:
      print("Unknown option %c. Aborting.\nUse --help to get a list of options.\n",
	    c);
      shouldexit=1;    
    } /* end of parsing commandline */
  }
  if (optind < argc) {
    print("Invalid arguments: ");
    while (optind < argc)
      print("%s ", argv[optind++]);
    print("\nExiting.\n");
    return 1;
  }
  return shouldexit;
}

/**
 * Allocate & bind a server socket using UDP
 * @param port the port in host byte order
 **/
int passivesock(unsigned short port) {
  struct sockaddr_in sin;	/* an Internet endpoint address		*/
  int	s, type;	/* socket descriptor and socket type	*/
  const int on = 1;

  memset(&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = INADDR_ANY;
  sin.sin_port = port;

  /* UDP */
  type = SOCK_DGRAM;
  /* Allocate a socket */
  s = socket(PF_INET, type, UDP_PROTOCOL_NUMBER);
  if (s < 0) 
    errexit("Can not create socket: %s\n", 
	    strerror(errno));  

  if ( setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) < 0 )
      perror("setsockopt");

  /* Bind the socket */
  if (bind(s, (struct sockaddr *)&sin, sizeof(sin)) < 0)
    errexit("Can not bind to UDP port: %s\n",
	    strerror(errno));

  return s;
}



/* end of startup.c */
