/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * The server side of the TCP connection between a GUI and GNUnet node
 * @file server/tcpserver.c
 * @author Tiberiu Stef
 * @author Christian Grothoff
 **/

#include "config.h"
#include <stdio.h>
#include <stdlib.h> 	 
#include <unistd.h>		
#include <pthread.h>		
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h> 

#include "server/tcpserver.h"
#include "lookup.h"
#include "statistics.h"
#include "util/contentdatabase.h"

/**
 * The thread that waits for new connections.
 **/
static pthread_t TCPLISTENER_listener_;

#ifndef LINUX
/**
 * Catch sigpipe on send, and do nothing
 **/
static void catcher(int sig) {
  fprintf(stderr, "signal %d caught\n", sig);
  signal(sig, catcher);
}
#endif

/**
 * Process a query from the client. Forwards to the network.
 * @return SYSERR if the TCP connection should be closed, otherwise OK
 **/ 
static int processQuery(GNUNET_TCP_SOCKET * sock,
			TCP_Query_Request * queryRequest) {
  QUERY_POLICY qp = QUERY_ANSWER|QUERY_FORWARD|QUERY_INDIRECT|QUERY_PRIORITY_BITMASK; 
  QUERY_Message msg;
#if PRINT_TCP
  HexName hex;
#endif  

  if (ntohs(queryRequest->header.size) != sizeof(TCP_Query_Request))
    return SYSERR;
  /* convert double-hash to triple-hash */
#if PRINT_TCP
  hash2hex(&queryRequest->query, &hex);
  print("TCP: received double-query %s with ttl %d and priority %d.\n",
	&hex,
	ntohl(queryRequest->ttl),
	ntohl(queryRequest->priority));
#endif
  hash(&queryRequest->query,
       sizeof(HashCode160),
       &msg.query);
  msg.priority = queryRequest->priority; /* no htonl here: is already in network byte order! */
  msg.ttl = queryRequest->ttl; /* no htonl here: is already in network byte order! */
  getHostIdentity(getPublicHostkey(),
		  &msg.returnTo);
  msg.header.size = htons(sizeof(QUERY_Message));
  msg.header.requestType = htons(GNET_PROTO_3QUERY);
#if PRINT_TCP > 2
  hash2hex(&msg.query, &hex);
  print("TCP: executing triple-query %s with priority %u.\n",
	&hex,
	ntohl(msg.priority));
#endif  
  exec3Query(qp, &msg, sock);   
#if PRINT_TCP > 3
  print("TCP: executing triple-query %s DONE.\n",
	&hex);
#endif 
  return OK;
}

/**
 * Process a request to insert content from the client.
 * @return SYSERR if the TCP connection should be closed, otherwise OK
 **/
static int processInsert(GNUNET_TCP_SOCKET * sock,
			 TCP_Insert_Request * insertRequest) {
  ContentIndex entry;

  if (ntohs(insertRequest->header.size) != sizeof(TCP_Insert_Request))
    return SYSERR;
  writeContent(&insertRequest->doubleHash, 
	       &insertRequest->content);
  memcpy(&entry.doubleHash,
	 &insertRequest->doubleHash,
	 sizeof(HashCode160));
  entry.importance = insertRequest->importance; /* no htonl here: is already in network byte order! */
  entry.fileNameIndex = htonl(-1); /* data/content */
  entry.fileOffset = htonl(-1); /* data/content */
  addEntry(&entry);
  return OK;
}

/**
 * Process a request to index content from the client.
 * @return SYSERR if the TCP connection should be closed, otherwise OK
 **/
static int processIndex(GNUNET_TCP_SOCKET * sock,
			TCP_Indexing_Request * indexingRequest) {
  if (ntohs(indexingRequest->header.size) != sizeof(TCP_Indexing_Request)) {
#if PRINT_TCP
    print("TCP: WARNING: indexing request malformed!\n");
#endif
    return SYSERR;
  }
#if PRINT_TCP 
  print("TCP: received indexing request\n");
#endif
  addEntry(&indexingRequest->contentIndex);
  return OK;
}

/**
 * Process a query to list a file as on-demand encoded from the client.
 * @return SYSERR if the TCP connection should be closed, otherwise OK
 **/
static int processListFile(GNUNET_TCP_SOCKET * sock,
			   TCP_File_List_Request * listFileRequest) {
  TCP_File_Index_Reply * reply;

  if (ntohs(listFileRequest->header.size) != sizeof(TCP_File_List_Request))
    return SYSERR;
  listFileRequest->filename[1023] = 0; /* ensure strlen terminates! */  
  if (strlen(listFileRequest->filename) != ntohs(listFileRequest->fileNameLength))
    return SYSERR;  
  reply = (TCP_File_Index_Reply*) listFileRequest;
  reply->index = htonl(appendFilename(listFileRequest->filename));
  /* send reply */
  GNUNET_STATISTICS.tcp_out_counts[TCP_REPLY_FILE_INDEX]++;
  return writeToSocket(sock,
		       (TCP_SOCKET_BUFFER*) reply);
}

/**
 * Process a request to send statistics to the client
 * @return SYSERR if the TCP connection should be closed, otherwise OK
 **/
static int sendStatistics(GNUNET_TCP_SOCKET * sock,
			  TYPED_TCP_HEADER * header) {
  Statistics stats;

  if (ntohs(header->size) != sizeof(TYPED_TCP_HEADER))
    return SYSERR;
  getStatistics(&stats);
  GNUNET_STATISTICS.tcp_out_counts[TCP_REPLY_STATISTICS]++;
  return writeToSocket(sock,
		       (TCP_SOCKET_BUFFER*) &stats);
}

/**
 * Handle data available on the TCP socket descriptor. This method
 * first aquires a slot to register this socket for the writeBack
 * method (@see writeBack) and then demultiplexes all TCP traffic
 * received to the appropriate handlers.  @param sockDescriptor the
 * socket that we are listening to (fresh)
 **/
static void processData(int sockDescriptor){
  TCP_SOCKET_BUFFER buff;
  TYPED_TCP_HEADER * hdr;
  GNUNET_TCP_SOCKET sock;
  int i;

  /* register the socket */
  initGNUnetServerSocket(sockDescriptor,
			 &sock);
  hdr = (TYPED_TCP_HEADER*) &buff;
#if PRINT_TCP > 1
  print("TCP: opened connection \n");
#endif 
  while (1){
    if (SYSERR == readFromSocket(&sock,
				 &buff) )
      break; /* connection closed */
    /* demultiplex */
    GNUNET_STATISTICS.octets_total_tcp_in+=ntohs(hdr->size);
    if (ntohs(hdr->tcpType) < TCP_MAX)
      GNUNET_STATISTICS.tcp_in_counts[ntohs(hdr->tcpType)]++;    
    switch (ntohs(hdr->tcpType)) {
    case TCP_REQUEST_QUERY:
      i = processQuery(&sock,
		       (TCP_Query_Request*) hdr);         
      break; 
    case TCP_REQUEST_INSERT:
      i = processInsert(&sock,
		    (TCP_Insert_Request*)hdr);
      break;
    case TCP_REQUEST_INDEX:
      i = processIndex(&sock,
		       (TCP_Indexing_Request*) hdr);
      break;
    case TCP_REQUEST_LIST_FILE:
      i = processListFile(&sock,
			  (TCP_File_List_Request*)hdr);
      break;
    case TCP_REQUEST_STATISTICS:
      i = sendStatistics(&sock,
			 hdr);
      break;
    default:
#if PRINT_WARNINGS
      print("Unsupported request type received from TCP client!\n");
#endif
      i = SYSERR;
      break;
    } /* end of switch */
    if (OK != i) {
#if PRINT_WARNINGS
      print("HEADER invalid: %d - %d\n",
	    ntohs(hdr->size),
	    ntohs(hdr->tcpType));
#endif
      break;
   }
  }
#if PRINT_TCP > 1
  print("TCP: closing connection.\n");
#endif
  cancelTCP(&sock);
  destroySocket(&sock);
}

/**
 * Initialize the TCP port and listen for incoming connections.
 **/
static void tcpListenMain() {
  int listenerFD,incomingFD;
  int lenOfIncomingAddr;
  int listenerPort = getGNUnetPort();
  struct sockaddr_in serverAddr, clientAddr;
  pthread_t incomingThread; 	/* thread handling a new connection */
  int secs = 5;
  const int on = 1;

  /* create the socket */
 CREATE_SOCKET:
  while ( (listenerFD = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
    LOOPPRINT("tcpListenMain (1)"); 
#if PRINT_WARNINGS
    print("Error opening socket. No proxy service started. Trying again in 30 seconds.\n");
#endif
    sleep(30);
  }
  
  /* fill in the inet address structure */
  bzero((char *) &serverAddr, sizeof(serverAddr));
  serverAddr.sin_family = AF_INET;
  serverAddr.sin_addr.s_addr=htonl(INADDR_ANY);
  serverAddr.sin_port=htons(listenerPort);
 
  if ( setsockopt(listenerFD, 
		  SOL_SOCKET, 
		  SO_REUSEADDR, 
		  &on, sizeof(on)) < 0 )
    perror("setsockopt");

  /*bind the socket */
  while (bind (listenerFD, 
	       (struct sockaddr *) &serverAddr,
	       sizeof(serverAddr)) < 0){    
    LOOPPRINT("tcpListenMain (2)"); 
#if PRINT_WARNINGS
    print("Error binding the TCP listener to port %d. No proxy service started.\nTrying again in %d seconds...\n",
	  listenerPort,secs);
#endif
    sleep(secs);
    secs+=5; /* slow progression... */
    close(listenerFD);
    /* create the socket */
    while ( (listenerFD = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
#if PRINT_WARNINGS
      print("Error opening socket. No proxy service started. Trying again in 30 seconds.\n");
#endif
      sleep(30);
    }
    goto CREATE_SOCKET;
  }
  
  /* start listening for new connections */
  listen(listenerFD,CONCOUNT); /* max: 5 pending connections */
  
  /*process incoming data */
  while (1) {
    LOOPPRINT("tcpListenMain (3)"); 
   /*wait for a connection and process it */
    lenOfIncomingAddr=sizeof(clientAddr);
    incomingFD=accept(listenerFD,
		      (struct sockaddr *)&clientAddr, 
		      &lenOfIncomingAddr);
    if(incomingFD < 0){
#if PRINT_WARNINGS
      print("Error accepting new connection.\n");
#endif
      continue;
    }
    /* verify clientAddr for eligibility here (ipcheck-style,
       user should be able to specify who is allowed to connect,
       otherwise we just close and reject the communication! */  
    if (NO == checkIPListed(getTCPWhitelist(),
			    clientAddr.sin_addr)) {
#if PRINT_WARNINGS
      print("Rejected unauthorized connection from %d.%d.%d.%d.\n",
	    ((*(int*)&clientAddr.sin_addr) >> 0) & 255, 
	    ((*(int*)&clientAddr.sin_addr) >> 8) & 255, 
	    ((*(int*)&clientAddr.sin_addr) >>16) & 255, 
	    ((*(int*)&clientAddr.sin_addr) >>24) & 255);
#endif     
      close(incomingFD);
      continue;
    }

#if PRINT_TCP
    print("TCP: starting server\n");
#endif
    if ((pthread_create(&incomingThread, NULL,
			(void * (*)(void *)) processData, 
			(void *) incomingFD)) != 0) {
#if PRINT_WARNINGS
      print("Error creating thread to handle new incoming connection.\n");    
#endif
      close(incomingFD);
    }
  } /* while (1) */
} 

/**
 * Initialize the TCP port and listen for incoming connections.
 **/
void startTCPListener() {
  pthread_create(&TCPLISTENER_listener_, NULL,
		 (void * (*)()) &tcpListenMain, NULL); 
}

/**
 * Shutdown the module.
 **/ 
void doneTCPListener() {
  /*  pthread_kill(&TCPLISTENER_listener_, SIG_TERM);*/
}


/* end of tcpserver.c */
