/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * TrustService manages the trust database.
 * The highest bit in the credit rating is used
 * to indicate if the credit ranking must be saved
 * to disk (memory cache is 'dirty').
 *
 * @author Tzvetan Horozov
 * @author Christian Grothoff
 **/ 

#include "config.h"
#include "server/connection.h"
#include "server/trustservice.h"

static FileName trustDirectory;


/**
 * Call once in a while :-)
 **/
static void cronFlushTrustBuffer(void * unused) {
#if PRINT_CRON
  print("CRON: enter cronFlushTrustBuffer\n");
#endif
  forAllConnectedHosts(&flushHostCredit, unused);
#if PRINT_CRON
  print("CRON: exit cronFlushTrustBuffer\n");
#endif
}

/**
 * Initialize this module.
 **/
void initTrustService() {
  trustDirectory = (FileName)getCreditDirectory();
  addCronJob(&cronFlushTrustBuffer,
	     300, 300,
	     NULL); /* sync in 5m intervals */  
}


/**
 * Increase the host credit by a value. No point in synchronization
 * as changes can't mess up much anyway.
 * @param hostId is the identity of the host
 * @param value is the int value by which the host credit is to be increased or
 *        decreased
 * @returns the actual change in trust (positive or negative)
 */
unsigned int changeHostCredit(HostIdentity * hostId, 
			      int value){
  BufferEntry * be;
#if PRINT_TRUST 
  HexName hex;

  hash2hex(&hostId->hashPubKey, &hex);
  print("TRUST: changing trust in host %s by %d\n",
	&hex, value);
#endif  
  be = lookForHost(hostId);
  if (be == NULL) 
    return 0; /* not connected! */
  if ( (be->trust & 0x7FFFFFF) + value < 0) {
    value = - be->trust;
    be->trust = 0x80000000;
  } else {
    be->trust += value;
    be->trust = be->trust | 0x80000000;
  }
  MUTEX_UNLOCK(&be->sem);
  return value;
}

/**
 * Obtain the trust record of the host.
 **/
unsigned int getHostCredit(HostIdentity * hostId) {
  BufferEntry * be;
  unsigned int trust;
#if PRINT_TRUST > 3
  HexName hex;

  hash2hex(&hostId->hashPubKey, &hex);
  print("TRUST: looking for trust for host %s\n",
	&hex);
#endif
  be = lookForHost(hostId);
  if (be == NULL) 
    return 0;
  trust = be->trust;
  MUTEX_UNLOCK(&be->sem);
  return trust & 0x7FFFFFFF;
}

/**
 * Read host-information from a file. The semaphore
 * of the given be must already be down.
 **/
void initHostTrust(BufferEntry * be) {
  HexName fil;
  FileName fn;

  hash2hex(&be->hostId.hashPubKey, 
	   &fil);
#if PRINT_CONNECTION > 2
  print("CONNECTION: reading trust rating for host %s\n",
	&fil);
#endif
  hash2hex(&be->hostId.hashPubKey,
	   &fil);
  fn = xmalloc(strlen((char*)trustDirectory)+sizeof(HexName)+1,
	       "readHost: filename");
  buildFileName(trustDirectory, &fil, fn); 
  if (sizeof(unsigned int) !=
      readFile(fn, 
	       sizeof(unsigned int), 
	       &be->trust))
    be->trust = 0;
  xfree(fn,"readHost: filename"); 
}

/**
 * Write host-infromation to a file - flush the buffer entry!
 * The semaphore must already be down when calling.
 **/
void flushHostCredit(BufferEntry * be,
		     void * unused) {
  HexName fil;
  FileName fn;

  if ((be->trust & 0x80000000) == 0)
    return; /* unchanged */
  be->trust = be->trust & 0x7FFFFFFF;
  hash2hex(&be->hostId.hashPubKey,
	   &fil);
  fn = xmalloc(strlen((char*)trustDirectory)+sizeof(HexName)+1,
	       "writeHost: filename");
  buildFileName(trustDirectory,
		&fil,
		fn);
  writeFile(fn, 
	    &be->trust, 
	    sizeof(unsigned int), 
	    "644");
  xfree(fn, 
	"writeHost: filename");
}

/* end of trustservice.c */
