/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * Download helper methods (which do the real work).
 * 
 * @author Christian Grothoff
 * @file textui/downloadutil.c
 **/

#include "config.h"
#include "textui/downloadutil.h"

#include "textui/requestmanager.h"
#include "textui/node.h"  
#include <unistd.h>
#include "util/cron.h"
#include "configuration.h"
#include "symcipher.h"


/**
 * Download a file.
 * @param hashC the hash of the file
 * @param crc32 the CRC of the file
 * @param filesize the size of the file
 * @param fileName the name of the file
 * @param model the download model used to
 *        update status information; points to NULL if
 *        no status updates shall be given, otherwise 
 *        to a method that takes two size_t arguments
 *        (retrieved so far, total).
 * @param data pointer that is passed to the model method
 **/
int downloadFile(HashCode160 * hashC,
		 Ulong crc32,
		 size_t filesize,
		 char * fileName,
		 void (*model)(size_t, size_t, void*),
		 void * data) {
  RNode root;
  int treeDepth;
  size_t fl;  
  Ulong subCrc;
  HexName hex;

  root.crcRoot = crc32;
  root.fileLength = filesize;
  root.downloadModel = model; 
  root.bytes_downloaded = 0;
  root.nullParent = NULL; /* must always be NULL! */

  hash2hex(hashC, &hex);
  /*  print("Downloading file %s of size %d and crc %ld and hash %s\n",
	fileName, filesize, crc32, (char*) &hex);*/
	

  treeDepth = 0; 
  fl = CONTENT_SIZE;
  while (fl < filesize) {
    treeDepth++;
    fl *= NODES_PER_INODE;
  }
  root.treeDepth = treeDepth;
  if (fileName == NULL)
    root.fileHandle = 1; /* stdout */
  else {
    root.fileHandle = open(fileName,
			   O_CREAT|O_NONBLOCK|O_RDWR,
			   S_IRUSR|S_IWUSR);
    if (root.fileHandle == -1) {
      fprintf(stderr,
	      "could not open file %s\n",
	      fileName);
      return -1;
    }
    if (ftruncate(root.fileHandle,
		  filesize) != 0) {
      fprintf(stderr,
	      "could not truncate file %s to filesize %d\n",
	      fileName, filesize);
      return -1;
    }    
  }
  /* initiate root-requesting! */
  if (treeDepth == 0) 
    root.root = xmalloc(sizeof(Node),
			"downloadFile: root (1)");
  else 
    root.root = xmalloc(sizeof(INode),
			"downloadFile: root (2)");  
  memcpy(&root.root->key,
	 hashC,
	 sizeof(HashCode160));
  hash(hashC,
       sizeof(HashCode160),
       &root.root->request);
  root.root->pos    = 0;
  root.root->depth  = 0;
  root.root->parent = &root;
  if (treeDepth == 0) 
    tryPresent(root.root);
  else
    requestManagerRequest(root.root, inodeReceive);
  requestManagerReceiveResults(&root, data);
  
  if (root.bytes_downloaded != root.fileLength) 
    return -1; /* error */
  if (treeDepth == 0) 
    subCrc = crc32N(&root.root->data.content[0],
		    nodeGetBlockLength(root.root));
  else
    subCrc = crc32N(&root.root->data.content[0],
		    inodeGetBlockLength((INode*)root.root));
  if (crc32 != subCrc) {
    print("%s %s %s (%d != %d)\n",
	  "CRC (Root) error. Collision in one of the blocks!\n",
	  "You hit the jackpot (51:2^160 chances). Please report this!\n",
	  "You may also want to check file integrity. ",
	  crc32, subCrc);
    if (treeDepth == 0) 
      subCrc = nodeGetBlockLength(root.root);
    else
      subCrc = inodeGetBlockLength((INode*)root.root);
    return -1;
  }
  xfree(root.root,
	"downloadFile: root");
  return 0; 
}
