/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Main function to download files on GNUnet.
 * Current version does not yet work at all:
 *
 * - abort is implemented but not used or tested
 * - check if (leaf) node already present is not
 *   implemented
 * - existing file is not truncated to size of new file
 *
 * @author Christian Grothoff
 * @file textui/gnunetdownload.c 
 **/

#include "config.h"
#include <unistd.h>
#include "getopt.h"
#include "configuration.h"
#include "textui/downloadutil.h"


/**
 * Prints the usage information for this command if the user errs.
 * Aborts the program.
 **/
static void printhelp() {
  printf("USAGE: gnunet-download [OPTIONS] -- HASHCODE CRC32 SIZE\n");
  printf("-h, --help                    : print this page\n");
  printf("-c FILENAME, --config=FILENAME: load config file (defaults: %s)\n",
	 DEFAULT_CONFIG_FILE);
  printf("-o FILENAME, --output=FILENAME: write the file to FILENAME (default: STDOUT)\n");
  printf("-v, --version                 : print the version number\n");
  printf("The '--' is important if the CRC checksum is negative\n");
}

/**
 * ParseOptions and store the results in filename,
 * crc32, hash1, filesize.
 * @return -1 on error, 0 on normal exit, 1 on continue to download
 **/
static int parseOptions(int argc,
			char ** argv,
			char ** fileName,
			Ulong * crc32,
			HashCode160 * hash1,
			size_t * filesize) {
  char* confFile;
  int c;  
  int option_index;

  *fileName = NULL;
  /* set default config file */
  confFile = DEFAULT_CONFIG_FILE;
  while (1) {
    static struct option long_options[] = {
      { "config",  1, 0, 'c' },
      { "version", 0, 0, 'v' },
      { "help",    0, 0, 'h' },
      { "output",  1, 0, 'o' },
      { 0,0,0,0 }
    };    
    option_index=0;
    c = getopt_long(argc,
		    argv, 
		    "vhc:o:", 
		    long_options, 
		    &option_index);    
    if (c == -1) 
      break;  // No more flags to process
    switch(c) {
    case 'c': 
      confFile = optarg; 
      print("Configfile specified: %s.\n",confFile);
      break;
    case 'o':
      *fileName = optarg;
      print("Writing to %s.\n",*fileName);
      break;
    case 'v': 
      print("GNUnet v%s (%s), gnunet-download v%s\n",
	    GNUNET_VERSION, GNUNET_BRANCH, GNUNET_TEXTUI_VERSION);
      return 0;
    case 'h': 
      printhelp(); 
      return 0;
    default: 
      print("Unknown option %c. Aborting.\nUse --help to get a list of options.\n",
	    c);
      return -1;
    } /* end of parsing commandline */
  } /* while (1) */
  if (argc - optind != 3) {
    print("Not enough arguments. You must specify a hash, filesize and crc.\n");
    printhelp();
    return -1;
  }
  if (*fileName == NULL) {
    print("You must specify a filename.\n");
    printhelp();
    return -1;
  }
  /* final checking & conversion of parsed arguments */
  hex2hash((HexName*)argv[optind++],
	   hash1);
  
  if (1 != sscanf(argv[optind++],
		  "%ld",
		  crc32)) 
    errexit(">>%s<< is invalid for the CRC code (must be a number)\n",
	    argv[--optind]);
  if (1 != sscanf(argv[optind++],
		  "%d",
		  filesize))
    errexit(">>%s<< is invalid for the filesize (must be a number)\n",
	    argv[--optind]);
  readConfig(confFile);

  return 1;
}


/**
 * the main function to insert files into GNet.
 * @param argc number of arguments from the command line
 * @param argv command line arguments
 * @return return value from insertFile: 0: ok, -1: file not found 
 **/   
int main(int argc, char ** argv) {
  int result;
  HashCode160 hash1;
  Ulong crc32;
  size_t filesize;
  char * fileName;

  result = parseOptions(argc, argv,
			&fileName,
			&crc32,
			&hash1,
			&filesize);
  if (result <= 0)
    return result;
  /* ok, now initialize */
  initCron();
  initTCPIO();
  initKnownhosts();
  /* initialize request manager */
  initRequestManager(socket);
  startCron();
  result = 0; /* do not abort */
  /* insert request for root-node here! */
  return downloadFile(&hash1, 
		      crc32, 
		      filesize,
		      fileName, 
		      NULL, 
		      &result);
}

/* end of gnunetdownload.c */
