/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Main function to search files on GNUnet.
 *
 * @author Christian Grothoff
 * @file textui/gnunetsearch.c 
 **/

#include <unistd.h>
#include "config.h"
#include "util/getopt.h"
#include "util/cron.h"
#include "configuration.h"
#include "textui/searchutil.h"

/**
 * Print the search result.
 **/
static void printResult(RootNode * rootNode,
			void * unused) {
  HexName res;
  
  hash2hex(&rootNode->hashCode,
	   &res);
  printf("%s %d %d\n",
	 &res.data[0], 
	 ntohl(rootNode->crc),
	 ntohl(rootNode->file_length));
  rootNode->description[MAX_DESC_LEN-1] = 0;
  rootNode->filename[MAX_FILENAME_LEN-1] = 0;
  rootNode->mimetype[MAX_MIMETYPE_LEN-1] = 0;
  printf("=> %s <= (filename: %s, mimetype: %s)\n",
	 rootNode->description,
	 rootNode->filename,
	 rootNode->mimetype);
}

/**
 * Prints the usage information for this command if the user errs.
 * Aborts the program.
 **/
static void printhelp() {
  printf("USAGE: gnunet-search [OPTIONS] KEYWORD [AND KEYWORD]*\n");
  printf("-h, --help                    : print this page\n");
  printf("-c FILENAME, --config=FILENAME: load config file (defaults: %s)\n",
	 DEFAULT_CONFIG_FILE);
  printf("-v, --version                 : print the version number\n");
  printf("-t TIMEOUT                    : wait TIMEOUT seconds before aborting\n");
}

/**
 * Parse the options, set the timeout.
 * @param argc the number of options
 * @param argv the option list (including keywords)
 * @param timeout the timeout to set (0 for none)
 * @param keywords the hashs of the keywords to set (= the keys, not the queries!)
 * @return -1 on error, 0 if we should exit without error, 1 if we actually are going to do something
 **/
static int parseOptions(int argc,
			char ** argv,
			int * timeout,
			HashCode160 * keywords[]) {
  char* confFile;
  int option_index;
  int c;  

  confFile = DEFAULT_CONFIG_FILE;
  *timeout = -1;
  while (1) {
    static struct option long_options[] = {
      { "config",  1, 0, 'c' },
      { "version", 0, 0, 'v' },
      { "help",    0, 0, 'h' },
      { "timeout", 1, 0, 't' },
      { 0,0,0,0 }
    };    
    option_index=0;
    c = getopt_long(argc,
		    argv, 
		    "vhc:t:", 
		    long_options, 
		    &option_index);    
    if (c == -1) 
      break;  // No more flags to process
    switch(c) {
    case 'c': 
      confFile = optarg; 
      print("Configfile specified: %s.\n",
	    confFile);
      break;
    case 'v': 
      print("GNUnet v%s (%s), gnunet-search v%s\n",
	    GNUNET_VERSION, GNUNET_BRANCH, GNUNET_TEXTUI_VERSION);
      return 0;
    case 'h': 
      printhelp(); 
      return 0;
    case 't':
      if (1 != sscanf(optarg, "%ud", timeout)) {
	printf("You must pass a number to the -t option.\n");
	return -1;
      }
      break;
    default: 
      print("Unknown option %c. Aborting.\nUse --help to get a list of options.\n",
	    c);
      return -1;
    } /* end of parsing commandline */
  } /* while (1) */
  if (argc - optind < 1) {
    print("Not enough arguments. You must specify a keyword.\n");
    printhelp();
    return -1;
  }
  readConfig(confFile);
  if (*timeout == -1)
    *timeout = getSearchTimeout();
  /* now for the keywords ... */
  return parseKeywords(argc-optind,
		       &argv[optind],
		       keywords);
}

typedef struct {
  int count;
  HashCode160 * keywords;
  TCP_Query_Request * messages;
  void * callback;
} ReceiveThreadArgs;

static int receiveThread(ReceiveThreadArgs * args) {
  receiveResults(args->count,
		 args->keywords,
		 args->messages,
		 args->callback,
		 NULL);
  return 0;
}

/**
 * The main function to search for files on GNet.
 * @param argc number of arguments from the command line
 * @param argv command line arguments
 * @return return value from gnunetsearch: 0: ok, -1: error
 **/   
int main(int argc, char ** argv) {
  pthread_t resultPrintThread;
  HashCode160 * keywords;
  TCP_Query_Request * messages;
  int timeout;
  int res;
  ReceiveThreadArgs args;

  res = parseOptions(argc, argv,
		     &timeout, &keywords);
  if (res <= 0)
    return res;
  buildMessages(res,
		keywords,
		&messages);
  args.count = res;
  args.keywords = keywords;
  args.messages = messages;
  args.callback = printResult;

  initCron();
  initTCPIO();
  initKnownhosts();
  initSearchUtil("localhost",
		 getGNUnetPort());
  pthread_create(&resultPrintThread, NULL,
		 (void * (*)()) receiveThread, 
		 &args);
  sendQueries(res,
	      timeout,
	      messages);
  printf("Time out (%ds).\n", timeout);
  doneTCPIO();
  freeConfig();
  return 0;
}

/* end of gnunetsearch.c */
