/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Main function to obtain statistics from GNUnet.
 *
 * @author Christian Grothoff
 * @file textui/gnunetstatistics.c 
 **/

#include <unistd.h>
#include "config.h"
#include "util/getopt.h"
#include "util/cron.h"
#include "statistics.h"

/**
 * Print the search result.
 **/
static void printResult(RootNode * rootNode,
			void * unused) {
}

/**
 * Prints the usage information for this command if the user errs.
 * Aborts the program.
 **/
static void printhelp() {
  printf("USAGE: gnunet-statistics\n");
  printf("-h, --help                    : print this page\n");
  printf("-c FILENAME, --config=FILENAME: load config file (defaults: %s)\n",
	 DEFAULT_CONFIG_FILE);
  printf("-v, --version                 : print the version number\n");
}

/**
 * Parse the options, set the timeout.
 * @param argc the number of options
 * @param argv the option list (including keywords)
 * @param keywords the hashs of the keywords to set (= the keys, not the queries!)
 * @return -1 on error, 0 if we should exit without error, 1 if we actually are going to do something
 **/
static int parseOptions(int argc,
			char ** argv) {
  char* confFile;
  int option_index;
  int c;  

  confFile = DEFAULT_CONFIG_FILE;
  while (1) {
    static struct option long_options[] = {
      { "config",  1, 0, 'c' },
      { "version", 0, 0, 'v' },
      { "help",    0, 0, 'h' },
      { 0,0,0,0 }
    };    
    option_index=0;
    c = getopt_long(argc,
		    argv, 
		    "vhc:", 
		    long_options, 
		    &option_index);    
    if (c == -1) 
      break;  // No more flags to process
    switch(c) {
    case 'c': 
      confFile = optarg; 
      print("Configfile specified: %s.\n",
	    confFile);
      break;
    case 'v': 
      print("GNUnet v%s (%s), gnunet-statistics v%s\n",
	    GNUNET_VERSION, GNUNET_BRANCH, GNUNET_TEXTUI_VERSION);
      return 0;
    case 'h': 
      printhelp(); 
      return 0;
    default: 
      print("Unknown option %c. Aborting.\nUse --help to get a list of options.\n",
	    c);
      return -1;
    } /* end of parsing commandline */
  } /* while (1) */
  readConfig(confFile);
  return 1;
}

/**
 * The main function to search for files on GNet.
 * @param argc number of arguments from the command line
 * @param argv command line arguments
 * @return return value from gnunetsearch: 0: ok, -1: error
 **/   
int main(int argc, char ** argv) {
  pthread_t resultPrintThread;
  int res;
  GNUNET_TCP_SOCKET sock;
  TYPED_TCP_HEADER query;
  TCP_SOCKET_BUFFER reply;
  Statistics * stat;
  time_t start;

  res = parseOptions(argc, argv);
  if (res <= 0)
    return res;

  initTCPIO();
  
  if (SYSERR == initGNUnetClientSocket(getGNUnetPort(),
				       "localhost",
				       &sock)) {
    print("Error connecting to gnunetd!\n");
    return -1;
  }
  query.size = htons(sizeof(TYPED_TCP_HEADER));
  query.tcpType = htons(TCP_REQUEST_STATISTICS);
  if (SYSERR == writeToSocket(&sock,
			      (TCP_SOCKET_BUFFER *)&query)) {
    print("Error sending request to gnunetd!\n");
    return -1;
  }
  time(&start);
  stat = (Statistics*) &reply;
  stat->header.tcpType = 0;
  while (htons(stat->header.tcpType) != TCP_REPLY_STATISTICS) {
    if (time(NULL) - start > 30) {
      print("Error: gnunetd does not reply!\n");
      return -1;
    }
    if (SYSERR == readFromSocket(&sock,
				 &reply)) {
      print("Error obtaining reply from gnunetd!\n");
      return -1;
    }
  }
  if (htons(stat->header.size) != sizeof(Statistics)) {
    print("Received invalid reply from gnunetd!\n");
    return -1;
  }
  printStatistics(stat);  
  return 0;
}

/* end of gnunetstatistics.c */
