/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * insertFile - breaks file meant for
 *              insertion into blocks, prepares
 *              for insertion into network
 * @author Krista Bennett
 * @author Christian Grothoff
 * @file src/textui/insertfile.c
 **/

#include "config.h"
#include "textui/insertfile.h"
#include "util/contentdatabase.h"
#include "lookup.h"

#define PRINT_BLOCKS 0
#define DEBUG_CRC 0
#define EXTRA_CHECKS 0

static GNUNET_TCP_SOCKET sock;

void initInsertfile() {
  if (SYSERR == initGNUnetClientSocket(getGNUnetPort(),
				       "localhost",
				       &sock))
    errexit("could not open socket!\n");  
}
void doneInsertfile() {
  destroySocket(&sock);
}

/**
 * ask gnunetd to append a filename,
 * @return the index
 **/
int askAppendFilename(char * filename) {
  TCP_File_List_Request request;
  TCP_File_Index_Reply reply;
  int result;

  filename = expandFileName(filename);   
  request.header.size = htons(sizeof(TCP_File_List_Request));
  request.header.tcpType = htons(sizeof(TCP_REQUEST_LIST_FILE));
  request.fileNameLength = htons(strlen(filename));
  request.reserved = 0;
  bzero(&request.filename[0], TCP_FILE_LIST_FILENAME);
  memcpy(&request.filename[0],
	 filename,
	 strlen(filename));
  if (SYSERR == writeToSocket(&sock,
			      (TCP_SOCKET_BUFFER*)&request)) {
#if PRINT_WARNINGS
    print("WARNING: could not send data to gnunetd. Is gnunetd running?\n");
#endif
    xfree(filename,
	  "askAppendFilename: expanded filename");
    return -1;
  } 
  request.header.tcpType = htons(sizeof(TCP_REPLY_FILE_INDEX));
  result = 10;
  while (result > 0) {
    if (SYSERR == readFromSocket(&sock,
				 (TCP_SOCKET_BUFFER*)&reply)) {
#if PRINT_WARNINGS
      print("WARNING: could not receive data from gnunetd. Is gnunetd running?\n");
#endif
      xfree(filename,
	    "askAppendFilename: expanded filename");
      return -1;
    } 
    result = ntohl(reply.index);
    (*(int*)(&request.fileNameLength)) = htonl(result);
    if (0 == memcmp(&request, &reply, sizeof(TCP_File_List_Request))) {
      xfree(filename,
	    "askAppendFilename: expanded filename");
      return result;
    } else {
      result--; /* try only 10 times */
      /* re-issue request */
      if (SYSERR == writeToSocket(&sock,
				  (TCP_SOCKET_BUFFER*)&request)) {
#if PRINT_WARNINGS
	print("WARNING: could not send data to gnunetd. Is gnunetd running?\n");
#endif
	xfree(filename,
	      "askAppendFilename: expanded filename");
	return -1;
      } 
    }
  }
#if PRINT_WARNINGS
  print("WARNING: could not get valid response from gnunetd. Is gnunetd running?\n");
#endif
  xfree(filename,
	"askAppendFilename: expanded filename");
  return -1;
}


#if PRINTING_TABLE
/**
 * Print the current IBLOCK table.
 **/
static void printTable(HashCode160 * hashBuffer[], 
		       unsigned short * pos,
		       int treeDepth) {
  int i;
  int j;
  HexName hex;
  
  print("IBLOCK TABLE:");
  for (i=0;i<= treeDepth;i++) {
    print("\nTable %d: ",i);
    for (j=0;j<pos[i];j+=sizeof(HashCode160)) {
      hash2hex(&hashBuffer[i][j],&hex);
      print("%d: %s ",j, &hex);
    }
  }
  print("\nEND TABLE\n");
}
#endif
   
/**
 * Adds random information at the end of a block portion
 *
 * @param data represents the data block
 * @param actual_length represents the length of 
 *        the useful information in the
 *	  block; the rest of the bytes should be replaced by
 *	  random information
 **/
static void padCONTENT_SIZE(CONTENT_Block * data, 
			    int actual_length) {
  while (actual_length < CONTENT_SIZE)
    data->content[actual_length++] = (char) rand();
}

/**
 * Map the given file into memory.
 * @param flen pointer to a location where to
 *        store the length of the file
 * @param filename the name of the file to map into memory
 * @return a pointer to the mmapped file.
 **/
static void * openFile(size_t * flen, 
		       FileName filename) {
  int fd;
  struct stat st; /* file info */
  void * res;

  /* open file, return error if unable to open for reading */
  if ((fd = open(filename,O_RDONLY)) < 0) 
    errexit("Insertion error: unable to open %s\n", filename);
  stat(filename, &st); /* get file size */   
  *flen = st.st_size;
  if (*flen <= 0) 
    return NULL;
  res = mmap(NULL, /* no preferred address */
	     *flen, /* file length */
	     PROT_READ, /* we will only read */
	     MAP_SHARED, /* others may access this ro area, too */
	     fd, 
	     0 /* offset */);
  close(fd);
  if (res == MAP_FAILED) 
    errexit("mmap failed for file %s, fixme to do without!\n",
	    filename);
  return res;
}

/**
 * Encrypt a block of data using its hashcode.
 * @param block 1024-bit data to be encrypted
 * @param hc hashcode of the data we're going to encrypt
 * @param result return parameter for 1024-bit block of encrypted data
 **/
static void encodeBlockWithHash(CONTENT_Block * block,
				HashCode160 * hc,
				CONTENT_Block * result) {
#if EXTRA_CHECKS
  CONTENT_Block dBlock;
#endif

  if (SYSERR == encryptContent(block, hc, result)) {
    print("Encryption failed.\n");
    exit(-1);
  }
#if EXTRA_CHECKS
  if (SYSERR == decryptContent(result, hc, &dBlock)) {
    print("Decryption failed.\n");
    exit(-1);
  }
  if (0 != memcmp(&dBlock, block, sizeof(CONTENT_Block)))
    print("ERROR: decrypted block is not equal to original!\n");
#endif
}

/**
 * Compute the depth of the tree.
 * @param flen file length from which to compute the depth
 * @return depth of the tree
 **/
unsigned char computeDepth(size_t flen) {
  unsigned char treeDepth;
  long fl;

  treeDepth = 0;
  fl = CONTENT_SIZE;
  while (fl < flen) {
    treeDepth++;
    fl = fl * NODES_PER_INODE;
  }  
  return treeDepth;
}

/**
 * Write a block to the disk. Use H(H(block)(
 * for the filename. Encrypt with H(block). 
 * If the block is less than CONTENT_SIZE bytes, pad the last
 * CONTENT_SIZE-len bytes. Do not include the padded
 * part in the hash.
 * @param block the block to write
 * @param len the part of the block that is valid (not to pad)
 * @param hc the single hash of the data
 **/
static void writeBlock(CONTENT_Block * block, 
		       int len,
		       HashCode160 * hc) {
  TCP_Insert_Request request;

#if PRINT_BLOCKS
  HexName hex;

  hash2hex(hc, &hex);
  fprintf(stderr,
	  "Writing block: %s\n",
	  (char*)&hex);
#endif  
  hash(hc, sizeof(HashCode160),
       &request.doubleHash);
  encodeBlockWithHash(block, hc, &request.content);
  request.importance = htonl(0xFFFF); /* FIXME: let user specify priority and add current base priority! */
  request.header.size = htons(sizeof(TCP_Insert_Request));
  request.header.tcpType = htons(TCP_REQUEST_INSERT);
  if (SYSERR == writeToSocket(&sock,
			      (TCP_SOCKET_BUFFER*)&request))
    print("WARNING: could not send data to gnunetd. Is gnunetd running?\n");
}

void writeIndex(int fileNameIndex,
		int fileOffset,
		HashCode160 * hc) {
  TCP_Indexing_Request request;

  hash(hc,
       sizeof(HashCode160),
       &request.contentIndex.doubleHash);
  request.header.size = htons(sizeof(TCP_Indexing_Request));
  request.header.tcpType = htons(TCP_REQUEST_INDEX);
  request.contentIndex.importance = htonl(0xFFFF); /* FIXME (see above) */
  request.contentIndex.fileNameIndex = htonl(fileNameIndex);
  request.contentIndex.fileOffset = htonl(fileOffset);
  if (SYSERR == writeToSocket(&sock,
			      (TCP_SOCKET_BUFFER*)&request))
    print("WARNING: could not send index information to gnunetd. Is gnunetd running?\n");  
}

/**
 * Write an indirection block with the given crc
 * under its hashcode to the content directory.
 * @param block the data in the indirection block
 * @param pos the length of the data that is used
 * @param crc32 the CRC to store with the block 
 * @param hc the hash of the block that is being written (must be SET!)
 **/
static void writeIndirection(CONTENT_Block * block,
			     unsigned short pos,
			     Ulong crc32,
			     HashCode160 * hc) {
#if PRINT_BLOCKS
  HexName hex;
  
  hash2hex(hc, &hex);
  fprintf(stderr,
	  "Writing indirection: %s\n",
	  (char*)&hex);
#endif  
  ((Ulong *)block)[pos/sizeof(Ulong)] 
    = htonl(crc32);

  hash(block,
       pos+sizeof(Ulong),
       hc);
  writeBlock(block,pos+sizeof(Ulong),hc);
}

/**
 * Walk over the indirection nodes (inodes)
 * up from a leaf to the top, adding hashcodes
 * to the node above if the node below is full.
 * Also computes the CRC values.
 * @param hashBuffer list of tree-depth hash buffers 
 * @param crc32Buffer list of tree-depth CRC buffers
 * @param pos list of tree-depth write-positions in the 
 *        hash buffer (and implicitly in the crc buffer)
 * @param crc the CRC code that should be appended at level j
 * @param hc the hash code that should be appended at level j
 * @param j the current level of traversal
 * @param offset the file position offset we're at (this is currently
 *               only used for debugging purposes)
 **/
static void walkIndirections(CONTENT_Block hashBuffer[],
			     Ulong ** crc32Buffer,
			     unsigned short * pos,
			     Ulong crc,
			     HashCode160 * hc,
			     unsigned char j,
			     int offset) {
#if DEBUG_CRC
  int i;
#endif

  /* append hash and CRC, increase pos accordingly */
  memcpy(&hashBuffer[j].content[pos[j]],
	 hc,sizeof(HashCode160));
  crc32Buffer[j][pos[j]/sizeof(HashCode160)]=htonl(crc);
  pos[j] += sizeof(HashCode160);

  /* while buffer of level j is full, flush buffer to file and
     propagate summary up to level j+1. */
  while (pos[j]+sizeof(HashCode160) > CONTENT_SIZE) {
    /* compute CRC of the individual CRCs (51 CRCs to 1 CRC) */
#if DEBUG_CRC
    for (i=0;i<(CONTENT_SIZE / sizeof(HashCode160));i++) 
      fprintf(stderr,
	      "computing crc at %d over sub-crc[%d] = %d\n",
	      pos[j], i,  (int)htonl(crc32Buffer[j][i]));
#endif
    crc = crc32N(crc32Buffer[j], 
		 sizeof(Ulong) *
		 (CONTENT_SIZE / sizeof(HashCode160)));
    writeIndirection(&hashBuffer[j], pos[j],
		     crc, hc); /* flush to file */
    pos[j] = 0;
    crc = crc32N(&hashBuffer[j], CONTENT_SIZE);
    j++;
    memcpy(&hashBuffer[j].content[pos[j]],
	   hc,sizeof(HashCode160));
    crc32Buffer[j][pos[j]/sizeof(HashCode160)]=htonl(crc);
    pos[j] += sizeof(HashCode160);    
  } 
}


/**
 * Creates root node for the tree and writes the top-level tree node.
 * Root looks like this (when written to file - first 32 bits that
 * are NOT written to file contain content block length for hash and
 * writeFile functions):
 *
 *    32     32     160      strlen of desc     CONTENT_SIZE - the rest
 * ____________________________________________________________
 * |      |      | Hash  |                     | 
 * | File | Root | of    | Content description |  Padding    
 * |      | Node | Root  |                     |
 * | Size | CRC  | Indir |                     |  (if needed)
 * |      |      | Node  |                     |
 * ------------------------------------------------------------
 *
 * @param hc the hashcode of the top-node of the node-tree
 * @param keywords the keywords for the root-node
 * @param num_keys number of keywords to associated with the root node
 * @param description the description of the content
 * @param filename the suggested filename of the file this root node
 *                 represents (for downloads)
 * @param mimetype the MIME-type of the file this root node represents
 * @param flen the length of the file
 * @param crc the CRC checksum for the file
 **/
static void writeRootNode(HashCode160 * hc,
			  char ** keywords,
			  int num_keys,
			  char * description,
			  char * filename,
			  char * mimetype,
			  size_t flen,
			  Ulong crc) {
  RootNode rc;
  int i;
#if PRINT_BLOCKS
  HexName hex;

  hash2hex(hc, &hex);
  fprintf(stderr,
	  "Writing root node: %s\n",
	  (char*)&hex);
#endif  
  /* create root node (this part is keyword independent) */
  padCONTENT_SIZE((CONTENT_Block*)&rc, 0); /* fill with random */
  rc.file_length = htonl(flen);
  rc.crc = htonl(crc);  
  memcpy(&rc.hashCode, hc, sizeof(HashCode160));
  rc.major_formatVersion = htons(ROOT_MAJOR_VERSION);
  rc.minor_formatVersion = htons(ROOT_MINOR_VERSION);
  if (strlen(description) >= MAX_DESC_LEN)
    description[MAX_DESC_LEN-1] = 0;
  memcpy(&rc.description[0], description, 
	 strlen(description)+1);
  if (strlen(filename) >= MAX_FILENAME_LEN)
    description[MAX_FILENAME_LEN-1] = 0;
  memcpy(&rc.filename[0], filename, 
	 strlen(filename)+1);
  if (strlen(mimetype) >= MAX_MIMETYPE_LEN)
    description[MAX_MIMETYPE_LEN-1] = 0;
  memcpy(&rc.mimetype[0], mimetype, 
	 strlen(mimetype)+1);  
  printf("inserting %s - %s - %s\n",
	 description, filename, mimetype);

  /* for each keyword, create a new root node and write the file */
  for (i = 0; i < num_keys; i++) {
    hash(keywords[i],strlen(keywords[i]),hc);
    writeBlock((CONTENT_Block*) &rc,
	       CONTENT_SIZE,
	       hc);
  }
}

/**
 * De-facto main method. Inserts a file under the given
 * name into the local gnet node.\
 * @param filename the name of the (incomming/source) file
 * @param description the description of the file
 * @param filenameRoot suggested filename for downloads
 * @param mimetype the MIME-type of this file
 * @param keywords the keywords that shall be used to retrieve the file
 * @param num_keys the number of keywords to be associated with the file
 * @param index: file-index if we are just indexing, -1 if we do 
 *                 full insertion.
 * @return OK or SYSERR
 **/
int insertFile(FileName filename, 
	       char * description,
     	       char * filenameRoot,
	       char * mimetype,
	       char ** keywords,
	       int num_keys,
	       int index) {

  unsigned char i;
#if DEBUG_CRC
  int j;
#endif
  unsigned char treeDepth;
  size_t flen;
  size_t fpos;
  unsigned char * mmappedFile;
  CONTENT_Block * hashBuffer;
  Ulong ** crc32Buffer;
  unsigned short * pos;
  HashCode160 hc;
  CONTENT_Block * block = NULL;   
  Ulong crc = 0;
  
  mmappedFile = openFile(&flen, filename);
  if (mmappedFile == NULL)
    return SYSERR; /* failed */

  if (flen == 0)
    return SYSERR; /* file empty - failed */
  treeDepth = computeDepth(flen);

  /* allocate memory */
  hashBuffer 
    = xmalloc(sizeof(CONTENT_Block) * (treeDepth+1),
	      "insertFile: hashBuffer");
  crc32Buffer 
    = xmalloc(sizeof(Ulong*) * (treeDepth+1),
	      "insertFile: crcBuffer");
  pos 
    = xmalloc(sizeof(unsigned short) * (treeDepth+1),
	      "insertFile: pos array");

  /* initialize buffers and offsets */
  for (i=0;i<treeDepth+1;i++) {
    crc32Buffer[i]
      = (Ulong*) xmalloc(sizeof(Ulong) *
			(CONTENT_SIZE / sizeof(HashCode160)),
			 "insertFile: crcBuffer i");
   pos[i] = 0;
  }

  /* walk through nodes - start at leaf, walk up complete nodes */
  fpos = 0;

  while (fpos < flen) {
    if (flen - fpos >= CONTENT_SIZE) {
      block = (CONTENT_Block*) &mmappedFile[fpos];    
      crc = crc32N(block,CONTENT_SIZE);
      /* write to file? or just create indirection nodes? */
      hash(block, CONTENT_SIZE, &hc);	  
      /* add entry to database! */
      if (-1 == index) 
        writeBlock(block,CONTENT_SIZE, &hc);
      else 
	writeIndex(index, fpos, &hc);      
    } 
    /* last (incomplete) block */
    else { 					
      block = xmalloc(CONTENT_SIZE,
		      "insertFile: block");
      padCONTENT_SIZE(block,flen-fpos);
      memcpy(block, &mmappedFile[fpos], flen-fpos);
      crc = crc32N(block,flen-fpos);
      hash(block, flen-fpos, &hc);	  
      writeBlock(block,flen-fpos,&hc);
    }
    if (treeDepth > 0)
      walkIndirections(hashBuffer, crc32Buffer, 
		       pos, crc, &hc, 0, fpos);   
    fpos += CONTENT_SIZE;
  }
  /* free leaf-specific stuff */
  if ((flen & (CONTENT_SIZE-1)) > 0)
    /* CONTENT_SIZE MUST be a power of 2! */
    xfree(block,
	  "insertFile: block"); /* free padded block if one was created in the last iteration */
  munmap(mmappedFile,flen);

  /* complete tree... */
  for (i=0;i<treeDepth;i++) {
    if (pos[i] == 0)
      continue; /* nothing to complete: full node! */
    crc = crc32N(crc32Buffer[i], 
		 sizeof(Ulong) *
		 (pos[i] / sizeof(HashCode160)));
    padCONTENT_SIZE(&hashBuffer[i], pos[i]);
    writeIndirection(&hashBuffer[i], pos[i], 
		     crc, &hc);
#if DEBUG_CRC
    for (j=0;j<pos[i]/sizeof(HashCode160);j++) 
      fprintf(stderr,
	      "computing %d crc at %d over sub-crc[%d] = %d\n",
	      i, pos[i], j, (int) crc32Buffer[i][j]);
#endif
    crc = crc32N(&hashBuffer[i], pos[i]+sizeof(Ulong));

    walkIndirections(hashBuffer, crc32Buffer,
		     pos, crc, &hc, i+1, -1);
  } 
  if (treeDepth == 0)
    memcpy(&hashBuffer[treeDepth],
	   &hc,
	   sizeof(HashCode160));
  /* write root-node (write "hc") */
  writeRootNode((HashCode160*)&hashBuffer[treeDepth], 
		keywords, 
		num_keys, 
		description, 
		filenameRoot,
		mimetype,
		flen, crc);
  /* free rest */
  /* deallocate memory */
  for (i=0;i<treeDepth;i++) 
    xfree(crc32Buffer[i],
	  "insertFile: crc buffer i"); 
  xfree(crc32Buffer,
	"insertFile: crcBuffer");
  xfree(hashBuffer,
	"insertFile: hashBuffer");
  xfree(pos,
	"insertFile: pos array");
  return OK;
}

/* end of insertfile.c */
