/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Main function to insert files into gnet.
 * @author Christian Grothoff
 * @author Krista Bennett
 * @file textui/insertmain.c 
 **/

#include "config.h"
#include "textui/insertfile.h"
#include "textui/keywords.h"
#include "util/returnvalues.h"
#include "lookup.h"
#include <unistd.h>
#include "getopt.h"
#include "util/cron.h"
#include "util/tcpio.h"

/**
 * Prints the usage information for this command if the user errs.
 * Aborts the program.
 **/
static void printhelp() {
  printf("USAGE: gnunet-insert [OPTIONS] -f FILENAME [DESCRIPTION [KEYWORD]*]\n");
  printf("-h, --help                    : print this page\n");
  printf("-c FILENAME, --config=FILENAME: load config file (defaults: %s)\n",
	 DEFAULT_CONFIG_FILE);
  printf("-v, --version                 : print the version number\n");
  printf("-V, --verbose                 : be verbose\n");
  printf("-f FILENAME, --file=FILENAME  : insert FILENAME (obligatory)\n");
  printf("-n, --noindex                 : do not index, full insertion (full copy!)\n");
  printf("-x, --extraction              : disable automatic keyword extraction\n");
  printf("-m MIMETYPE, --mime=MIMETYPE  : set mimetype to be MIMETYPE\n");
  printf("-r FILENAME, --root=FILENAME  : publish FILENAME as the name of the file\n");
}

/**
 * the main function to insert files into GNet.
 * @param argc number of arguments from the command line
 * @param argv command line arguments
 * @return return 0 for ok, -1 on error
 **/   
int main(int argc, char ** argv) {
  int res;
  int verbose;
  char * confFile;
  char * filename;
  char * description;
  char * filenameRoot; /* file name for the root-node */
  char * mimetype;
  char ** keywords;
  int num_keywords;
  int c;
  
  int index_content;
  unsigned int extract_keys;
  mimetype = NULL;
  
  /* set default config file */
  confFile = DEFAULT_CONFIG_FILE;
  /* assume we encode content unless otherwise specified */
  index_content = YES; /* default: index */
  extract_keys = YES; /* default: extract keys */
  verbose = NO;
  filename = NULL;  
  filenameRoot = NULL;
  while (1) {
    int option_index=0;
    static struct option long_options[] = {
      { "config",        1, 0, 'c' },
      { "version",       0, 0, 'v' },
      { "help",          0, 0, 'h' },
      { "noindex",       0, 0, 'n' },
      { "file",          1, 0, 'f' },
      { "root",          1, 0, 'r' },
      { "mime",          1, 0, 'm' },
      { "noextraction",  1, 0, 'x' },
      { "verbose",       0, 0, 'V' },
      { 0,0,0,0 }
    };    
    c = getopt_long(argc,
		    argv, 
		    "vVhnxm:r:c:f:", 
		    long_options, 
		    &option_index);    
    if (c == -1) 
      break;  // No more flags to process
    switch(c) {
    case 'c': 
      confFile = optarg; 
      print("Configfile specified: %s.\n",confFile);
      break;
    case 'm':
      mimetype = optarg;
      break;
    case 'x':
      extract_keys = 0; /* no extraction */
      break;
    case 'r':
      filenameRoot = optarg;
      break;
    case 'V':
      verbose = YES;
      break;
    case 'f': 
      filename = optarg; 
      /* if filename is '/home/user/foo', use 'foo' as the filenameRoot */
      if (filenameRoot == NULL) {
	filenameRoot = filename;
	for (res = 0;res<strlen(filename);res++)
	  if (filename[res] == '/')
	    filenameRoot = &filename[res+1];
      }
      break;
    case 'v': 
      print("GNUnet v%s (%s), gnunet-insert v%s\n",
	    GNUNET_VERSION, GNUNET_BRANCH, GNUNET_TEXTUI_VERSION);
      return 0;
    case 'h': 
      printhelp(); 
      return 0;
    case 'n':
      index_content = NO; /* full insertion */
      break;
    default: 
      print("Unknown option %c. Aborting.\nUse --help to get a list of options.\n",
	    c);
      return -1;
    } /* end of parsing commandline */
  } /* while (1) */

#ifndef HAVE_LIBEXTRACTOR
  extract_keys = 0;
#endif
  readConfig(confFile);
  initRAND(); 
  if (filename == NULL) {
    print("You must specify a filename (option -f)\n");
    return -1;
  }
  if (argc - optind > 0)  
    description = argv[optind++];
  else
    description = NULL;
  keywords = &(argv[optind]);
  num_keywords = argc - optind;
  initCron();
  initTCPIO();
  initInsertfile();

  if (extract_keys) {
    extractKeywords(filename,
		    &description,
		    &mimetype,
		    &keywords,
		    &num_keywords);
  }
  if (num_keywords <= 0) {
    if (extract_keys) 
      print("Automatic keyword extraction failed, please specify manually.\n");
    else
      print("Not enough arguments.\n");
    printhelp();
    return -1;
  }  
  if (YES == verbose) {
    int k;
    printf("Using keywords:\n");
    for (k=0;k<num_keywords;k++)
      printf("%3d: %s\n",
	     k,
	     keywords[k]);
  }

  if (index_content == YES) 
    index_content = askAppendFilename(filename);
  else
    index_content = -1;
  if (description == NULL)
    description = "No description supplied";
  if (mimetype == NULL)
    mimetype = "unknown";
  res = insertFile(filename,
		   description,
		   filenameRoot,
		   mimetype,
		   keywords,
		   num_keywords,
		   index_content);
  
  if (res == OK) {
#ifdef PRINT_STATUS
    if (index_content != -1)
	printf("File %s successfully indexed.\n", filename);
    else
	printf("File %s successfully inserted.\n", filename);
#endif
  } else
    printf("Error inserting the file (%d).\n",res);
  doneTCPIO();
  freeConfig();
  if (res == OK)
    return 0;
  else 
    return -1;
}  

/* end of insertmain.c */
