/*
     This file is part of GNUnet.
     (C) 2001, 2002 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * Main function to insert multiple files into GNUnet.
 * @author Christian Grothoff
 * @author James Blackwell
 * @file textui/insertmulti.c 
 **/

#include "config.h"
#include "textui/insertfile.h"
#include "util/cron.h"
#include <unistd.h>
#include "getopt.h"
#include "lookup.h"
#include "util/tcpio.h"
#include "textui/keywords.h"

static ExtractorList * extractors;

/**
 * Prints the usage information for this command if the user errs.
 * Aborts the program.
 **/
static void printhelp() {
  printf("USAGE: gnunet-insert-multi [OPTIONS] FILENAMES*\n");
  printf("-h, --help                    : ");
  printf(" print this page\n");
  printf("-c FILENAME, --config=FILENAME: ");
  printf("load config file (defaults: %s)\n",
	 DEFAULT_CONFIG_FILE);
  printf("-v, --version                 : ");
  printf("print the version number\n");
  printf("-n, --noindex                 : ");
  printf("do not index, full insertion\n");
}

/**
 * @returns 0 if ok, -1 on error
 **/
static int dofile(char * filename,
		  int index_content) {
  int res = 0;
  int num_keywords;
  char ** keywords;
  char * filenameRoot = NULL;
  char * mimetype = NULL;
  char * description = NULL;

#ifdef PRINT_STATUS
  print("Working on file %s...\n",
	 filename);
#endif
  num_keywords = 0;
  extractKeywordsMulti(filename,
		       &description,
		       &mimetype,
		       &keywords,
		       &num_keywords,
		       extractors);
  if (num_keywords <= 0) {
    print("Automatic keyword extraction failed, did not insert file %s.\n",
	  filename);
    return -1;
  }  
  if (index_content == YES)
    index_content = appendFilename(filename);
  else
    index_content = -1;
  if (description == NULL)
    description = strdup("No description supplied");
  if (mimetype == NULL)
    mimetype = strdup("unknown");
  
  /* if filename is '/home/user/foo', use 'foo' as the filenameRoot */
  filenameRoot = filename;
  for (res = 0;res<strlen(filename);res++)
    if (filename[res] == '/')
      filenameRoot = &filename[res+1];

  /* We finally get to do the real work! */
  res = insertFile(filename, 
		   description,
		   filenameRoot,
		   mimetype,
		   keywords, 
		   num_keywords, 
		   index_content);
  if (res == OK) {
#ifdef PRINT_STATUS
    if (index_content != -1) {
      printf("File %s successfully indexed.\n",
	     filename);
    } else {
      printf("File %s successfully inserted.\n",
	     filename);
    }
#endif
  } 
#ifdef PRINT_WARNINGS
  else
    printf("Error inserting the file (%d).\n",res);
#endif
  while (num_keywords > 0) 
    xfree(keywords[--num_keywords],
	  "doFile: keyword");
  xfree(mimetype,
	"dofile: mime type");
  xfree(description,
	"dofile: description");
  return res;
}


/**
 * the main function to insert files into GNet.
 * @param argc number of arguments from the command line
 * @param argv command line arguments
 * @return return value from doFile: 0: ok, -1: file not found 
 **/   
int main(int argc, char ** argv) {
  char * confFile;
  int c;
  int i;
  int result=0;
  unsigned int index_content;
  
  /* set default config file */
  confFile = DEFAULT_CONFIG_FILE;
  /* assume we encode content unless otherwise specified */
  index_content = YES; /* index only (default) */
  while (1) {
    int option_index=0;
    static struct option long_options[] = {
      { "config",  1, 0, 'c' },
      { "version", 0, 0, 'v' },
      { "help",    0, 0, 'h' },
      { "noindex", 0, 0, 'n' },
      { 0,0,0,0 }
    };    
    c = getopt_long(argc,
		    argv, 
		    "vhnc:", 
		    long_options, 
		    &option_index);    
    if (c == -1) 
      break;  // No more flags to process
    switch(c) {
    case 'c': 
      confFile = optarg; 
      print("Configfile specified: %s.\n",confFile);
      break;
    case 'v': 
      print("GNUnet v%s (%s), gnunet-insert-mutli v%s\n",
	    GNUNET_VERSION, GNUNET_BRANCH, GNUNET_TEXTUI_VERSION);
      return 0;
    case 'h': 
      printhelp(); 
      return 0;
    case 'n':
      index_content = NO; /* NOT on demand */
      break;
    default: 
      print("Unknown option %c. Aborting.\nUse --help to get a list of options.\n",
	    c);
      return -1;
    } /* end of parsing commandline */
  } /* while (1) */
  if (argc - optind < 1) {
    printf("Not enough arguments.\n");
    printhelp();
    return -1;
  }
  readConfig(confFile);
  initRAND(); 
  initCron();
  initTCPIO(); 
  initInsertfile();
#ifdef HAVE_LIBEXTRACTOR
  extractors = getExtractors();
#endif
  for (i=optind;i<argc;i++) 
    if (-1 == dofile(argv[i],
		     index_content))
      result = -1;
#ifdef HAVE_LIBEXTRACTOR
  removeAll(extractors);
#endif
  doneTCPIO();
  freeConfig();
  return result;
}  

/* end of insertmp3.c */
